package org.lsst.ccs.drivers.reb;

/**
 *  Java interface to the global client routines
 *
 *  @author Owen Saxton
 */
public class GlobalClient {

    /*
     *  Data fields.
     */
    @Deprecated
    public static final int
        HDW_TYPE_DAQ = RegClient.HDW_TYPE_DAQ,
        HDW_TYPE_DAQ0 = RegClient.HDW_TYPE_DAQ0,
        HDW_TYPE_DAQ1 = RegClient.HDW_TYPE_DAQ1,
        HDW_TYPE_DAQ2 = RegClient.HDW_TYPE_DAQ2,
        HDW_TYPE_DAQ4 = RegClient.HDW_TYPE_DAQ4,
        HDW_TYPE_PCI0 = RegClient.HDW_TYPE_PCI0,
        HDW_TYPE_PCI1 = RegClient.HDW_TYPE_PCI1,
        HDW_TYPE_PCI  = RegClient.HDW_TYPE_PCI;

    private Impl impl;
    private ClientFactory clientFactory = new ClientFactory();


    /**
     *  Finalizer.
     *
     *  @throws  Throwable
     */
    @Override
    protected void finalize() throws Throwable
    {
        super.finalize();
        if (impl != null) {
            impl.deleteGlobalClient();
        }
    }


    /**
     *  Sets an alternative client factory.
     *
     *  The client factory is used to create the objects that implement
     *  access to either real (by default) or simulated hardware.
     *
     *  @param  clientFactory  The ClientFactory to use
     */
    public void setClientFactory(ClientFactory clientFactory) {
        this.clientFactory = clientFactory;
    }


    /**
     *  Opens a connection to a DAQ V2/4 partition.
     *
     *  @param  part  The partition name
     *  @throws  REBException 
     */
    public void open(String part) throws REBException
    {
        if (impl != null) {
            throw new REBException("Global connection already open");
        }
        Impl newImpl = clientFactory.createGlobalClient(RegClient.HDW_TYPE_DAQ);
        newImpl.newGlobalClient(part);
        impl = newImpl;
    }


    /**
     *  Opens a connection to a set of REBs, i.e. a non DAQ 2/4 system.
     * 
     *  @param  hdw     The hardware type
     *  @param  ids     The array of IDs of the REBs to connect to
     *  @param  ifcs    The names of the hardware interfaces to use for the
     *                  corresponding IDs.  If null, or an element is null or
     *                  empty, the default interface is used.
     *  @throws  REBException 
     */
    public void open(int hdw, int[] ids, String[] ifcs) throws REBException
    {
        if (impl != null) {
            throw new REBException("Global connection already open");
        }
        Impl newImpl = clientFactory.createGlobalClient(hdw);
        newImpl.newGlobalClient(ids, ifcs);
        impl = newImpl;
    }


    /**
     *  Closes a connection.
     *
     *  @throws  REBException 
     */
    public void close() throws REBException
    {
        checkOpen();
        impl.deleteGlobalClient();
        impl = null;
    }


    /**
     *  Gets the hardware type.
     *
     *  @return  The encoded hardware type
     *  @throws  REBException 
     */
    public int getHwType() throws REBException
    {
        checkOpen();
        return impl.getHwType();
    }


    /**
     *  Sets the list of registers to be read when the image is acquired.
     *
     *  @param  rebType    The encoded REB type
     *  @param  registers  The array of register addresses
     *  @throws  REBException 
     */
    public void setRegisterlist(int rebType, int[] registers) throws REBException
    {
        checkOpen();
        impl.setRegisterList(rebType, registers);
    }


    /**
     *  Acquires an image (by setting the trigger).
     *
     *  @param  name  The name of the image
     *  @return  The trigger time (nsec since the Epoch)
     *  @throws  REBException 
     */
    public long acquireImage(String name) throws REBException
    {
        checkOpen();
        return impl.triggerImage(name);
    }


    /**
     *  Acquires an image (by setting the trigger).
     *
     *  @param  name  The name of the image
     *  @param  folder  The image folder to use
     *  @param  opcode  The sequencer opcode
     *  @param  annotation  The annotation for the image
     *  @param  ids  An array of REB ids to use; if empty, use the whole partition
     *  @return  The trigger time (nsec since the Epoch)
     *  @throws  REBException 
     */
    public long acquireImage(String name, String folder, int opcode, String annotation, int... ids) throws REBException
    {
        checkOpen();
        return impl.triggerImage(name, folder, opcode, annotation, ids);
    }


    /**
     *  Starts the sequencer.
     *
     *  @param  opcode  The sequencer opcode
     *  @return  The trigger time (nsec since the Epoch)
     *  @throws  REBException 
     */
    public long startSequencer(int opcode) throws REBException
    {
        checkOpen();
        return impl.startSequencer(opcode);
    }


    /**
     *  Checks that connection is open
     *
     *  @throws  REBException 
     */
    private void checkOpen() throws REBException
    {
        if (impl == null) {
            throw new REBException("Global connection not open");
        }
    }


    /**
     *  Inner interface to support choosing the hardware.
     */
    public interface Impl {

        /**
         *  Creates a new object.
         *
         *  Creates needed C++ objects.
         *
         *  @param  part    The name of the DAQ partition to use
         *  @throws  REBException
         */
        public default void newGlobalClient(String part) throws REBException {
        }


        /**
         *  Creates a new object.
         *
         *  @param  ids   The IDs of the REBs to connect to
         *  @param  ifcs  The names of the hardware interfaces to use for the
         *                corresponding IDs.  If null, or an element is null or
         *                empty, the default interface is used.
         *  @throws  REBException
         */
        public default void newGlobalClient(int[] ids, String[] ifcs) throws REBException {
        }


        /**
         *  Deletes an object.
         *
         *  Deletes referenced C++ objects.
         */
        public default void deleteGlobalClient() {
        }


        /**
         *  Gets the hardware type.
         *
         *  @return  The encoded hardware type
         */
        public default int getHwType() {
            return RegClient.HDW_TYPE_DAQ1;
        }


        /**
         *  Sets the registers to be read at image acquisition.
         *
         *  @param  rebType    The encode REB type
         *  @param  registers  The array of register addresses
         */
        public default void setRegisterList(int rebType, int[] registers) {
        }


        /**
         *  Triggers image acquisition.
         *
         *  @param  name  The image name
         *  @return  The trigger time (nsec since the Epoch)
         *  @throws  REBException
         */
        public default long triggerImage(String name) throws REBException {
            return 1_000_000 * System.currentTimeMillis();
        }


        /**
         *  Triggers image acquisition.
         *
         *  @param  name  The image name
         *  @param  folder  The image folder to use
         *  @param  opcode  The sequencer opcode
         *  @param  annotation  The image annotation
         *  @param  ids  An array of REB ids to use; if empty, use whole partition 
         *  @return  The trigger time (nsec since the Epoch)
         *  @throws  REBException
         */
        public default long triggerImage(String name, String folder, int opcode, String annotation, 
                                         int... ids) throws REBException {
            return triggerImage(name);
        }


        /**
         *  Starts the sequencer.
         *
         *  @param  opcode  The sequencer opcode
         *  @return  The trigger time (nsec since the Epoch)
         *  @throws  REBException
         */
        public default long startSequencer(int opcode) throws REBException {
            return 1_000_000 * System.currentTimeMillis();
        }
    }

}
