package org.lsst.ccs.drivers.reb;

import java.util.HashMap;
import java.util.Map;

/**
 ******************************************************************************
 **
 **  ASIC ADC reading routines.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class SlowAdcs extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    public final static int
        REG_TEMP_SLCT     = 0x600100,
        REG_TEMP_READ     = 0x600101,
        REG_TEMP_MUX      = 0x600200,
        TEMP_CABAC_TOP    = 0,
        TEMP_CABAC_BOTTOM = 1,
        TEMP_ASPIC_TOP    = 2,
        TEMP_ASPIC_BOTTOM = 3,
        NUM_TEMPS         = 4,
        MUX_T_B_SLOW      = 0,
        MUX_T_FAST_B_SLOW = 1,
        MUX_T_SLOW_B_FAST = 2,
        MUX_T_B_FAST      = 3,
        NUM_MUX_SET       = 4,

        REG_ASPIC_START   = 0x600100,
        REG_ASPIC_READ    = 0x601000,
        REG_MUX_CONFIG    = 0x600101,
        REG_MUX_READ      = 0x601010,
        NUM_ASPIC_TEMPS   = 6,
        NUM_BIAS_VOLTS    = 16,
        NUM_CURR_CHANS    = 8,

        CHAN_OD           = 0,
        CHAN_OG           = 1,
        CHAN_RD           = 2,
        CHAN_GD           = 3,
        CHAN_VREF5        = 4,
        CHAN_VREF25       = 5,

        CHAN_OD_0         = 0,
        CHAN_OG_0         = 1,
        CHAN_RD_0         = 2,
        CHAN_GD_0         = 3,
        CHAN_OD_1         = 4,
        CHAN_OG_1         = 5,
        CHAN_RD_1         = 6,
        CHAN_GD_1         = 7,
        CHAN_OD_2         = 8,
        CHAN_OG_2         = 9,
        CHAN_RD_2         = 10,
        CHAN_GD_2         = 11,
        CHAN_VREF5_0      = 12,
        CHAN_VREF5_1      = 13,
        CHAN_VREF5_2      = 14,
        CHAN_VREF25_1     = 15;

    public final static double
        VOLT_SCALE_0    = 1.5 / 65536,
        VOLT_SCALE      = 5.0 / 4096,
        VOLT_SCALE_HI   = VOLT_SCALE * 11,
        CURR_SCALE      = VOLT_SCALE / 1000,
        TEMP_OFFSET     = 2.477 / 0.0056,                // From Claire Juramy
        TEMP_SCALE      = VOLT_SCALE * (-1.0 / 0.0056);  //        "

    private final static int[][]
        voltsCfg = {{0x3c, 0x5c, 0x7c, 0x1c, 0x9c},
                    {0xdc, 0xfc, 0x1d, 0xbc, 0x3d, 0x5d},
                    {0x9d, 0xbd, 0xdd, 0x7d, 0xfd}};
    private final static double[][]
        voltsScale = {{VOLT_SCALE_HI, VOLT_SCALE_HI, VOLT_SCALE_HI,
                       VOLT_SCALE_HI, VOLT_SCALE},
                      {VOLT_SCALE_HI, VOLT_SCALE_HI, VOLT_SCALE_HI,
                       VOLT_SCALE_HI, VOLT_SCALE, VOLT_SCALE},
                      {VOLT_SCALE_HI, VOLT_SCALE_HI, VOLT_SCALE_HI,
                       VOLT_SCALE_HI, VOLT_SCALE}};

    private final static Map<Integer, Integer> chanMap = new HashMap<>();
    static {
        chanMap.put(CHAN_GD_0, 0x00 | CHAN_GD);
        chanMap.put(CHAN_OD_0, 0x00 | CHAN_OD);
        chanMap.put(CHAN_OG_0, 0x00 | CHAN_OG);
        chanMap.put(CHAN_RD_0, 0x00 | CHAN_RD);
        chanMap.put(CHAN_VREF5_0, 0x00 | CHAN_VREF5);
        chanMap.put(CHAN_GD_1, 0x10 | CHAN_GD);
        chanMap.put(CHAN_OD_1, 0x10 | CHAN_OD);
        chanMap.put(CHAN_OG_1, 0x10 | CHAN_OG);
        chanMap.put(CHAN_RD_1, 0x10 | CHAN_RD);
        chanMap.put(CHAN_VREF5_1, 0x10 | CHAN_VREF5);
        chanMap.put(CHAN_VREF25_1, 0x10 | CHAN_VREF25);
        chanMap.put(CHAN_GD_2, 0x20 | CHAN_GD);
        chanMap.put(CHAN_OD_2, 0x20 | CHAN_OD);
        chanMap.put(CHAN_OG_2, 0x20 | CHAN_OG);
        chanMap.put(CHAN_RD_2, 0x20 | CHAN_RD);
        chanMap.put(CHAN_VREF5_2, 0x20 | CHAN_VREF5);
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public SlowAdcs()
    {
        super();
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  reg  The associated register client object
    **
    ***************************************************************************
    */
    public SlowAdcs(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Reads the version 1 slow ADC directly.
    **
    **  @param  config  The ADC/MUX configuration value to use
    **
    **  @return  The ADC value (counts)
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int readDirect(int config) throws REBException
    {
        checkVersion(OPTN_SLOW_ADCS, VERSION_1);
        write(REG_MUX_CONFIG, config);
        return read(REG_MUX_READ) & 0xfff;
    }


   /**
    ***************************************************************************
    **
    **  Reads a range of ASPIC temperature values.
    **
    **  @param  first  The number of the first temperature to read
    **
    **  @param  count  The number of temperatures to read
    **
    **  @return  The array of values (Celsius)
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double[] readAspicTemps(int first, int count) throws REBException
    {
        checkVersion(OPTN_SLOW_ADCS, VERSION_1);
        if (first < 0 || count < 0 || first + count > NUM_ASPIC_TEMPS) {
            throw new REBException("Invalid ASPIC temperature range");
        }
        double[] data = new double[count];
        write(REG_ASPIC_START, 0);
        for (int j = 0; j < count; j++) {
            int value = read(REG_ASPIC_READ + first + j) & 0xfff;
            data[j] = TEMP_OFFSET + TEMP_SCALE * value;
        }

        return data;
    }


   /**
    ***************************************************************************
    **
    **  Reads an ASPIC temperature.
    **
    **  @param  strip  The strip number
    **
    **  @param  side   The side: 0 (top) or 1 (bottom)
    **
    **  @return  The temperature
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double readAspicTemp(int strip, int side) throws REBException
    {
        checkVersion(OPTN_SLOW_ADCS, VERSION_1);
        checkStrip(strip);
        checkSide(side);
        write(REG_MUX_CONFIG, 4 * strip + side + 2);
        return TEMP_OFFSET + (read(REG_MUX_READ) & 0xfff) * TEMP_SCALE;
    }


   /**
    ***************************************************************************
    **
    **  Reads a bias voltage.
    **
    **  @param  strip  The strip number
    **
    **  @param  chan   The channel number
    **
    **  @return  The voltage
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double readVoltage(int strip, int chan) throws REBException
    {
        checkVersion(OPTN_SLOW_ADCS, VERSION_1);
        checkStrip(strip);
        try {
            write(REG_MUX_CONFIG, voltsCfg[strip][chan]);
            return (read(REG_MUX_READ) & 0xfff) * voltsScale[strip][chan];
        }
        catch (IndexOutOfBoundsException e) {
            throw new REBException(("Invalid voltage channel: " + chan));
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads a bias voltage.
    **
    **  @param  chan   The overall channel number
    **
    **  @return  The voltage
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double readVoltage(int chan) throws REBException
    {
        Integer desc = chanMap.get(chan);
        if (desc == null) {
            throw new REBException(("Invalid voltage channel: " + chan));
        }
        return readVoltage(desc >> 4, desc & 0x0f);
    }


   /**
    ***************************************************************************
    **
    **  Reads a CCD current.
    **
    **  @param  strip  The strip number
    **
    **  @param  side   The side: 0 (top) or 1 (bottom)
    **
    **  @param  chan   The channel number
    **
    **  @return  The current (amps)
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double readCurrent(int strip, int side, int chan) throws REBException
    {
        checkVersion(OPTN_SLOW_ADCS, VERSION_1);
        checkStrip(strip);
        checkSide(side);
        if (chan < 0 || chan >= NUM_CURR_CHANS) {
            throw new REBException(("Invalid current channel: " + chan));
        }
        write(REG_MUX_CONFIG, (4 * strip + side) | (chan << 5) | 0x10);
        return (read(REG_MUX_READ) & 0xfff) * CURR_SCALE;
    }


   /**
    ***************************************************************************
    **
    **  Reads a CCD current.
    **
    **  @param  chan   The overall channel number: 16*strip + 8*side + chan
    **
    **  @return  The current (amps)
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double readCurrent(int chan) throws REBException
    {
        return readCurrent(chan >> 4, (chan >> 3) & 1, chan & 7);
    }


   /**
    ***************************************************************************
    **
    **  Reads the slow ADC.
    **
    **  @param  chan  The number of the channel to read
    **
    **  @return  The converted ADC value (volts)
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double readSlow(int chan) throws REBException
    {
        checkVersion(OPTN_SLOW_ADCS, VERSION_0);
        if (chan < 0 || chan >= NUM_TEMPS) {
            throw new REBException("Invalid slow ADC channel number");
        }
        write(REG_TEMP_SLCT, chan);
        int value = ((read(REG_TEMP_READ) ^ 0x00800000) << 8) >> 14;

        return VOLT_SCALE_0 * value;
    }


   /**
    ***************************************************************************
    **
    **  Sets the ASIC MUX.
    **
    **  @param  set  The value to set
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void setMux(int set) throws REBException
    {
        checkVersion(OPTN_SLOW_ADCS, VERSION_0);
        if (set < 0 || set >= NUM_MUX_SET) {
            throw new REBException("Invalid MUX setting value: " + set);
        }
        write(REG_TEMP_MUX, set);
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of ASPIC temperatures.
    **
    **  @return  The number of ASPIC temperatures
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int getNumAspicTemps() throws REBException
    {
        return getVersion(OPTN_SLOW_ADCS, true) == VERSION_1 ? NUM_ASPIC_TEMPS : 0;
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of bias voltages.
    **
    **  @return  The number of bias voltages
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int getNumBiasVolts() throws REBException
    {
        return getVersion(OPTN_SLOW_ADCS, true) == VERSION_1 ? NUM_BIAS_VOLTS : 0;
    }


   /**
    ***************************************************************************
    **
    **  Checks a strip number.
    **
    **  @param  strip  The strip number
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private static void checkStrip(int strip) throws REBException
    {
        if (strip < 0 || strip >= Asic.NUM_STRIPS) {
            throw new REBException("Invalid strip number: " + strip);
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a side number.
    **
    **  @param  side  The side number
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private static void checkSide(int side) throws REBException
    {
        if (side < 0 || side >= Asic.NUM_SIDES) {
            throw new REBException(("Invalid board side: " + side));
        }
    }

}
