package org.lsst.ccs.drivers.reb;

/**
 ******************************************************************************
 **
 **  Java interface to the image client routines
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class ImageClient {

   /**
    ***************************************************************************
    **
    **  Defines the data listener.
    **
    ***************************************************************************
    */
    public interface Listener {

       /**
        ***********************************************************************
        **
        **  Receives image data.
        **
        **  @param  image  The received image
        **
        ***********************************************************************
        */
        public void processImage(Image image);

    }


   /**
    ***************************************************************************
    **
    **  Thread that obtains the image data for the listener.
    **
    ***************************************************************************
    */
    public class ReadThread extends Thread {

        Listener listener;   // Attached listener
        Image listenImage;   // Image supplied by listener
        int state = RUNNING;


       /**
        ***********************************************************************
        **
        **  Constructor.
        **
        **  @param  lstnr  The listener
        **
        **  @param  image  The image to use
        **
        ***********************************************************************
        */
        public ReadThread(Listener lstnr, Image image)
        {
            listener = lstnr;
            listenImage = image;
        }


       /**
        ***********************************************************************
        **
        **  Receives image data.
        **
        ***********************************************************************
        */
        @Override
        public void run()
        {
            while (state == RUNNING) {
                Image image = impl.waitForImage(listenImage);
                if (state != RUNNING) break;
                impl.getImage(image);
                Listener l = listener;
                if (state != RUNNING) break;
                l.processImage(image);
            }
            if (state == CLOSING) {
                impl.deleteImageClient();
            }
        }


       /**
        ***********************************************************************
        **
        **  Sets the thread state.
        **
        **  @param  newState  The state to set
        **
        ***********************************************************************
        */
        public void setState(int newState)
        {
            state = newState;
        }

    }


   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    public static final int
        HDW_TYPE_DAQ0 = 0,
        HDW_TYPE_DAQ1 = 1,
        HDW_TYPE_PCI  = 2;

    private final static int
        RUNNING = 0,
        CLOSING = 1,
        ENDING  = 2;

    private ReadThread reader;
    private Listener imgListener;
    private Image listenImage;
    private Impl impl;
    private final RegClient reg;


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public ImageClient()
    {
        this(null);
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  reg  The associated register client
    **
    ***************************************************************************
    */
    public ImageClient(RegClient reg)
    {
        this.reg = reg;
    }


   /**
    ***************************************************************************
    **
    **  Finalizer.
    **
    **  @throws  Throwable
    **
    ***************************************************************************
    */
    @Override
    protected void finalize() throws Throwable
    {
        super.finalize();
        if (impl != null) {
            impl.deleteImageClient();
        }
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection using the default network interface.
    **
    **  @param  id  The ID of the REB to connect to
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void open(int id) throws REBException
    {
        open(id, null);
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection using the specified network interface.
    **
    **  @param  id   The ID of the REB to connect to
    **
    **  @param  ifc  The name of the network interface to use.  If null or
    **               empty, the default interface is used.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void open(int id, String ifc) throws REBException
    {
        open(HDW_TYPE_DAQ0, id, ifc);
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection using the specified network interface.
    **
    **  @param  hdw  The hardware type
    **
    **  @param  id   The ID of the REB to connect to
    **
    **  @param  ifc  The name of the network interface to use.  If null or
    **               empty, the default interface is used.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void open(int hdw, int id, String ifc) throws REBException
    {
        if (impl != null) {
            throw new REBException("Image connection already open");
        }
        if (hdw == HDW_TYPE_DAQ0) {
            impl = new ImageClientDaq0();
        }
        else if (hdw == HDW_TYPE_DAQ1) {
            impl = new ImageClientDaq1();
        }
        else if (hdw == HDW_TYPE_PCI) {
            impl = new ImageClientPci(reg);
        }
        else {
            throw new REBException("Invalid hardware type (" + hdw + ")");
        }
        impl.newImageClient(id, ifc);
        if (imgListener != null) {
            reader = new ReadThread(imgListener, listenImage);
            reader.setDaemon(true);
            reader.start();
        }
    }


   /**
    ***************************************************************************
    **
    **  Closes a connection.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void close() throws REBException
    {
        checkOpen();
        if (reader != null) {
            reader.setState(ENDING);
            reader = null;
        }
        impl.deleteImageClient();
        impl = null;
    }


   /**
    ***************************************************************************
    **
    **  Awaits an image.
    **
    **  Waits until a new image has been generated.
    **
    **  @param  image  An Image object in which to save the reference and
    **                 metadata for the new image, or null if a new image
    **                 object is to be created.
    **
    **  @return  The Image object containing the new image reference and
    **           metadata.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public Image awaitImage(Image image) throws REBException
    {
        checkOpen();
        return impl.waitForImage(image);
    }


   /**
    ***************************************************************************
    **
    **  Reads an image.
    **
    **  Gets the pixel data for an image.
    **
    **  @param  image  The Image object containing the valid metadata for an
    **                 image.  If the contained pixel data byte array is not
    **                 null and is large enough, the pixel data us stored
    **                 there.  Otherwise a new array is created.
    **
    **  @return  True if the pixel data was successfully read, false otherwise.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public boolean readImage(Image image) throws REBException
    {
        checkOpen();
        return impl.getImage(image);
    }


   /**
    ***************************************************************************
    **
    **  Deletes an image's metadata reference.
    **
    **  Deletes referenced C++ object.
    **
    **  @param  image  The image whose metadata reference is to be deleted.
    **
    ***************************************************************************
    */
    public void deleteImageMetadataRef(Image image)
    {
        impl.deleteImageMetadataRef(image);
    }


   /**
    ***************************************************************************
    **
    **  Resets the front end system.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void resetFrontEnd() throws REBException
    {
        checkOpen();
        impl.reset();
    }


   /**
    ***************************************************************************
    **
    **  Sets the data listener.
    **
    **  Sets the data listener object, whose class must implement the Listener
    **  interface, and therefore must contain the processImage method.  This
    **  method is called with a complete Image object as its argument whenever
    **  a new image becomes available.
    **
    **  @param  listener  The listener object to be set as the listener.  Any
    **                    existing listener is replaced.
    **
    **  @param  image     An image object to be used to contain received
    **                    images, or null if a new object is to be created
    **                    each time.
    **
    ***************************************************************************
    */
    public void setListener(Listener listener, Image image)
    {
        clearListener();
        imgListener = listener;
        listenImage = image;
        if (impl != null) {
            reader = new ReadThread(listener, image);
            reader.setDaemon(true);
            reader.start();
        }
    }


   /**
    ***************************************************************************
    **
    **  Clears the data listener.
    **
    **  Clears the data listener object if it exists.
    **
    ***************************************************************************
    */
    public void clearListener()
    {
        if (reader != null) {
            reader.setState(ENDING);
            reader = null;
        }
        imgListener = null;
        listenImage = null;
    }


   /**
    ***************************************************************************
    **
    **  Checks that connection is open
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private void checkOpen() throws REBException
    {
        if (impl == null) {
            throw new REBException("Image connection not open");
        }
    }


   /**
    ***************************************************************************
    **
    **  Inner interface to support choosing the hardware.
    **
    ***************************************************************************
    */
    interface Impl {

       /**
        ***********************************************************************
        **
        **  Creates a new object.
        **
        **  Creates needed C++ objects.
        **
        **  @param  id   The ID of the REB to connect to
        **
        **  @param  ifc  The name of the network interface to use.  If null or
        **               empty, the default interface is used.
        **
        ***********************************************************************
        */
        public void newImageClient(int id, String ifc) throws REBException;


       /**
        ***********************************************************************
        **
        **  Deletes an object.
        **
        **  Deletes referenced C++ objects.
        **
        ***********************************************************************
        */
        public void deleteImageClient();


       /**
        ***********************************************************************
        **
        **  Awaits an image.
        **
        **  Waits until a new image has been generated.
        **
        **  @param  image  An Image object in which to save the reference and
        **                 metadata for the new image, or null if a new image
        **                 object is to be created.
        **
        **  @return  The Image object containing the new image reference and
        **           metadata.
        **
        ***********************************************************************
        */
        public Image waitForImage(Image image);


       /**
        ***********************************************************************
        **
        **  Reads an image.
        **
        **  Gets the pixel data for an image.
        **
        **  @param  image  The Image object containing the valid metadata for an
        **                 image.  If the contained pixel data byte array is not
        **                 null and is large enough, the pixel data us stored
        **                 there.  Otherwise a new array is created.
        **
        **  @return  True if the pixel data was successfully read, false
        **           otherwise.
        **
        ***********************************************************************
        */
        public boolean getImage(Image image);


       /**
        ***********************************************************************
        **
        **  Resets the front end system.
        **
        ***********************************************************************
        */
        public void reset();


       /**
        ***********************************************************************
        **
        **  Deletes an image's metadata reference.
        **
        **  Deletes referenced C++ object.
        **
        **  @param  image  The image whose metadata reference is to be deleted.
        **
        ***********************************************************************
        */
        public void deleteImageMetadataRef(Image image);

    }

}
