package org.lsst.ccs.drivers.reb;

import java.util.HashMap;
import java.util.Map;

/**
 ******************************************************************************
 **
 **  Base register set access routines.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class BaseSet extends RegClient {

   /**
    ***************************************************************************
    **
    **  Public constants.
    **
    ***************************************************************************
    */
    public final static int
        RSET_STATUS       = 0,
        RSET_TIME_BASE    = 1,
        RSET_SEQUENCER    = 2,
        RSET_POWER_ADCS   = 3,
        RSET_TEMP_ADCS    = 4,
        RSET_FAST_ADCS    = 5,

        REG_SCHEMA        = 0x000000,
        REG_VERSION       = 0x000001,
        REG_ID            = 0x000002,
        REG_TIME_BASE     = 0x000004,
        REG_STATE         = 0x000008,
        REG_TRIGGER       = 0x000009,
        REG_TRIG_TIME     = 0x00000a,

        REG_SN_REB_START  = 0x800000,
        REG_SN_REB_VALUE  = 0x800001,
        REG_SN_DREB_START = 0x800010,
        REG_SN_DREB_VALUE = 0x800011,

        REG_DCDC_SYNC     = 0x900000,

        REG_BACK_BIAS     = 0xd00000,

        OPTN_SEQUENCER    = 0,
        OPTN_CABAC        = 1,
        OPTN_ASPIC        = 2,
        OPTN_REB_SN       = 3,
        OPTN_DREB_SN      = 4,
        OPTN_DCDC_SYNC    = 5,
        OPTN_BOARD_DACS   = 6,
        OPTN_BOARD_POWER  = 7,
        OPTN_BOARD_TEMP   = 8,
        OPTN_STATUS       = 9,
        OPTN_CCD_TEMP     = 10,
        OPTN_SLOW_ADCS    = 11,
        OPTN_FAST_ADCS    = 12,
        OPTN_BACK_BIAS    = 13,

        VERSION_UNSET     = -3,
        VERSION_UNKNOWN   = -2,
        VERSION_UNSUPP    = -1,
        VERSION_0         = 0,
        VERSION_1         = 1,
        VERSION_2         = 2,
        VERSION_3         = 3,
        VERSION_4         = 4,
        VERSION_5         = 5,     // WGREB
        VERSION_6         = 6,     // REB3

        TYPE_UNKNOWN      = -1,
        TYPE_SCIENCE      = 0,
        TYPE_CORNER       = 1,

        CLOCK_PERIOD_0    = 5,      // Nanoseconds
        CLOCK_PERIOD_1    = 10;     // Nanoseconds

    public final static long
        SN_READ_OKAY      = 0x0001000000000000L,
        SN_READ_ERROR     = 0x0002000000000000L,
        SN_READ_TIMEOUT   = 0x0004000000000000L,
        SN_VALUE_MASK     = 0x0000ffffffffffffL;

   /**
    ***************************************************************************
    **
    **  Private data.
    **
    ***************************************************************************
    */
    private final static int WAIT_TIMEOUT = 1000;
    private int fwVersion = VERSION_UNSET;
    private int handleInst;
    private final Map<Integer, Integer> versionMap = new HashMap<>();


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public BaseSet()
    {
        super();
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  reg  The associated register client object
    **
    ***************************************************************************
    */
    public BaseSet(RegClient reg)
    {
        ctxt = reg.ctxt;
    }


   /**
    ***************************************************************************
    **
    **  Gets the firmware version.
    **
    **  @return  The firmware version number
    **
    **  @throws  REBException if the firmware version is unknown
    **
    ***************************************************************************
    */
    public int getFwVersion() throws REBException
    {
        setVersions();
        return fwVersion;
    }


   /**
    ***************************************************************************
    **
    **  Gets the version of a particular option.
    **
    **  @param  option  The ID of the option
    **
    **  @return  The option version number
    **
    **  @throws  REBException if the option ID is invalid or is not supported
    **           by the firmware, or the firmware version is unknown
    **
    ***************************************************************************
    */
    public int getVersion(int option) throws REBException
    {
        return getVersion(option, false);
    }


   /**
    ***************************************************************************
    **
    **  Gets the version of a particular option.
    **
    **  @param  option   The ID of the option
    **
    **  @param  unsOkay  If true, returns VERSION_UNSUPP if unsupported
    **
    **  @return  The option version number
    **
    **  @throws  REBException if the option ID is invalid or is not supported
    **           by the firmware, or the firmware version is unknown
    **
    ***************************************************************************
    */
    public int getVersion(int option, boolean unsOkay) throws REBException
    {
        setVersions();
        Integer version = versionMap.get(option);
        if (version == null) {
            throw new REBException("Unrecognized option");
        }
        if (version == VERSION_UNKNOWN) {
            throw new REBException("Unknown firmware version");
        }
        if (version == VERSION_UNSUPP && !unsOkay) {
            throw new REBException("Option not supported by the firmware");
        }

        return version;
    }


   /**
    ***************************************************************************
    **
    **  Tests for a particular option version.
    **
    **  @param  option   The ID of the option
    **
    **  @param  version  The array of possible versions to check against
    **
    **  @return  Whether the versions match
    **
    **  @throws  REBException if the firmware version is not recognized
    **
    ***************************************************************************
    */
    public boolean isVersion(int option, int... version) throws REBException
    {
        int currVersion = getVersion(option, true);
        for (int testVersion : version) {
            if (testVersion == currVersion) return true;
        }

        return false;
        
    }


   /**
    ***************************************************************************
    **
    **  Checks for a particular option version.
    **
    **  @param  option   The ID of the option
    **
    **  @param  version  The array of possible versions to check against
    **
    **  @throws  REBException if the firmware version is not recognized
    **           or doesn't match the specified version
    **
    ***************************************************************************
    */
    public void checkVersion(int option, int... version) throws REBException
    {
        if (!isVersion(option, version)) {
            throwIncompatException();
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks for not a particular option version.
    **
    **  @param  option   The ID of the option
    **
    **  @param  version  The array of possible versions to check against
    **
    **  @throws  REBException if the firmware version is not recognized
    **           or matches the specified version
    **
    ***************************************************************************
    */
    public void checkNotVersion(int option, int... version) throws REBException
    {
        if (isVersion(option, version)) {
            throwIncompatException();
        }
    }


   /**
    ***************************************************************************
    **
    **  Throws incompatible firmware version exception.
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    public void throwIncompatException() throws REBException
    {
        throw new REBException("Incompatible firmware version");
    }


   /**
    ***************************************************************************
    **
    **  Enables the time-base.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void enable() throws REBException
    {
        enable(RSET_TIME_BASE);
    }


   /**
    ***************************************************************************
    **
    **  Disables the time-base.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void disable() throws REBException
    {
        disable(RSET_TIME_BASE);
    }


   /**
    ***************************************************************************
    **
    **  Gets the raw time-base value.
    **
    **  @return  The time-base value as clock ticks
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public long getTimeRaw() throws REBException
    {
        return readLong(REG_TIME_BASE);
    }


   /**
    ***************************************************************************
    **
    **  Gets the time-base value as millisecond Unix time.
    **
    **  @return  The time-base value as millisecond Unix time
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public long getTime() throws REBException
    {
        return getPeriod() * getTimeRaw() / 1000000;
    }


   /**
    ***************************************************************************
    **
    **  Sets the raw time-base value.
    **
    **  @param  time  The time-base value as clock ticks
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void setTimeRaw(long time) throws REBException
    {
        writeLong(REG_TIME_BASE, time);
    }


   /**
    ***************************************************************************
    **
    **  Sets the time-base value as millisecond Unix time.
    **
    **  @param  time  The time-base value as millisecond Unix time
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void setTime(long time) throws REBException
    {
        setTimeRaw(1000000 * time / getPeriod());
    }


   /**
    ***************************************************************************
    **
    **  Sets the time-base value to the current system time.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void setTime() throws REBException
    {
        setTime(System.currentTimeMillis());
    }


   /**
    ***************************************************************************
    **
    **  Enables a register set.
    **
    **  @param  regSet  The register set identifier
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void enable(int regSet) throws REBException
    {
        //        update(REG_TRIGGER, 1<< regSet, 1 << regSet);
        //        write(REG_TRIGGER, read(REG_STATE) | (1 << regSet));
        write(REG_TRIGGER,
              (read(REG_STATE) & ((1 << RSET_STATUS) | (1 << RSET_TIME_BASE)))
                | (1 << regSet));
    }


   /**
    ***************************************************************************
    **
    **  Disables a register set.
    **
    **  @param  regSet  The register set identifier
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void disable(int regSet) throws REBException
    {
        //        update(REG_TRIGGER, 1<< regSet, 0);
        //        write(REG_TRIGGER, read(REG_STATE) & ~(1 << regSet));
        write(REG_TRIGGER,
              (read(REG_STATE) & ((1 << RSET_STATUS) | (1 << RSET_TIME_BASE)))
                & ~(1 << regSet));
    }


   /**
    ***************************************************************************
    **
    **  Waits for a register set to become disabled.
    **
    **  This generally indicates the completion of a read operation
    **
    **  @param  regSet  The register set identifier
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void waitDone(int regSet) throws REBException
    {
        long limit = System.currentTimeMillis() + WAIT_TIMEOUT;
        boolean busy = true;
        while (busy && System.currentTimeMillis() < limit) {
            busy = (read(REG_STATE) & (1 << regSet)) != 0;
        }
        if (busy) {
            throw new REBException("Completion wait timeout");
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the raw trigger time for a register set.
    **
    **  @param  regSet  The register set identifier
    **
    **  @return  The trigger time as clock ticks
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public long getTriggerTimeRaw(int regSet) throws REBException
    {
        return readLong(REG_TRIG_TIME + 2 * regSet);
    }


   /**
    ***************************************************************************
    **
    **  Gets the trigger time for a register set as Unix millisecond time.
    **
    **  @param  regSet  The register set identifier
    **
    **  @return  The trigger time as Unix millisecond time
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public long getTriggerTime(int regSet) throws REBException
    {
        return getPeriod() * getTriggerTimeRaw(regSet) / 1000000;
    }


   /**
    ***************************************************************************
    **
    **  Gets the REB serial number.
    **
    **  @return  The 48-bit REB serial number
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public long getRebSerial() throws REBException
    {
        getVersion(OPTN_REB_SN);
        write(REG_SN_REB_START, 0);
        long value = readLong(REG_SN_REB_VALUE);
        if ((value & SN_READ_OKAY) == 0) {
            String problem = (value & SN_READ_ERROR) != 0 ? "error" : "timeout";
            throw new REBException("Serial number read " + problem);
        }
        return value & SN_VALUE_MASK;
    }


   /**
    ***************************************************************************
    **
    **  Gets the DREB serial number.
    **
    **  @return  The 48-bit DREB serial number
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public long getDrebSerial() throws REBException
    {
        getVersion(OPTN_DREB_SN);
        write(REG_SN_DREB_START, 0);
        long value = readLong(REG_SN_DREB_VALUE);
        if ((value & SN_READ_OKAY) == 0) {
            String problem = (value & SN_READ_ERROR) != 0 ? "error" : "timeout";
            throw new REBException("Serial number read " + problem);
        }
        return value & SN_VALUE_MASK;
    }


   /**
    ***************************************************************************
    **
    **  Sets the DC/DC synchronization.
    **
    **  @param  enable  If true, enable synchronization; otherwise disable it
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void setDcdcSync(boolean enable) throws REBException
    {
        getVersion(OPTN_DCDC_SYNC);
        write(REG_DCDC_SYNC, enable ? 1 : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the DC/DC synchronization.
    **
    **  @return  Whether DC/DC synchronization is in effect
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public boolean isDcdcSync() throws REBException
    {
        getVersion(OPTN_DCDC_SYNC);
        return (read(REG_DCDC_SYNC) & 1) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets the back bias state.
    **
    **  @param  on  If true, turn the back bias on; otherwise turn it off
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void setBackBias(boolean on) throws REBException
    {
        getVersion(OPTN_BACK_BIAS);
        write(REG_BACK_BIAS, on ? 1 : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the back bias setting.
    **
    **  @return  Whether the back bias is turned on
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public boolean isBackBiasOn() throws REBException
    {
        getVersion(OPTN_BACK_BIAS);
        return (read(REG_BACK_BIAS) & 1) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets the version of each option.
    **
    **  @throws  REBException if the firmware version is not recognized
    **
    ***************************************************************************
    */
    public void setVersions() throws REBException
    {
        try {
            checkOpen();
        }
        catch (REBException e) {
            fwVersion = VERSION_UNSET;
            throw e;
        }
        if (fwVersion != VERSION_UNSET && handleInst == ctxt.instance) {
            return;
        }

        handleInst = ctxt.instance;
        int schema = read(REG_SCHEMA);
        int hwVersion = read(REG_VERSION);
        int pgpVersion = hwVersion >>> 16;
        hwVersion &= 0xffff;
        if (schema == 0x00f1de00) {
            fwVersion = VERSION_0;
        }
        else if (pgpVersion == 0xb01c) {
            if (schema == 0x00000001) {
                fwVersion = VERSION_1;
            }
            else if (schema == 0x00000002) {
                fwVersion = VERSION_2;
            }
            else if (schema == 0x00000003) {
                fwVersion = VERSION_3;
            }
            else if (schema == 0x00000004) {
                fwVersion = VERSION_4;
            }
        }
        else if (pgpVersion == 0xb01e || pgpVersion == 0xb01f
                   || pgpVersion == 0xb020 || pgpVersion == 0xb022) {
            fwVersion = hwVersion < 0x100 ? VERSION_5 : VERSION_6;
        }
        else {
            fwVersion = VERSION_UNKNOWN;
        }

        int version;

        version = fwVersion < VERSION_0 ? VERSION_UNKNOWN
                    : fwVersion == VERSION_0 ? VERSION_0
                    : fwVersion <= VERSION_3 ? VERSION_1
                    : fwVersion <= VERSION_4 ? VERSION_2 : VERSION_3;
        versionMap.put(OPTN_SEQUENCER, version);

        version = fwVersion < VERSION_0 ? VERSION_UNKNOWN
                    : fwVersion < VERSION_2 ? VERSION_UNSUPP
                    : fwVersion < VERSION_5 ? VERSION_0
                    : fwVersion < VERSION_6 ? VERSION_1 : VERSION_UNSUPP;
        versionMap.put(OPTN_CABAC, version);

        version = fwVersion < VERSION_0 ? VERSION_UNKNOWN
                    : fwVersion < VERSION_4 ? VERSION_UNSUPP
                    : fwVersion == VERSION_4 ? VERSION_0 : VERSION_1; 
        versionMap.put(OPTN_ASPIC, version);

        version = fwVersion < VERSION_0 ? VERSION_UNKNOWN
                    : fwVersion < VERSION_3 ? VERSION_UNSUPP : VERSION_0;
        versionMap.put(OPTN_REB_SN, version);
        versionMap.put(OPTN_DCDC_SYNC, version);

        version = fwVersion < VERSION_0 ? VERSION_UNKNOWN
                    : fwVersion < VERSION_3 ? VERSION_UNSUPP
                    : fwVersion < VERSION_5 ? VERSION_0 : VERSION_UNSUPP;
        versionMap.put(OPTN_DREB_SN, version);

        version = fwVersion < VERSION_0 ? VERSION_UNKNOWN
                    : fwVersion == VERSION_0 ? VERSION_0
                    : fwVersion < VERSION_5 ? VERSION_1
                    : fwVersion < VERSION_6 ? VERSION_2 : VERSION_3;
        versionMap.put(OPTN_BOARD_DACS, version);

        version = fwVersion < VERSION_0 ? VERSION_UNKNOWN
                    : fwVersion == VERSION_0 ? VERSION_UNSUPP
                    : fwVersion == VERSION_1 ? VERSION_0
                    : fwVersion < VERSION_5 ? VERSION_1
                    : fwVersion < VERSION_6 ? VERSION_2 : VERSION_3;
        versionMap.put(OPTN_BOARD_POWER, version);
        versionMap.put(OPTN_BOARD_TEMP, version);

        version = fwVersion < VERSION_0 ? VERSION_UNKNOWN
                    : fwVersion < VERSION_3 ? VERSION_0 : VERSION_1;
        versionMap.put(OPTN_STATUS, version);

        version = fwVersion < VERSION_0 ? VERSION_UNKNOWN
                    : fwVersion < VERSION_4 ? VERSION_UNSUPP : VERSION_0; 
        versionMap.put(OPTN_CCD_TEMP, version);

        version = fwVersion < VERSION_0 ? VERSION_UNKNOWN
                    : fwVersion < VERSION_5 ? VERSION_UNSUPP
                    : fwVersion < VERSION_6 ? VERSION_0 : VERSION_1;
        versionMap.put(OPTN_SLOW_ADCS, version);

        version = fwVersion < VERSION_0 ? VERSION_UNKNOWN
                    : fwVersion != VERSION_5 ? VERSION_UNSUPP : VERSION_0; 
        versionMap.put(OPTN_FAST_ADCS, version);

        version = fwVersion < VERSION_0 ? VERSION_UNKNOWN
                    : fwVersion < VERSION_5 ? VERSION_UNSUPP : VERSION_0; 
        versionMap.put(OPTN_BACK_BIAS, version);

        if (fwVersion == VERSION_UNKNOWN) {
            throw new REBException("Unknown firmware version");
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the clock period.
    **
    **  @return  The board's clock period in 10 ns units
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private int getPeriod() throws REBException
    {
        return fwVersion == VERSION_0 ? CLOCK_PERIOD_0 : CLOCK_PERIOD_1;
    }

}
