package org.lsst.ccs.drivers.reb;

import java.io.IOException;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;

/**
 ******************************************************************************
 **
 **  Program to test the Java REB register access routines
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class TestReg {

    /*
    **  Private fields
    */
    private final static Map<String, Integer> cabacMap = new LinkedHashMap<>();
    static {
        cabacMap.put("OD_EXP0", Cabac.CABAC_FLD_OD_EXP0);
        cabacMap.put("OD_EXP1", Cabac.CABAC_FLD_OD_EXP1);
        cabacMap.put("OD_RDO0", Cabac.CABAC_FLD_OD_RDO0);
        cabacMap.put("OD_RDO1", Cabac.CABAC_FLD_OD_RDO1);
        cabacMap.put("PCLK0", Cabac.CABAC_FLD_PCLK0);
        cabacMap.put("PCLK1", Cabac.CABAC_FLD_PCLK1);
        cabacMap.put("PCLK2", Cabac.CABAC_FLD_PCLK2);
        cabacMap.put("PCLK3", Cabac.CABAC_FLD_PCLK3);
        cabacMap.put("SCLK0", Cabac.CABAC_FLD_SCLK0);
        cabacMap.put("SCLK1", Cabac.CABAC_FLD_SCLK1);
        cabacMap.put("SCLK2", Cabac.CABAC_FLD_SCLK2);
        cabacMap.put("RG", Cabac.CABAC_FLD_RG);
        cabacMap.put("GD", Cabac.CABAC_FLD_GD);
        cabacMap.put("OG", Cabac.CABAC_FLD_OG);
        cabacMap.put("RD", Cabac.CABAC_FLD_RD);
        cabacMap.put("SPH", Cabac.CABAC_FLD_SPH);
        cabacMap.put("SPL", Cabac.CABAC_FLD_SPL);
        cabacMap.put("MUX0", Cabac.CABAC_FLD_MUX0);
        cabacMap.put("MUX1", Cabac.CABAC_FLD_MUX1);
        cabacMap.put("PULSE", Cabac.CABAC_FLD_PULSE);
        cabacMap.put("MUXE", Cabac.CABAC_FLD_MUXE);
    }

    private final static String CUP = "\033[A";  // Move cursor up
    private final BaseSet bset = new BaseSet();
    private final SequencerUtils seq = new SequencerUtils(bset);
    private final StatusSet sset = new StatusSet(bset);
    private final TempAdcs tmp = new TempAdcs(bset);
    private final PowerAdcs pwr = new PowerAdcs(bset);
    private final BoardDacs dac = new BoardDacs(bset);
    private final Cabac cab = new Cabac(bset);
    private final Aspic asp = new Aspic(bset);
    private final BlockingQueue<Integer> consQueue = new ArrayBlockingQueue<>(1);
    private int rebId = -1;
    private boolean contActive;


   /**
    ***************************************************************************
    **
    **  Enumeration for register set identifiers.
    **
    ***************************************************************************
    */
    public enum RsetId {

        STATUS(BaseSet.RSET_STATUS),
        TIMEBASE(BaseSet.RSET_TIME_BASE),
        SEQUENCER(BaseSet.RSET_SEQUENCER),
        POWERADCS(BaseSet.RSET_POWER_ADCS),
        TEMPADCS(BaseSet.RSET_TEMP_ADCS);

        int value;

        RsetId(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }


   /**
    ***************************************************************************
    **
    **  Enumeration for scan mode options.
    **
    ***************************************************************************
    */
    public enum ScanOption {
        ON, OFF, RESET;
    }


   /**
    ***************************************************************************
    **
    **  Enumeration for on/off option.
    **
    ***************************************************************************
    */
    public enum OnOff {
        ON, OFF;
    }


   /**
    ***************************************************************************
    **
    **  Enumeration for CABAC field names.
    **
    ***************************************************************************
    */
    public enum CabacField {

        OD_EXP0(Cabac.CABAC_FLD_OD_EXP0),
        OD_EXP1(Cabac.CABAC_FLD_OD_EXP1),
        OD_RDO0(Cabac.CABAC_FLD_OD_RDO0),
        OD_RDO1(Cabac.CABAC_FLD_OD_RDO1),
        PCLK0(Cabac.CABAC_FLD_PCLK0),
        PCLK1(Cabac.CABAC_FLD_PCLK1),
        PCLK2(Cabac.CABAC_FLD_PCLK2),
        PCLK3(Cabac.CABAC_FLD_PCLK3),
        SCLK0(Cabac.CABAC_FLD_SCLK0),
        SCLK1(Cabac.CABAC_FLD_SCLK1),
        SCLK2(Cabac.CABAC_FLD_SCLK2),
        RG(Cabac.CABAC_FLD_RG),
        GD(Cabac.CABAC_FLD_GD),
        OG(Cabac.CABAC_FLD_OG),
        RD(Cabac.CABAC_FLD_RD),
        MUX0(Cabac.CABAC_FLD_MUX0),
        MUX1(Cabac.CABAC_FLD_MUX1),
        PULSE(Cabac.CABAC_FLD_PULSE),
        MUXE(Cabac.CABAC_FLD_MUXE),
        SPH(Cabac.CABAC_FLD_SPH),
        SPL(Cabac.CABAC_FLD_SPL);

        int value;

        CabacField(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }


   /**
    ***************************************************************************
    **
    **  Enumeration for DAC names.
    **
    ***************************************************************************
    */
    public enum DacName {

        RAILS (BoardDacs.DAC_CLOCK_RAILS),
        CSGATE(BoardDacs.DAC_CS_GATE),
        CABAC (BoardDacs.DAC_CABAC_ALT);

        int value;

        DacName(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }


   /**
    ***************************************************************************
    **
    **  Enumeration for DAC channel names.
    **
    ***************************************************************************
    */
    public enum DacChan {

        SCLKL  (BoardDacs.CHAN_SCLK_L),
        SCLKLSH(BoardDacs.CHAN_SCLK_L_SH),
        SCLKH  (BoardDacs.CHAN_SCLK_H),
        SCLKHSH(BoardDacs.CHAN_SCLK_H_SH),
        RGL    (BoardDacs.CHAN_RG_L),
        RGLSH  (BoardDacs.CHAN_RG_L_SH),
        RGH    (BoardDacs.CHAN_RG_H),
        RGHSH  (BoardDacs.CHAN_RG_H_SH),
        PCLKL  (BoardDacs.CHAN_PCLK_L),
        PCLKLSH(BoardDacs.CHAN_PCLK_L_SH),
        PCLKH  (BoardDacs.CHAN_PCLK_H),
        PCLKHSH(BoardDacs.CHAN_PCLK_H_SH),
        HEATER1(BoardDacs.CHAN_HEATER_1),
        HEATER2(BoardDacs.CHAN_HEATER_2),
        CSGATE1(BoardDacs.CHAN_CSGATE_1),
        CSGATE2(BoardDacs.CHAN_CSGATE_2),
        CSGATE3(BoardDacs.CHAN_CSGATE_3),
        FSBCTRL(BoardDacs.CHAN_FSB_CTRL),
        ODCTRL (BoardDacs.CHAN_OD_CTRL),
        GD     (BoardDacs.CHAN_GD),
        RD     (BoardDacs.CHAN_RD),
        OG     (BoardDacs.CHAN_OG),
        OGSH   (BoardDacs.CHAN_OG_SH);

        int value;

        DacChan(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }


   /**
    ***************************************************************************
    **
    **  Inner class for detecting console input while reading continuously.
    **
    ***************************************************************************
    */
    class ConsThread extends Thread {

        @Override
        public void run()
        {
            while (true) {
                take(consQueue);
                while (contActive) {
                    awaitTerminal();
                    if (contActive) {
                        clearTerminal();
                    }
                    contActive = false;
                }
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public TestReg()
    {
        ConsThread cons = new ConsThread();
        cons.setDaemon(true);
        cons.start();
    }


   /**
    ***************************************************************************
    **
    **  Connects to a REB.
    **
    **  @param  id  The REB ID
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="connect", description="Connect to a REB")
    public void connect(@Argument(name="id", description="REB ID")
                        int id) throws REBException
    {
        rebId = id;
        bset.open(rebId);
    }


   /**
    ***************************************************************************
    **
    **  Connects to a REB.
    **
    **  @param  id       The REB ID
    **
    **  @param  ifcName  The network interface name
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="connect", description="Connect to a REB")
    public void connect(@Argument(name="id", description="REB ID")
                        int id,
                        @Argument(name="ifcname",
                                  description="Network interface name")
                        String ifcName) throws REBException
    {
        rebId = id;
        bset.open(rebId, ifcName);
    }


   /**
    ***************************************************************************
    **
    **  Disconnects from a REB.
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="disconnect", description="Disconnect from a REB")
    public void disconnect() throws REBException
    {
        bset.close();
        rebId = -1;
    }


   /**
    ***************************************************************************
    **
    **  Shows connection parameters.
    **
    **  @return  The result string
    **
    ***************************************************************************
    */
    @Command(name="show", description="Show connection parameters")
    public String show()
    {
        boolean connected = false;
        int version = -1;
        try {
             bset.checkOpen();
             connected = true;
        }
        catch (REBException e) {
        }
        try {
            version = bset.getFwVersion();
        }
        catch (REBException e) {
        }
        return   "Connection parameters:" +
               "\n  Connected  = " + connected +
               "\n  REB id     = " + rebId +
               "\n  FWVersion  = " + version;
    }


   /**
    ***************************************************************************
    **
    **  Shows register contents.
    **
    **  @param  address  The address of the register to read
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="read", description="Show one register")
    public String read(@Argument(name="address",
                                 description="Register address to read")
                       int address) throws REBException
    {
        return String.format("%06x: %08x", address, bset.read(address));
    }


   /**
    ***************************************************************************
    **
    **  Shows register contents.
    **
    **  @param  address  The address of the first register to read
    **
    **  @param  count    The number of registers to read
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="read", description="Show one or more registers")
    public String read(@Argument(name="address",
                                 description="First register address to read")
                       int address,
                       @Argument(name="count",
                                 description="Number of registers to read")
                       int count) throws REBException
    {
        int[] values = new int[count];
        bset.read(address, values);
        StringBuilder result = new StringBuilder(9 * (count + (count + 3) / 4));
        for (int j = 0; j < count; j++, address++) {
            if ((j & 3) == 0) {
                if (j != 0) result.append("\n");
                result.append(String.format("%06x:", address));
            }
            result.append(String.format(" %08x", values[j]));
        }

        return result.toString();
    }


   /**
    ***************************************************************************
    **
    **  Writes to a register.
    **
    **  @param  address  The address of the register to write
    **
    **  @param  value    The value to write
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="write", description="Write a register")
    public void write(@Argument(name="address",
                                description="Register address to write")
                      int address,
                      @Argument(name="value", description="Value to write")
                      int value) throws REBException
    {
        bset.write(address, value);
    }


   /**
    ***************************************************************************
    **
    **  Updates a register.
    **
    **  @param  address  The address of the register to update
    **
    **  @param  mask     The mask of bits to update
    **
    **  @param  value    The value to write under the mask
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="update", description="Update a register")
    public void update(@Argument(name="address",
                                 description="Address of register to update")
                       int address,
                       @Argument(name="mask",
                                 description="Mask of bits to update")
                       int mask,
                       @Argument(name="value", description="New value to write")
                       int value) throws REBException
    {
        bset.update(address, mask, value);
    }


   /**
    ***************************************************************************
    **
    **  Shows a pair of registers.
    **
    **  @param  address  The address of the first register to read
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="readlong", description="Show a pair of registers")
    public String readLong(@Argument(name="address",
                                     description="Address of first register to read")
                           int address) throws REBException
    {
        return String.format("%06x: %016x\n", address, bset.readLong(address));
    }


   /**
    ***************************************************************************
    **
    **  Writes a pair of registers.
    **
    **  @param  address  The address of the first register to write
    **
    **  @param  value    The 8-byte little-endian value to write
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="writelong", description="Write a pair of registers")
    public void writeLong(@Argument(name="address",
                                    description="Address of first register to write")
                          int address,
                          @Argument(name="value", description="Value to write")
                          long value) throws REBException
    {
        bset.writeLong(address, value);
    }


   /**
    ***************************************************************************
    **
    **  Enables a register set.
    **
    **  @param  regset  The register set: STATUS, TIMEBASE, SEQUENCER,
    **                  POWERADCS or TEMPADCS
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="enable", description="Enable a register set")
    public void enable(@Argument(name="regset",
                                 description="Register set name")
                       RsetId regset) throws REBException
    {
        bset.enable(regset.getValue());
    }


   /**
    ***************************************************************************
    **
    **  Disables a register set.
    **
    **  @param  regset  The register set: STATUS, TIMEBASE, SEQUENCER,
    **                  POWERADCS or TEMPADCS
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="disable", description="Disable a register set")
    public void disable(@Argument(name="regset",
                                 description="Register set name")
                        RsetId regset) throws REBException
    {
        bset.disable(regset.getValue());
    }


   /**
    ***************************************************************************
    **
    **  Shows a trigger time.
    **
    **  @param  regset  The register set: STATUS, TIMEBASE, SEQUENCER,
    **                  POWERADCS or TEMPADCS
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="trigtime", description="Show a trigger time")
    public String trigtime(@Argument(name="regset",
                                     description="Register set name")
                           RsetId regset) throws REBException
    {
        long time = bset.getTriggerTime(regset.getValue());
        GregorianCalendar tm = new GregorianCalendar();
        tm.setTimeInMillis(time);

        return String.format("Time = %tY-%<tm-%<td %<tH:%<tM:%<tS.%<tL", tm);
    }


   /**
    ***************************************************************************
    **
    **  Shows the time.
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="time", description="Show the time")
    public String time() throws REBException
    {
        long time = bset.getTime();
        GregorianCalendar tm = new GregorianCalendar();
        tm.setTimeInMillis(time);

        return String.format("Time = %tY-%<tm-%<td %<tH:%<tM:%<tS.%<tL", tm);
    }


   /**
    ***************************************************************************
    **
    **  Sets the time.
    **
    **  @param  time  The time to set (yyyy-mm-dd hh:mm:ss)
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="time", description="Set the time")
    public String setTime(@Argument(name="time", description="The time to set")
                          String time) throws REBException
    {
        String result = CUP;    // Move cursor up
        if (time.equals(".")) {
            bset.setTime();
        }
        else {
            SimpleDateFormat fmt = new SimpleDateFormat();
            fmt.setLenient(true);
            fmt.applyPattern("y-M-d H:m:s");
            ParsePosition pos = new ParsePosition(0);
            Date date = fmt.parse(time, pos);
            if (date == null) {
                result = "Expected format is yyyy-mm-dd hh:mm:ss";
            }
            else {
                bset.setTime(date.getTime());
            }
        }

        return result;
    }


   /**
    ***************************************************************************
    **
    **  Shows the time base.
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="timebase", description="Show the time base")
    public String timebase() throws REBException
    {
        return "Value = " + bset.getTimeRaw();
    }


   /**
    ***************************************************************************
    **
    **  Sets the time base.
    **
    **  @param  time  The time base value
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="timebase", description="Set the time base")
    public void timebase(@Argument(name="time", description="The value to set")
                         long time) throws REBException
    {
        bset.setTimeRaw(time);
    }


   /**
    ***************************************************************************
    **
    **  Loads a test program into the sequencer.
    **
    **  @param  totTime  The total time per cycle (in clock [10 ns] units)
    **
    **  @param  ncycle   The number of cycles
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="loadtest", description="Load sequencer with test program")
    public void loadTest(@Argument(name="totTime",
                                   description="The total time per cycle")
                         int totTime,
                         @Argument(name="ncycle",
                                   description="The number of cycles")
                         int ncycle) throws REBException
    {
        loadTest(totTime, ncycle, 10);
    }


   /**
    ***************************************************************************
    **
    **  Loads a test program into the sequencer.
    **
    **  @param  totTime  The total time per cycle (in clock [10 ns] units)
    **
    **  @param  ncycle   The number of cycles
    **
    **  @param  onTime   The time the ADC read is to take (in clock units)
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="loadtest", description="Load sequencer with test program")
    public void loadTest(@Argument(name="totTime",
                                   description="The total time per cycle")
                         int totTime,
                         @Argument(name="ncycle",
                                   description="The number of cycles")
                         int ncycle,
                         @Argument(name="ontime",
                                   description="The time the ADC is on per cycle")
                         int onTime) throws REBException
    {
        seq.writeLines(0, new int[]{0});          // Default state
        seq.writeLines(1, new int[]{0x1000, 0});  // ADC trigger on, then off
        seq.writeTimes(1, new int[]{onTime, totTime - onTime, 0});
        if (seq.getVersion(Sequencer.OPTN_SEQUENCER) == Sequencer.VERSION_0) {
            seq.writeStack(new int[]{0x10000000 | (ncycle & 0x03ffffff)});
            seq.writeDataSource(Sequencer.SEQ_SRC_FPGA_PTN);
        }
        else {
            seq.writeProgExec(0, 1, ncycle);
            seq.writeProgEnd(1);
            seq.writeDataSource(Sequencer.SEQ_SRC_PATTERN);
        }
        seq.writeSliceCount(ncycle);
    }


   /**
    ***************************************************************************
    **
    **  Loads a command into the sequencer.
    **
    **  @param  command  The command to load
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="loadcommand", description="Load command into sequencer")
    public void loadCommand(@Argument(name="command",
                                      description="The command to load")
                            String command) throws REBException
    {
        seq.loadCommand(command);
    }


   /**
    ***************************************************************************
    **
    **  Loads a file of commands into the sequencer.
    **
    **  @param  fileName  The name of the command file to load
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **  @throws  IOException
    **
    ***************************************************************************
    */
    @Command(name="loadfile", description="Load command file into sequencer")
    public String loadFile(@Argument(name="filename",
                                     description="The name of the file to load")
                           String fileName) throws REBException, IOException
    {
        int nSlice = seq.loadFile(fileName);
        return "Slice count = " + nSlice;
    }


   /**
    ***************************************************************************
    **
    **  Sends a stop request to the sequencer.
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="stop", description="Send a stop request to the sequencer")
    public void stop() throws REBException
    {
        seq.sendStop();
    }


   /**
    ***************************************************************************
    **
    **  Sends a step request to the sequencer.
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="step", description="Send a step request to the sequencer")
    public void step() throws REBException
    {
        seq.sendStep();
    }


   /**
    ***************************************************************************
    **
    **  Shows the slice count register.
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="slice", description="Show the slice count")
    public String sliceCount() throws REBException
    {
        return "Slice count = " + seq.readSliceCount();
    }


   /**
    ***************************************************************************
    **
    **  Sets the slice count register.
    **
    **  @param  count  The value to set
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="slice", description="Set the slice count")
    public void sliceCount(@Argument(name="count",
                                     description="The value to set")
                           int count) throws REBException
    {
        seq.writeSliceCount(count);
    }


   /**
    ***************************************************************************
    **
    **  Shows the data source register.
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="source", description="Show the data source")
    public String dataSource() throws REBException
    {
        return "Data source = " + seq.readDataSource();
    }


   /**
    ***************************************************************************
    **
    **  Sets the data source register.
    **
    **  @param  value  The value to set: 0 = sensor; 1 = simulator
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="source", description="Set the data source")
    public void dataSource(@Argument(name="value",
                                     description="The value to set")
                           int value) throws REBException
    {
        seq.writeDataSource(value);
    }


   /**
    ***************************************************************************
    **
    **  Shows the stripe selection register.
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="stripe", description="Show the stripe selection")
    public String stripeSelect() throws REBException
    {
        return "Stripe selection = " + seq.readStripeSelect();
    }


   /**
    ***************************************************************************
    **
    **  Sets the stripe selection register.
    **
    **  @param  value  The value to set: 0 - 7
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="stripe", description="Set the stripe selection")
    public void stripeSelect(@Argument(name="value",
                                       description="The value to set")
                             int value) throws REBException
    {
        seq.writeStripeSelect(value);
    }


   /**
    ***************************************************************************
    **
    **  Shows the scan mode.
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="scan", description="Show the scan mode")
    public String scanMode() throws REBException
    {
        return "Scan mode is " + (seq.isScanEnabled() ? "on" : "off");
    }


   /**
    ***************************************************************************
    **
    **  Sets/resets the scan mode.
    **
    **  @param  option  The scan mode option (ON, OFF or RESET)
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="scan", description="Set/reset the scan mode")
    public void sliceCount(@Argument(name="option",
                                     description="The scan mode option")
                           ScanOption option) throws REBException
    {
        if (option == ScanOption.RESET) {
            seq.resetScan();
        }
        else {
            seq.enableScan(option == ScanOption.ON);
        }
    }


   /**
    ***************************************************************************
    **
    **  Shows the contents of the status block.
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="status", description="Show status block contents")
    public String showStatus() throws REBException
    {
        Status status = sset.readStatus();
        StringBuilder result = new StringBuilder();
        result.append("Status Block:\n");
        result.append(String.format("  Version               = %08x  ",
                                    status.getVersion()));
        result.append(String.format("  Link Status           = %08x\n",
                                    status.getLinkStatus()));
        result.append(String.format("  Cell error count      = %-8s  ",
                                    status.getCellErrorCount()));
        result.append(String.format("  Link down count       = %s\n",
                                    status.getLinkDownCount()));
        result.append(String.format("  Link error count      = %-8s  ",
                                    status.getLinkErrorCount()));
        result.append(String.format("  VC buffer status      = %08x\n",
                                    status.getVcBufferStatus()));
        int[] counts = status.getVcRxCounts();
        result.append("  VC rcve counts        =");
        for (int j = 0; j < counts.length; j++) {
            result.append(String.format(" %s", counts[j]));
        }
        result.append("\n");
        counts = status.getVcTxCounts();
        result.append("  VC xmit counts        =");
        for (int j = 0; j < counts.length; j++) {
            result.append(String.format(" %s", counts[j]));
        }
        result.append("\n");
        result.append(String.format("  Sent image count      = %-8s  ",
                                    status.getSentCount()));
        result.append(String.format("  Discarded image count = %s\n",
                                    status.getDiscCount()));
        result.append(String.format("  Truncated image count = %-8s  ",
                                    status.getTruncCount()));
        result.append(String.format("  Image format          = %s",
                                    status.getFormat()));

        return result.toString();
    }


   /**
    ***************************************************************************
    **
    **  Shows all the board temperatures.
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="temperature", description="Show all board temperatures")
    public String temperature() throws REBException
    {
        return formatAdcs(tmp.readAdcs(), 0);
    }


   /**
    ***************************************************************************
    **
    **  Shows a board temperature.
    **
    **  @param  first  The index (0 - 11) of the temperature sensor
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="temperature", description="Show a board temperature")
    public String temperature(@Argument(name="first",
                                        description="The temperature to show")
                              int first) throws REBException
    {
        double values[] = new double[1];
        values[0] = tmp.readAdc(first);

        return formatAdcs(values, first);
    }


   /**
    ***************************************************************************
    **
    **  Shows board temperatures.
    **
    **  @param  first  The index (0 - 11) of the first temperature sensor
    **
    **  @param  count  The number of temperatures to show
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="temperature", description="Show board temperatures")
    public String temperature(@Argument(name="first",
                                        description="The first temperature to show")
                              int first,
                              @Argument(name="count",
                                        description="The number of temperatures to show")
                              int count) throws REBException
    {
        return formatAdcs(tmp.readAdcs(first, count), first);
    }


   /**
    ***************************************************************************
    **
    **  Shows all the board power values.
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="power", description="Show all board power values")
    public String power() throws REBException
    {
        return formatAdcs(pwr.readAdcs(), 0);
    }


   /**
    ***************************************************************************
    **
    **  Shows a board power value.
    **
    **  @param  first  The index (0 - 7) of the power value to show
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="power", description="Show a board power value")
    public String power(@Argument(name="first",
                                  description="The power value to show")
                        int first) throws REBException
    {
        double values[] = new double[1];
        values[0] = pwr.readAdc(first);

        return formatAdcs(values, first);
    }


   /**
    ***************************************************************************
    **
    **  Shows board power values.
    **
    **  @param  first  The index (0 - 7) of the first power value to show
    **
    **  @param  count  The number of values to show
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="power", description="Show board power values")
    public String power(@Argument(name="first",
                                  description="The first power value to show")
                        int first,
                        @Argument(name="count",
                                  description="The number of power values to show")
                        int count) throws REBException
    {
        return formatAdcs(pwr.readAdcs(first, count), first);
    }


   /**
    ***************************************************************************
    **
    **  Shows the DAC version.
    **
    **  @return  The result string
    **
    ***************************************************************************
    */
    @Command(name="dacversion", description="Show DAC firmware version")
    public String dacVersion()
    {
        return "DAC version = " + dac.getVersion();
    }


   /**
    ***************************************************************************
    **
    **  Sets a DAC value.
    **
    **  @param  chan    The channel name
    **
    **  @param  value   The value to set
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="dacset", description="Set a DAC value")
    public void dacSet(@Argument(name="chan",
                                 description="The channel name")
                       DacChan chan,
                       @Argument(name="value", description="The value to set")
                       int value) throws REBException
    {
        dac.set(chan.getValue(), value);
    }


   /**
    ***************************************************************************
    **
    **  Loads a DAC.
    **
    **  @param  name  The DAC name
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="dacload", description="Load values into a DAC")
    public void dacLoad(@Argument(name="dac",
                                  description="The DAC name")
                        DacName name) throws REBException
    {
        dac.load(name.getValue());
    }


   /**
    ***************************************************************************
    **
    **  Loads all DACs.
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="dacload", description="Load values into all DACs")
    public void dacLoad() throws REBException
    {
        dac.load();
    }


   /**
    ***************************************************************************
    **
    **  Shows the CABAC version.
    **
    **  @return  The result string
    **
    ***************************************************************************
    */
    @Command(name="cabacversion", description="Show CABAC firmware version")
    public String cabacacVersion()
    {
        return "CABAC version = " + cab.getVersion();
    }


   /**
    ***************************************************************************
    **
    **  Sets the CABAC power regulator enables.
    **
    **  @param  enables  The mask of regulator enables
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="cabacpower", description="Set CABAC power regulator enables")
    public void cabacacPower(@Argument(name="enables",
                                       description="The regulator enable bits")
                             int enables) throws REBException
    {
        cab.setRegulator(enables);
    }


   /**
    ***************************************************************************
    **
    **  Shows the CABAC power regulator enables.
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="cabacpower", description="Show CABAC power regulator")
    public String cabacacPower() throws REBException
    {
        return String.format("CABAC power enables = 0x%02x",
                             cab.getRegulator());
    }


   /**
    ***************************************************************************
    **
    **  Fetches a CABAC stripe.
    **
    **  @param  stripe  The CABAC stripe number (0 - 2)
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="cabacfetch", description="Fetch CABAC stripe")
    public void cabacFetch(@Argument(name="stripe",
                                     description="The stripe number (0 - 2)")
                           int stripe) throws REBException
    {
        cab.fetch(stripe);
    }


   /**
    ***************************************************************************
    **
    **  Loads a CABAC stripe.
    **
    **  @param  stripe  The CABAC stripe number (0 - 2)
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="cabacload", description="Load CABAC stripe")
    public void cabacLoad(@Argument(name="stripe",
                                    description="The stripe number (0 - 2)")
                          int stripe) throws REBException
    {
        cab.load(stripe);
    }


   /**
    ***************************************************************************
    **
    **  Clears a CABAC stripe.
    **
    **  @param  stripe  The CABAC stripe number (0 - 2)
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="cabacclear", description="Clear CABAC stripe")
    public void cabacClear(@Argument(name="stripe",
                                     description="The stripe number (0 - 2)")
                           int stripe) throws REBException
    {
        cab.reset(stripe);
    }


   /**
    ***************************************************************************
    **
    **  Reads the input registers of a CABAC.
    **
    **  @param  cabac  The CABAC number: 0 = top; 1 = bottom
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="cabacread", description="Show CABAC input registers")
    public String cabacRead(@Argument(name="cabac",
                                      description="The CABAC number (0 or 1)")
                            int cabac) throws REBException
    {
        int[] data = cab.readRaw(cabac);
        StringBuilder result = new StringBuilder("Values =");
        for (int value : data) {
            result.append(String.format(" 0x%08x", value));
        }

        return result.toString();
    }


   /**
    ***************************************************************************
    **
    **  Writes the output registers of a CABAC.
    **
    **  @param  cabac   The CABAC number: 0 = top; 1 = bottom
    **
    **  @param  values  The values to write
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="cabacwrite", description="Write CABAC output registers")
    public void cabacWrite(@Argument(name="cabac",
                                     description="The CABAC number (0 or 1)")
                           int cabac,
                           @Argument(name="values",
                                     description="The 5 values to set")
                           int... values) throws REBException
    {
        cab.writeRaw(cabac, values);
    }


   /**
    ***************************************************************************
    **
    **  Copies the input registers of a CABAC to the output.
    **
    **  @param  cabac  The CABAC number: 0 = top; 1 = bottom
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="cabaccopy", description="Copy CABAC registers")
    public void cabacCopy(@Argument(name="cabac",
                                    description="The CABAC number (0 or 1)")
                          int cabac) throws REBException
    {
        cab.copyRaw(cabac);
    }


   /**
    ***************************************************************************
    **
    **  Shows all fields of a CABAC.
    **
    **  @param  cabac  The CABAC number: 0 = top; 1 = bottom
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="cabac", description="Show all fields of a CABAC")
    public String cabac(@Argument(name="cabac",
                                  description="The CABAC number (0 or 1)")
                        int cabac) throws REBException
    {
        int[] data = cab.get(cabac);
        StringBuilder result = new StringBuilder();
        int count = 0;
        for (String name : cabacMap.keySet()) {
            if (count > 0 && (count % 4) == 0) {
                result.append("\n");
            }
            result.append(String.format("  %-7s = %-5s",
                                        name, data[cabacMap.get(name)]));
            count++;
        }

        return result.toString();
    }


   /**
    ***************************************************************************
    **
    **  Shows a CABAC field.
    **
    **  @param  cabac  The CABAC number: 0 = top; 1 = bottom
    **
    **  @param  field  The field name
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="cabac", description="Show a CABAC field")
    public String cabac(@Argument(name="cabac",
                                  description="The CABAC number (0 or 1)")
                        int cabac,
                        @Argument(name="field",
                                  description="The field name")
                        CabacField field) throws REBException
    {
        return field.name() + " = " + cab.get(cabac, field.getValue());
    }


   /**
    ***************************************************************************
    **
    **  Sets all fields of a CABAC.
    **
    **  @param  cabac   The CABAC number: 0 = top; 1 = bottom
    **
    **  @param  values  The 21 values to set
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="cabacsetall", description="Set all CABAC fields")
    public void cabacSet(@Argument(name="cabac",
                                   description="The CABAC number (0 or 1)")
                         int cabac,
                         @Argument(name="values",
                                   description="The 21 values to set")
                         int... values) throws REBException
    {
        cab.set(cabac, values);
    }


   /**
    ***************************************************************************
    **
    **  Sets a CABAC field.
    **
    **  @param  cabac  The CABAC number: 0 = top; 1 = bottom
    **
    **  @param  field  The field name
    **
    **  @param  value  The value to set
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="cabacset", description="Set a CABAC field")
    public void cabacSet(@Argument(name="cabac",
                                   description="The CABAC number (0 or 1)")
                         int cabac,
                         @Argument(name="field",
                                   description="The field name")
                         CabacField field,
                         @Argument(name="value",
                                   description="The value to set")
                         int value) throws REBException
    {
        cab.set(cabac, field.getValue(), value);
    }


   /**
    ***************************************************************************
    **
    **  Shows the ASPIC version.
    **
    **  @return  The result string
    **
    ***************************************************************************
    */
    @Command(name="aspicversion", description="Show ASPIC firmware version")
    public String aspicVersion()
    {
        return "ASPIC version = " + asp.getVersion();
    }


   /**
    ***************************************************************************
    **
    **  Shows ASPIC gain & RC.
    **
    **  @param  strips  The mask of strips to read
    **
    **  @param  side    The side to read: 0 = top; 1 = bottom
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="aspicgainrc", description="Show ASPIC gain & RC")
    public String aspicGainRc(@Argument(name="strips",
                                        description="The mask of strips")
                              int strips,
                              @Argument(name="side",
                                        description="The side")
                              int side) throws REBException
    {
        int[][] value = asp.readGainRc(strips, side);
        StringBuilder text = new StringBuilder();
        for (int j = 0, k = 0; j < Aspic.NUM_STRIPS; j++, strips >>= 1) {
            if ((strips & 1) == 0) continue;
            text.append(k == 0 ? "Strip" : ";");
            text.append(String.format(" %s: gain = %s, rc = %s",
                                      j, value[k][0], value[k][1]));
            k++;
        }
        if (text.length() == 0) {
            text.append(CUP);
        }

        return text.toString();
    }


   /**
    ***************************************************************************
    **
    **  Sets ASPIC gain & RC.
    **
    **  @param  strips  The mask of strips to write
    **
    **  @param  sides   The mask of sides to write: 1 = top; 2 = bottom
    **
    **  @param  gain    The gain
    **
    **  @param  rc      The RC value
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="aspicgainrc", description="Set ASPIC gain & RC")
    public void aspicGainRc(@Argument(name="strips",
                                      description="The mask of strips")
                            int strips,
                            @Argument(name="sides",
                                      description="The mask of sides")
                            int sides,
                            @Argument(name="gain", description="The gain")
                            int gain,
                            @Argument(name="rc", description="The RC value")
                            int rc) throws REBException
    {
        asp.writeGainRc(strips, sides, gain, rc);
    }


   /**
    ***************************************************************************
    **
    **  Shows ASPIC clamp value.
    **
    **  @param  strips  The mask of strips to read
    **
    **  @param  side    The side to read: 0 = top; 1 = bottom
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="aspicclamp", description="Show ASPIC clamp value")
    public String aspicClamp(@Argument(name="strips",
                                       description="The mask of strips")
                             int strips,
                             @Argument(name="side",
                                       description="The side")
                             int side) throws REBException
    {
        int[] value = asp.readClamp(strips, side);
        StringBuilder text = new StringBuilder();
        for (int j = 0, k = 0; j < Aspic.NUM_STRIPS; j++, strips >>= 1) {
            if ((strips & 1) == 0) continue;
            text.append(k == 0 ? "Strip" : ";");
            text.append(String.format(" %s: 0x%02x", j, value[k]));
            k++;
        }
        if (text.length() == 0) {
            text.append(CUP);
        }

        return text.toString();
    }


   /**
    ***************************************************************************
    **
    **  Sets ASPIC clamp value.
    **
    **  @param  strips  The mask of strips to write
    **
    **  @param  sides   The mask of sides to write: 1 = top; 2 = bottom
    **
    **  @param  clamp   The clamp mask
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="aspicclamp", description="Set ASPIC clamp value")
    public void aspicClamp(@Argument(name="strips",
                                     description="The mask of strips")
                           int strips,
                           @Argument(name="sides",
                                     description="The mask of sides")
                           int sides,
                           @Argument(name="clamp",
                                     description="The clamp value")
                           int clamp) throws REBException
    {
        asp.writeClamp(strips, sides, clamp);
    }


   /**
    ***************************************************************************
    **
    **  Shows ASPIC modes.
    **
    **  @param  strips  The mask of strips to read
    **
    **  @param  side    The side to read: 0 = top; 1 = bottom
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="aspicmodes", description="Show ASPIC modes")
    public String aspicModes(@Argument(name="strips",
                                       description="The mask of strips")
                             int strips,
                             @Argument(name="side",
                                       description="The side")
                             int side) throws REBException
    {
        int[] value = asp.readModes(strips, side);
        StringBuilder text = new StringBuilder();
        for (int j = 0, k = 0; j < Aspic.NUM_STRIPS; j++, strips >>= 1) {
            if ((strips & 1) == 0) continue;
            text.append(k == 0 ? "Strip" : ";");
            text.append(String.format(" %s: 0x%02x", j, value[k]));
            k++;
        }
        if (text.length() == 0) {
            text.append(CUP);
        }

        return text.toString();
    }


   /**
    ***************************************************************************
    **
    **  Sets ASPIC modes.
    **
    **  @param  strips  The mask of strips to write
    **
    **  @param  sides   The mask of sides to write: 1 = top; 2 = bottom
    **
    **  @param  modes   The modes value
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="aspicmodes", description="Set ASPIC modes")
    public void aspicModes(@Argument(name="strips",
                                     description="The mask of strips")
                           int strips,
                           @Argument(name="sides",
                                     description="The mask of sides")
                           int sides,
                           @Argument(name="clamp",
                                     description="The modes value")
                           int modes) throws REBException
    {
        asp.writeModes(strips, sides, modes);
    }


   /**
    ***************************************************************************
    **
    **  Shows DREB serial number.
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="sndreb", description="Show DREB serial number")
    public String snDreb() throws REBException
    {
        return "DREB serial no. = " + bset.getDrebSerial();
    }


   /**
    ***************************************************************************
    **
    **  Shows REB serial number.
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="snreb", description="Show REB serial number")
    public String snReb() throws REBException
    {
        return "REB serial no. = " + bset.getRebSerial();
    }


   /**
    ***************************************************************************
    **
    **  Shows DC/DC synchronization.
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="dcdcsync", description="Show DC/DC synchronization")
    public String dcDcSync() throws REBException
    {
        return "DC/DC synchronization is " + (bset.isDcdcSync() ? "on" : "off");
    }


   /**
    ***************************************************************************
    **
    **  Sets the DC/DC synchronization state.
    **
    **  @param  state  The state to set: ON or OFF
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="dcdcsync", description="Set DC/DC synchronization state")
    public void dcDcSync(@Argument(name = "state",
                                   description="The state to set")
                         OnOff state) throws REBException
    {
        bset.setDcdcSync(state == OnOff.ON);
    }


   /**
    ***************************************************************************
    **
    **  Continuously read a set of registers.
    **
    **  @param  address  The address of the first register to read
    **
    **  @param  count    The number of registers to read
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="contread", description="Continuously read registers")
    public String contRead(@Argument(name="address",
                                     description="The first register to read")
                           int address,
                           @Argument(name="count",
                                     description="The number of registers to read")
                           int count) throws REBException
    {
        int[] values = new int[count];
        int nRead = 0;
        long startTime = System.currentTimeMillis();
        System.out.println("Press any key to terminate...");
        contActive = true;
        consQueue.offer(0);
        try {
            while (contActive) {
                bset.read(address, values);
                nRead += count;
            }
        }
        finally {
            double rate = (1000.0 * nRead)
                            / (System.currentTimeMillis() - startTime);
            return String.format("Read rate = %.4g Hz", rate);
        }
    }


   /**
    ***************************************************************************
    **
    **  Continuously reads the board temperatures.
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="conttemp", description="Continuously read temperatures")
    public String contTemp() throws REBException
    {
        int nRead = 0;
        long startTime = System.currentTimeMillis();
        System.out.println("Press any key to terminate...");
        contActive = true;
        consQueue.offer(0);
        try {
            while (contActive) {
                tmp.readAdcs();
                nRead++;
            }
        }
        finally {
            double rate = (1000.0 * nRead)
                            / (System.currentTimeMillis() - startTime);
            return String.format("Read rate = %.4g Hz", rate);
        }
    }


   /**
    ***************************************************************************
    **
    **  Continuously read the board power values.
    **
    **  @return  The result string
    **
    **  @throws  REBException
    **
    ***************************************************************************
    */
    @Command(name="contpower", description="Continuously read power values")
    public String contPower() throws REBException
    {
        int nRead = 0;
        long startTime = System.currentTimeMillis();
        System.out.println("Press any key to terminate...");
        contActive = true;
        consQueue.offer(0);
        try {
            while (contActive) {
                pwr.readAdcs();
                nRead++;
            }
        }
        finally {
            double rate = (1000.0 * nRead)
                            / (System.currentTimeMillis() - startTime);
            return String.format("Read rate = %.4g Hz", rate);
        }
    }


   /**
    ***************************************************************************
    **
    **  Formats ADC values.
    **
    ***************************************************************************
    */
    private static String formatAdcs(double[] values, int index)
    {
        StringBuilder result = new StringBuilder();
        for (int j = 0; j < values.length; j++) {
            if ((j & 3) == 0) {
                result.append(j == 0 ? "ADCs:" : "\n     ");
            }
            result.append(String.format("  %2s: %6g", j + index, values[j]));
        }

        return result.toString();
    }


   /**
    ***************************************************************************
    **
    **  Waits for an item to arrive on a queue.
    **
    ***************************************************************************
    */
    private static void take(BlockingQueue queue)
    {
        while (true) {
            try {
                queue.take();
                return;
            }
            catch (InterruptedException e) {
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Waits for terminal input.
    **
    ***************************************************************************
    */
    private static void awaitTerminal()
    {
        while (true) {
            try {
                if (System.in.available() > 0) return;
                try {
                    Thread.sleep(50);
                }
                catch (InterruptedException e) {
                }
            }
            catch (IOException e) {
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Clears terminal input.
    **
    ***************************************************************************
    */
    private static void clearTerminal()
    {
        try {
            while (System.in.available() > 0) {
                System.in.read();
            }
        }
        catch (IOException e) {
        }
    }

}
