package org.lsst.ccs.drivers.reb;

/**
 ******************************************************************************
 **
 **  Temperature ADCs reading routines.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class TempAdcs extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    public final static int
        REG_TEMP_ADCS   = 0x600010,
        ADC_DREB_T1     = 0,
        ADC_DREB_T2     = 1,
        ADC_REB_T1      = 2,
        ADC_REB_T2      = 3,
        ADC_REB_T3      = 4,
        ADC_REB_T4      = 5,
        ADC_REB_T5      = 6,
        ADC_REB_T6      = 7,
        ADC_REB_T7      = 8,
        ADC_REB_T8      = 9,
        ADC_REB_T9      = 10,
        NUM_TEMP_REGS_S = 11,
        NUM_TEMP_REGS_C = 6,
        NUM_TEMP_REGS_M = NUM_TEMP_REGS_S,
        ERROR_MASK      = 0x00010000,
        VALUE_MASK      = 0x0000ffff;

    public final static double
        TEMP_SCALE      = 1.0 / 128.0,
        TEMP_ERROR      = -8888.0;
 

   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public TempAdcs()
    {
        super();
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  reg  The associated register client object
    **
    ***************************************************************************
    */
    public TempAdcs(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Enables the temperature ADC reading.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    @Override
    public void enable() throws REBException
    {
        getVersion(OPTN_BOARD_TEMP);
        enable(RSET_TEMP_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Waits for the data to be available.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void waitDone() throws REBException
    {
        getVersion(OPTN_BOARD_TEMP);
        waitDone(RSET_TEMP_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Gets the time of the read enable.
    **
    **  @return  The Unix millisecond time of the read enable
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public long getTriggerTime() throws REBException
    {
        checkNotVersion(OPTN_BOARD_TEMP, VERSION_0);
        return getTriggerTime(RSET_TEMP_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Reads one temperature value.
    **
    **  @param  adc  The number of the ADC to read
    **
    **  @return  The ADC value (Celsius)
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double readAdc(int adc) throws REBException
    {
        if (adc < 0 || adc >= getNumRegs()) {
            throw new REBException("Invalid temperature ADC number");
        }
        enable();
        waitDone();
        int value = read(REG_TEMP_ADCS + adc);

        return (value & ERROR_MASK) != 0
                 ? TEMP_ERROR : TEMP_SCALE * (value & VALUE_MASK);
    }


   /**
    ***************************************************************************
    **
    **  Reads all the temperature values.
    **
    **  @return  The array of ADC values (Celsius)
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double[] readAdcs() throws REBException
    {
        return readAdcs(0, getNumRegs());
    }


   /**
    ***************************************************************************
    **
    **  Reads a range of temperature values.
    **
    **  @param  first  The number of the first ADC to read
    **
    **  @param  count  The number of ADCs to read
    **
    **  @return  The array of ADC values (Celsius)
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public double[] readAdcs(int first, int count) throws REBException
    {
        if (first < 0 || count < 0 || first + count > getNumRegs()) {
            throw new REBException("Invalid temperature ADC range");
        }
        enable();
        waitDone();
        int[] rawData = new int[count];
        double[] data = new double[count];
        read(REG_TEMP_ADCS + first, rawData);
        for (int j = 0; j < count; j++) {
            int value = rawData[j];
            data[j] = (value & ERROR_MASK) != 0
                        ? TEMP_ERROR : TEMP_SCALE * (value & VALUE_MASK);
        }

        return data;
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of ADC registers.
    **
    **  @return  The number of ADC registers
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int getNumRegs() throws REBException
    {
        return getVersion(OPTN_BOARD_TEMP) == VERSION_2 ? NUM_TEMP_REGS_C
                                                        : NUM_TEMP_REGS_S;
    }

}
