package org.lsst.ccs.drivers.reb;

/**
 ******************************************************************************
 **
 **  Java interface to the REB registers.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class RegClient {

   /**
    ***************************************************************************
    **
    **  Inner class to contain a copy of the register access handle.
    **
    ***************************************************************************
    */
    class Handle {

        long value;
        int instance;

    }


   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    Handle handle;


   /**
    ***************************************************************************
    **
    **  Static initializer.
    **
    ***************************************************************************
    */
    static {
        System.loadLibrary("Reb");
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public RegClient()
    {
        handle = new Handle();
    }


   /**
    ***************************************************************************
    **
    **  Finalizer.
    **
    ***************************************************************************
    */
    @Override
    protected void finalize() throws Throwable
    {
        super.finalize();
        deleteRegClient(handle.value);
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection using the default network interface.
    **
    **  @param  id  The ID of the REB to connect to
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void open(int id) throws REBException
    {
        open(id, null);
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection using the specified network interface.
    **
    **  @param  id   The ID of the REB to connect to
    **
    **  @param  ifc  The name of the network interface to use.  If null or
    **               empty, the default interface is used.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void open(int id, String ifc) throws REBException
    {
        if (handle.value != 0) {
            throw new REBException("Register connection already open");
        }
        handle.value = newRegClient(id, ifc);
        handle.instance++;
    }


   /**
    ***************************************************************************
    **
    **  Closes a connection.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void close() throws REBException
    {
        checkOpen();
        deleteRegClient(handle.value);
        handle.value = 0;
    }


   /**
    ***************************************************************************
    **
    **  Checks that connection is open.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    protected void checkOpen() throws REBException
    {
        if (handle.value == 0) {
            throw new REBException("Register connection not open");
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads from a register.
    **
    **  @param  address  The address of the register to read
    **
    **  @return  The value contained in the register
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int read(int address) throws REBException
    {
        checkOpen();
        return readReg(handle.value, address);
    }


   /**
    ***************************************************************************
    **
    **  Reads from a set of registers.
    **
    **  @param  address  The address of the first register to read
    **
    **  @param  values   An array to receive the register values
    **
    **  @param  offset   The offset to the array element where the first value
    **                   is to be put
    **
    **  @param  count    The number of registers to read
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void read(int address, int[] values, int offset, int count)
        throws REBException
    {
        checkOpen();
        readRegs(handle.value, address, values, offset, count);
    }


   /**
    ***************************************************************************
    **
    **  Reads from a set of registers.
    **
    **  @param  address  The address of the first register to read
    **
    **  @param  values   An array to receive the register values.  The size
    **                   of the array determines how many registers to read.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void read(int address, int[] values) throws REBException
    {
        checkOpen();
        readRegs(handle.value, address, values, 0, values.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes to a register.
    **
    **  @param  address  The address of the register to write
    **
    **  @param  value    The value to write to the register
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void write(int address, int value) throws REBException
    {
        checkOpen();
        writeReg(handle.value, address, value);
    }


   /**
    ***************************************************************************
    **
    **  Writes to a set of registers.
    **
    **  @param  address  The address of the first register to write
    **
    **  @param  values   An array containing the values to write
    **
    **  @param  offset   The offset to the array element containing the first
    **                   value to write
    **
    **  @param  count    The number of registers to write
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void write(int address, int[] values, int offset, int count)
        throws REBException
    {
        checkOpen();
        writeRegs(handle.value, address, values, offset, count);
    }


   /**
    ***************************************************************************
    **
    **  Writes to a set of registers.
    **
    **  @param  address  The address of the first register to write
    **
    **  @param  values   An array containing the values to write.  The size of
    **                   array is the number of registers to write.
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void write(int address, int[] values) throws REBException
    {
        checkOpen();
        writeRegs(handle.value, address, values, 0, values.length);
    }


   /**
    ***************************************************************************
    **
    **  Updates a register.
    **
    **  @param  address  The address of the register to update
    **
    **  @param  mask     A mask in which set bits indicate which bits of
    **                   value are to be written
    **
    **  @param  value    The value to write to the register
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int update(int address, int mask, int value) throws REBException
    {
        checkOpen();
        return updateReg(handle.value, address, mask, value);
    }


   /**
    ***************************************************************************
    **
    **  Reads a register pair as a long value.
    **
    **  @param  address  The address of the first register to read
    **
    **  @return  The value obtained by combining the values as a little-endian
    **           pair
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public long readLong(int address) throws REBException
    {
        checkOpen();
        int[] vals = new int[2];
        readRegs(handle.value, address, vals, 0, vals.length);
        return ((long)vals[1] << 32) | (vals[0] & 0xffffffffL);
    }


   /**
    ***************************************************************************
    **
    **  Writes a long value to a register pair.
    **
    **  @param  address  The address of the first register to write
    **
    **  @param  value    The value to write, as a little-endian pair
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void writeLong(int address, long value) throws REBException
    {
        checkOpen();
        int[] vals = {(int)value, (int)(value >> 32)};
        writeRegs(handle.value, address, vals, 0, vals.length);
    }


   /**
    ***************************************************************************
    **
    **  Creates a new register client object.
    **
    **  @param  id   The ID of the REB to be accessed by the object
    **
    **  @param  ifc  The name of the network interface to use.  If null or
    **               empty, the default interface is used.
    **
    **  @return  The handle for the created object
    **
    ***************************************************************************
    */
    private native long newRegClient(int id, String ifc);


   /**
    ***************************************************************************
    **
    **  Deletes the object.
    **
    **  @param  handle  The handle of the register client object
    **
    ***************************************************************************
    */
    private native void deleteRegClient(long handle);


   /**
    ***************************************************************************
    **
    **  Reads from a register.
    **
    **  @param  handle   The handle of the register client object
    **
    **  @param  address  The address of the register to read
    **
    **  @return  The value contained in the register
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private native int readReg(long handle, int address) throws REBException;


   /**
    ***************************************************************************
    **
    **  Reads from a set of registers.
    **
    **  @param  handle   The handle of the register client object
    **
    **  @param  address  The address of the first register to read
    **
    **  @param  values   An array to receive the register values
    **
    **  @param  offset   The offset to the array element where the first value
    **                   is to be put
    **
    **  @param  count    The number of registers to read
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private native void readRegs(long handle, int address, int[] values,
                                 int offset, int count)
        throws REBException;


   /**
    ***************************************************************************
    **
    **  Writes to a register.
    **
    **  @param  handle   The handle of the register client object
    **
    **  @param  address  The address of the register to write
    **
    **  @param  value    The value to write to the register
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private native void writeReg(long handle, int address, int value)
        throws REBException;


   /**
    ***************************************************************************
    **
    **  Writes to a set of registers.
    **
    **  @param  handle   The handle of the register client object
    **
    **  @param  address  The address of the first register to write
    **
    **  @param  values   An array containing the values to write
    **
    **  @param  offset   The offset to the array element containing the first
    **                   value to write
    **
    **  @param  count    The number of registers to write
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private native void writeRegs(long handle, int address, int[] values,
                                  int offset, int count)
        throws REBException;


   /**
    ***************************************************************************
    **
    **  Updates a register.
    **
    **  @param  handle   The handle of the register client object
    **
    **  @param  address  The address of the register to update
    **
    **  @param  mask     A mask in which set bits indicate which bits of
    **                   value are to be written
    **
    **  @param  value    The value to write to the register
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private native int updateReg(long handle, int address, int mask, int value)
        throws REBException;

}
