package org.lsst.ccs.drivers.reb;

import java.util.HashMap;
import java.util.Map;

/**
 ******************************************************************************
 **
 **  Routines for setting the on-board DACs.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class BoardDacs extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    public final static int
        REG_BOARD_DACS   = 0x400000,
        REG_BOARD_DACS_0 = 0x500001,
        OFF_CLOCK_RAILS  = 0,
        OFF_CS_GATE      = 0x10,
        OFF_CABAC_ALT    = 0x100,
        DACS_SET         = 0,
        DACS_LOAD        = 1,
        DAC_CLOCK_RAILS  = 0,
        DAC_CS_GATE      = 1,
        DAC_CABAC_ALT    = 2,
        NUM_BOARD_DACS   = 2,
        NUM_BOARD_DACS_2 = 3,
        CHAN_SCLK_L      = 0,
        CHAN_SCLK_L_SH   = 1,
        CHAN_SCLK_H      = 2,
        CHAN_SCLK_H_SH   = 3,
        CHAN_RG_L        = 4,
        CHAN_RG_L_SH     = 5,
        CHAN_RG_H        = 6,
        CHAN_RG_H_SH     = 7,
        CHAN_PCLK_L      = 8,
        CHAN_PCLK_L_SH   = 9,
        CHAN_PCLK_H      = 10,
        CHAN_PCLK_H_SH   = 11,
        CHAN_HEATER_1    = 12,
        CHAN_HEATER_2    = 13,
        CHAN_CSGATE_1    = 14,
        CHAN_CSGATE_2    = 15,
        CHAN_CSGATE_3    = 16,
        CHAN_FSB_CTRL    = 17,
        CHAN_OD_CTRL     = 18,
        CHAN_GD          = 19,
        CHAN_RD          = 20,
        CHAN_OG          = 21,
        CHAN_OG_SH       = 22;
    
   /**
    ***************************************************************************
    **
    **  Maps.
    **
    ***************************************************************************
    */
    private static final Map<Integer, Integer> dacMap = new HashMap<>();
    static {
        dacMap.put(DAC_CLOCK_RAILS, OFF_CLOCK_RAILS);
        dacMap.put(DAC_CS_GATE,     OFF_CS_GATE);
        dacMap.put(DAC_CABAC_ALT,   OFF_CABAC_ALT);
    }

    private static final Map<Integer, Integer> chanMap_0 = new HashMap<>();
    static {
        chanMap_0.put(CHAN_SCLK_L,   (DAC_CLOCK_RAILS << 16) | 0);
        chanMap_0.put(CHAN_SCLK_H,   (DAC_CLOCK_RAILS << 16) | 1);
        chanMap_0.put(CHAN_RG_L,     (DAC_CLOCK_RAILS << 16) | 2);
        chanMap_0.put(CHAN_RG_H,     (DAC_CLOCK_RAILS << 16) | 3);
        chanMap_0.put(CHAN_PCLK_L,   (DAC_CLOCK_RAILS << 16) | 4);
        chanMap_0.put(CHAN_PCLK_H,   (DAC_CLOCK_RAILS << 16) | 5);
        chanMap_0.put(CHAN_HEATER_1, (DAC_CLOCK_RAILS << 16) | 6);
        chanMap_0.put(CHAN_HEATER_2, (DAC_CLOCK_RAILS << 16) | 7);
        chanMap_0.put(CHAN_CSGATE_1, (DAC_CS_GATE << 16) | 0);
        chanMap_0.put(CHAN_CSGATE_2, (DAC_CS_GATE << 16) | 1);
        chanMap_0.put(CHAN_CSGATE_3, (DAC_CS_GATE << 16) | 2);
    }

    private static final Map<Integer, Integer> chanMap_1 = new HashMap<>();
    static {
        chanMap_1.put(CHAN_SCLK_L,    (DAC_CLOCK_RAILS << 16) | 0x00);
        chanMap_1.put(CHAN_SCLK_L_SH, (DAC_CLOCK_RAILS << 16) | 0x01);
        chanMap_1.put(CHAN_SCLK_H,    (DAC_CLOCK_RAILS << 16) | 0x02);
        chanMap_1.put(CHAN_SCLK_H_SH, (DAC_CLOCK_RAILS << 16) | 0x03);
        chanMap_1.put(CHAN_RG_L,      (DAC_CLOCK_RAILS << 16) | 0x04);
        chanMap_1.put(CHAN_RG_L_SH,   (DAC_CLOCK_RAILS << 16) | 0x05);
        chanMap_1.put(CHAN_RG_H,      (DAC_CLOCK_RAILS << 16) | 0x06);
        chanMap_1.put(CHAN_RG_H_SH,   (DAC_CLOCK_RAILS << 16) | 0x07);
        chanMap_1.put(CHAN_PCLK_L,    (DAC_CLOCK_RAILS << 16) | 0x10);
        chanMap_1.put(CHAN_PCLK_L_SH, (DAC_CLOCK_RAILS << 16) | 0x11);
        chanMap_1.put(CHAN_PCLK_H,    (DAC_CLOCK_RAILS << 16) | 0x12);
        chanMap_1.put(CHAN_PCLK_H_SH, (DAC_CLOCK_RAILS << 16) | 0x13);
        chanMap_1.put(CHAN_CSGATE_1,  (DAC_CS_GATE << 16) | 0);
        chanMap_1.put(CHAN_CSGATE_2,  (DAC_CS_GATE << 16) | 1);
        chanMap_1.put(CHAN_CSGATE_3,  (DAC_CS_GATE << 16) | 2);
        chanMap_1.put(CHAN_FSB_CTRL,  (DAC_CS_GATE << 16) | 3);
        chanMap_1.put(CHAN_HEATER_1,  (DAC_CS_GATE << 16) | 4);
        chanMap_1.put(CHAN_HEATER_2,  (DAC_CS_GATE << 16) | 5);
        chanMap_1.put(CHAN_OD_CTRL,   (DAC_CS_GATE << 16) | 6);
        chanMap_1.put(CHAN_GD,        (DAC_CABAC_ALT << 16) | 0);
        chanMap_1.put(CHAN_RD,        (DAC_CABAC_ALT << 16) | 1);
        chanMap_1.put(CHAN_OG,        (DAC_CABAC_ALT << 16) | 2);
        chanMap_1.put(CHAN_OG_SH,     (DAC_CABAC_ALT << 16) | 3);
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public BoardDacs()
    {
        super();
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  reg  The associated register client object
    **
    ***************************************************************************
    */
    public BoardDacs(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Gets the DAC version.
    **
    **  @return  The version number: VERSION_0 & VERSION_1 for REB0; VERSION_2
    **           for WREB (REB1?)
    **
    ***************************************************************************
    */
    public int getVersion()
    {
        try {
            return getVersion(OPTN_BOARD_DACS);
        }
        catch (REBException e) {
            return VERSION_UNSUPP;
        }
    }


   /**
    ***************************************************************************
    **
    **  Loads all DAC values.
    **
    **  All the values previously set in the DACs are loaded into their output
    **  registers
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void load() throws REBException
    {
        int version = getVersion(OPTN_BOARD_DACS);
        int ndacs = version <= VERSION_1 ? NUM_BOARD_DACS : NUM_BOARD_DACS_2;
        for (int dac = 0; dac < ndacs; dac++) {
            load(dac);
        }
    }


   /**
    ***************************************************************************
    **
    **  Loads one DAC's values.
    **
    **  All the values previously set in a DAC are loaded into its output
    **  registers
    **
    **  @param  dac  The DAC number
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void load(int dac) throws REBException
    {
        int version = getVersion(OPTN_BOARD_DACS);
        if (version == VERSION_0) return;
        int base = getBase(version, dac);
        write(base + DACS_LOAD, 0);
    }


   /**
    ***************************************************************************
    **
    **  Sets a DAC value.
    **
    **  The supplied value is set into the DAC, but not loaded into the DAC's
    **  output register until the load method is called
    **
    **  @param  chan   The overall channel number
    **
    **  @param  value  The value to be set
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void set(int chan, int value) throws REBException
    {
        int version = getVersion(OPTN_BOARD_DACS);
        Integer desc = (version <= VERSION_1 ? chanMap_0 : chanMap_1).get(chan);
        if (desc == null) {
            throw new REBException("Invalid channel number");
        }
        int base = getBase(version, desc >> 16);
        write(base + DACS_SET, ((desc & 0xffff) << 12) | (value & 0x0fff));
    }


   /**
    ***************************************************************************
    **
    **  Gets a DAC's base register address.
    **
    **  @param  version  The board DACS system version
    **
    **  @param  dac      The DAC number
    **
    **  @return  The base address of the DAC
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private int getBase(int version, int dac) throws REBException
    {
        int ndacs = version <= VERSION_1 ? NUM_BOARD_DACS : NUM_BOARD_DACS_2;
        if (dac < 0 || dac >= ndacs) {
            throw new REBException("Invalid DAC number");
        }
        return (version == VERSION_0 ? REG_BOARD_DACS_0 : REG_BOARD_DACS)
                 + dacMap.get(dac);
    }

}
