package org.lsst.ccs.drivers.reb;

/**
 ******************************************************************************
 **
 **  ASPIC access routines.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class Aspic extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    public final static int
        REG_ASPIC       = 0xb00000,
        REG_ASPIC_LOAD  = REG_ASPIC,
        REG_ASPIC_RESET = REG_ASPIC + 1,
        REG_ASPIC_READ  = REG_ASPIC + 0x11,
        NUM_STRIPS      = 3,
        SIDE_TOP        = 0,
        SIDE_BOTTOM     = 1,
        NUM_SIDES       = 2,
        NUM_ASPICS      = NUM_STRIPS * NUM_SIDES,
        RC_MASK         = 0x0f,
        GAIN_POSN       = 4,
        GAIN_INDEX      = 0,
        RC_INDEX        = 1,
        MODE_TM         = 0x01,
        MODE_AF1        = 0x02,
        ADDR_GAIN_RC    = 0,
        ADDR_CLAMP      = 1,
        ADDR_MODES      = 2,
        VALUE_MASK      = 0xff,
        ADDR_MASK       = 0x03,
        ADDR_POSN       = 16,
        WRITE_POSN      = 23,
        SIDE_MASK       = 0x03,
        SIDE_POSN       = 24,
        STRIP_MASK      = 0x07,
        STRIP_POSN      = 26;
 

   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public Aspic()
    {
        super();
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  reg  The associated register client object
    **
    ***************************************************************************
    */
    public Aspic(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Gets the ASPIC version.
    **
    **  @return  The version number: VERSION_UNSUPP for ASPIC0; VERSION_0
    *            for ASPIC1
    **
    ***************************************************************************
    */
    public int getVersion()
    {
        try {
            return getVersion(OPTN_ASPIC, true);
        }
        catch (REBException e) {
            return VERSION_UNSUPP;
        }
    }


   /**
    ***************************************************************************
    **
    **  Writes ASPIC gain and RC.
    **
    **  @param  strips  The mask of strips to write
    **
    **  @param  sides   The mask of sides: 1 = top; 2 = bottom
    **
    **  @param  gain    The gain value
    **
    **  @param  rc      The RC value
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void writeGainRc(int strips, int sides, int gain, int rc)
        throws REBException
    {
        writeParam(strips, sides, ADDR_GAIN_RC,
                   (rc & RC_MASK) | (gain << GAIN_POSN));
    }


   /**
    ***************************************************************************
    **
    **  Writes ASPIC clamp.
    **
    **  @param  strips  The mask of strips to write
    **
    **  @param  sides   The mask of sides: 1 = top; 2 = bottom
    **
    **  @param  clamp   The clamp value
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void writeClamp(int strips, int sides, int clamp) throws REBException
    {
        writeParam(strips, sides, ADDR_CLAMP, clamp);
    }


   /**
    ***************************************************************************
    **
    **  Writes ASPIC special modes.
    **
    **  @param  strips  The mask of strips to write
    **
    **  @param  sides   The mask of sides: 1 = top; 2 = bottom
    **
    **  @param  modes   The modes value
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void writeModes(int strips, int sides, int modes) throws REBException
    {
        writeParam(strips, sides, ADDR_MODES, modes);
    }


   /**
    ***************************************************************************
    **
    **  Reads ASPIC gain and RC.
    **
    **  @param  strips  The mask of strips to read
    **
    **  @param  side    The side number: 0 = top; 1 = bottom
    **
    **  @return  Array of two-element arrays (gain [0] and RC [1]), one per
    **           read strip
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int[][] readGainRc(int strips, int side) throws REBException
    {
        int[] value = readParam(strips, side, ADDR_GAIN_RC);
        int[][] vals = new int[value.length][];
        for (int j = 0; j < value.length; j++) {
            vals[j] = new int[]{value[j] >> GAIN_POSN, value[j] & RC_MASK};
        }

        return vals;
    }


   /**
    ***************************************************************************
    **
    **  Reads ASPIC clamp.
    **
    **  @param  strips  The mask of strips to read
    **
    **  @param  side    The side number: 0 = top; 1 = bottom
    **
    **  @return  Three-element array of values, one per strip
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int[] readClamp(int strips, int side) throws REBException
    {
        return readParam(strips, side, ADDR_CLAMP);
    }


   /**
    ***************************************************************************
    **
    **  Reads ASPIC special modes.
    **
    **  @param  strips  The mask of strips to read
    **
    **  @param  side    The side number: 0 = top; 1 = bottom
    **
    **  @return  Three-element array of values, one per strip
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public int[] readModes(int strips, int side) throws REBException
    {
        return readParam(strips, side, ADDR_MODES);
    }


   /**
    ***************************************************************************
    **
    **  Resets ASPIC registers.
    **
    **  @param  strip  The number of the strip to reset
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    public void reset(int strip) throws REBException
    {
        getVersion(OPTN_ASPIC);
        write(REG_ASPIC_RESET, strip);
    }


   /**
    ***************************************************************************
    **
    **  Writes an ASPIC parameter.
    **
    **  @param  strips  The mask of strips to write
    **
    **  @param  sides   The mask of sides: 1 = top; 2 = bottom
    **
    **  @param  addr    The register address
    **
    **  @param  value   The value to write
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private void writeParam(int strips, int sides, int addr, int value)
        throws REBException
    {
        getVersion(OPTN_ASPIC);
        write(REG_ASPIC_LOAD, makeCommand(strips, sides, addr, 1, value));
    }


   /**
    ***************************************************************************
    **
    **  Reads an ASPIC parameter.
    **
    **  @param  strips  The mask of strips to read
    **
    **  @param  side    The side number: 0 = top; 1 = bottom
    **
    **  @param  addr    The register address
    **
    **  @return  The array of read values, one per strip
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private int[] readParam(int strips, int side, int addr) throws REBException
    {
        getVersion(OPTN_ASPIC);
        write(REG_ASPIC_LOAD, makeCommand(strips, 1 << side, addr, 0, 0));
        int[] value = new int[Integer.bitCount(strips)];
        for (int j = 0, k = 0; j < value.length; j++) {
            if (((strips >> j) & 1) != 0) {
                value[k++] = read(REG_ASPIC_READ + j);
            }
        }

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Makes a command word.
    **
    **  @param  strips  The mask of strips
    **
    **  @param  sides   The mask of sides: 1 = top; 2 = bottom
    **
    **  @param  addr    The ASPIC register address
    **
    **  @param  write   1 to write; 0 to read
    **
    **  @param  value   The value to write
    **
    **  @return  The generated command word
    **
    **  @throws  REBException 
    **
    ***************************************************************************
    */
    private int makeCommand(int strips, int sides, int addr, int write,
                            int value) throws REBException
    {
        if ((strips & STRIP_MASK) != strips) {
            throw new REBException("Invalid strips mask");
        }
        if ((sides & SIDE_MASK) != sides) {
            throw new REBException("Invalid sides mask");
        }
        sides = sides == 2 ? 1 : sides == 1 ? 2 : sides;
        return (value & VALUE_MASK) | (addr << ADDR_POSN)
                 | (sides << SIDE_POSN) | (strips << STRIP_POSN)
                 | (write << WRITE_POSN);
    }

}
