package org.lsst.ccs.drivers.reb;

/**
 ******************************************************************************
 **
 **  Power ADCs reading routines.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class PowerAdcs extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants.
    **
    ***************************************************************************
    */
    public final static int
        REG_POWER_ADCS  = 0x600000,
        ADC_V6_VOLTAGE  = 0,
        ADC_V6_CURRENT  = 1,
        ADC_V9_VOLTAGE  = 2,
        ADC_V9_CURRENT  = 3,
        ADC_V24_VOLTAGE = 4,
        ADC_V24_CURRENT = 5,
        ADC_V40_VOLTAGE = 6,
        ADC_V40_CURRENT = 7,
        NUM_POWER_REGS  = 8,
        ERROR_MASK      = 0x00010000,
        VALUE_MASK      = 0x0000ffff;

    public final static float
        VOLTAGE_SCALE   = 1 / 640F,
        CURRENT_SCALE   = 1 / 640000F,
        POWER_ERROR     = -8888F;


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public PowerAdcs()
    {
        super();
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  reg  The associated register client object
    **
    ***************************************************************************
    */
    public PowerAdcs(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Enables the power ADC reading.
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    @Override
    public void enable() throws REBException
    {
        checkNotVersion(VERSION_0);
        enable(RSET_POWER_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Waits for the data to be available.
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void waitDone() throws REBException
    {
        checkNotVersion(VERSION_0);
        waitDone(RSET_POWER_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Gets the time of the read enable.
    **
    **  @return  The Unix millisecond time of the read enable
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public long getTriggerTime() throws REBException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        return getTriggerTime(RSET_POWER_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Reads one power value.
    **
    **  @param  adc  The number of the ADC to read
    **
    **  @return  The ADC value (volts)
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public float readAdc(int adc) throws REBException
    {
        if (adc < 0 || adc >= NUM_POWER_REGS) {
            throw new REBException("Invalid power ADC number");
        }
        enable();
        waitDone();
        int value = read(REG_POWER_ADCS + adc);

        return (value & ERROR_MASK) != 0 ? POWER_ERROR
                 : ((adc & 1) == 0 ? VOLTAGE_SCALE : CURRENT_SCALE)
                     * (value & VALUE_MASK);
    }


   /**
    ***************************************************************************
    **
    **  Reads all the power values.
    **
    **  @return  The array of ADC values (volts)
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public float[] readAdcs() throws REBException
    {
        return readAdcs(0, NUM_POWER_REGS);
    }


   /**
    ***************************************************************************
    **
    **  Reads a range of power values.
    **
    **  @param  first  The number of the first ADC to read
    **
    **  @param  count  The number of ADCs to read
    **
    **  @return  The array of ADC values (volts)
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public float[] readAdcs(int first, int count) throws REBException
    {
        if (first < 0 || count < 0 || first + count > NUM_POWER_REGS) {
            throw new REBException("Invalid power ADC range");
        }
        enable();
        waitDone();
        int[] rawData = new int[count];
        float[] data = new float[count];
        read(REG_POWER_ADCS + first, rawData);
        for (int j = 0, adc = first; j < count; j++, adc++) {
            int value = rawData[j];
            data[j] = (value & ERROR_MASK) != 0 ? POWER_ERROR
                        : ((adc & 1) == 0 ? VOLTAGE_SCALE : CURRENT_SCALE)
                            * (value & VALUE_MASK);
        }

        return data;
    }

}
