package org.lsst.ccs.drivers.reb;

/**
 ******************************************************************************
 **
 **  CABAC access routines.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class Cabac extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants.
    **
    ***************************************************************************
    */
    public final static int
        REG_CABAC         = 0x500000,
        CABAC_OFF_LOAD    = 0,
        CABAC_OFF_FETCH   = 1,
        CABAC_OFF_RESET   = 2,
        CABAC_OFF_TOP     = 0x10,
        CABAC_OFF_BOTTOM  = 0x20,
        CABAC_OFF_WRITE   = 0,
        CABAC_OFF_READ    = 0x100,
        NUM_CABAC_REGS    = 5,
        CABAC_ID_TOP      = 0,
        CABAC_ID_BOTTOM   = 1,
        NUM_CABAC_STRIPES = 3,
        NUM_CABACS        = 6;

    public final static int
        CABAC_FLD_OD_EXP0 = 0,
        CABAC_FLD_OD_EXP1 = 1,
        CABAC_FLD_OD_RDO0 = 2,
        CABAC_FLD_OD_RDO1 = 3,
        CABAC_FLD_PCLK0   = 4,
        CABAC_FLD_PCLK1   = 5,
        CABAC_FLD_PCLK2   = 6,
        CABAC_FLD_PCLK3   = 7,
        CABAC_FLD_SCLK0   = 8,
        CABAC_FLD_SCLK1   = 9,
        CABAC_FLD_SCLK2   = 10,
        CABAC_FLD_RG      = 11,
        CABAC_FLD_GD      = 12,
        CABAC_FLD_OG      = 13,
        CABAC_FLD_RD      = 14,
        CABAC_FLD_SPH     = 15,
        CABAC_FLD_SPL     = 16,
        CABAC_FLD_MUX0    = 17,
        CABAC_FLD_MUX1    = 18,
        CABAC_FLD_PULSE   = 19,
        CABAC_FLD_MUXE    = 20,
        NUM_CABAC_FLDS    = 21;
        
    public final static int[]
        FIELD_LENGTHS  = {  8,   8,   8,   8,   8,   8,   8,   8,
                            8,   8,   8,   8,   8,   8,   8,   8,  
                            8,   4,   4,   1,   1},
        FIELD_STARTS   = {  0,   8,  16,  24,  32,  40,  48,  56,
                           64,  72,  80,  88,  96, 104, 112, 120,
                          128, 136, 140, 144, 145};


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public Cabac()
    {
        super();
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  reg  The associated register client object
    **
    ***************************************************************************
    */
    public Cabac(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Fetches the contents of a CABAC stripe to the registers.
    **
    **  This transfers the contents of both (top and bottom) CABACs in the
    **  stripe to the CABAC input registers in the FPGA.
    **
    **  @param  stripe  The number of the stripe to be fetched
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void fetch(int stripe) throws REBException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        checkStripe(stripe);
        write(REG_CABAC + CABAC_OFF_FETCH, stripe);
    }


   /**
    ***************************************************************************
    **
    **  Loads a CABAC stripe from the registers.
    **
    **  This transfers the contents of the CABAC output registers in the FPGA
    **  to both (top and bottom) CABACs in the stripe.
    **
    **  @param  stripe  The number of the stripe to be loaded
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void load(int stripe) throws REBException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        checkStripe(stripe);
        write(REG_CABAC + CABAC_OFF_LOAD, stripe);
    }


   /**
    ***************************************************************************
    **
    **  Resets a CABAC stripe.
    **
    **  The contents of both (top and bottom) CABACs in the stripe are reset
    **  to all zeroes.
    **
    **  @param  stripe  The number of the stripe to be reset
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void reset(int stripe) throws REBException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        checkStripe(stripe);
        write(REG_CABAC + CABAC_OFF_RESET, stripe);
    }


   /**
    ***************************************************************************
    **
    **  Reads CABAC input registers in raw format.
    **
    **  The contents of the five input registers for the top or bottom CABAC
    **  are returned.
    **
    **  @param  cabac  The CABAC ID: 0 = top; 1 = bottom
    **
    **  @return  The 5-element array of values read from the registers
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public int[] readRaw(int cabac) throws REBException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        int base = getBase(cabac) + CABAC_OFF_READ;
        int[] values = new int[NUM_CABAC_REGS];
        read(base, values);

        return values;
    }


   /**
    ***************************************************************************
    **
    **  Writes CABAC output registers in raw format.
    **
    **  The five output registers for the top or bottom CABAC are set to the
    **  supplied values.
    **
    **  @param  cabac   The CABAC ID: 0 = top; 1 = bottom
    **
    **  @param  values  The 5-element array of values to be written
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void writeRaw(int cabac, int[] values) throws REBException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        int base = getBase(cabac) + CABAC_OFF_WRITE;
        write(base, values);
    }


   /**
    ***************************************************************************
    **
    **  Copies CABAC input registers to the output ones.
    **
    **  The input registers for the top or bottom CABAC are copied to its
    **  output registers.
    **
    **  @param  cabac  The CABAC ID: 0 = top; 1 = bottom
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void copyRaw(int cabac) throws REBException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        int rbase = getBase(cabac) + CABAC_OFF_READ;
        int wbase = getBase(cabac) + CABAC_OFF_WRITE;
        int[] values = new int[NUM_CABAC_REGS];
        read(rbase, values);
        write(wbase, values);
    }


   /**
    ***************************************************************************
    **
    **  Gets one CABAC field value.
    **
    **  The value of the specified field is extracted from the top or bottom
    **  CABAC input registers.
    **
    **  @param  cabac  The CABAC ID: 0 = top; 1 = bottom
    **
    **  @param  field  The field ID
    **
    **  @return  The value of the field
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public int get(int cabac, int field) throws REBException
    {
        return get(cabac, field, 1)[0];
    }


   /**
    ***************************************************************************
    **
    **  Gets all CABAC field values.
    **
    **  The values of all fields are extracted from the top or bottom CABAC
    **  input registers.
    **
    **  @param  cabac  The CABAC ID: 0 = top; 1 = bottom
    **
    **  @return  The 21-element array of field values
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public int[] get(int cabac) throws REBException
    {
        return get(cabac, 0, NUM_CABAC_FLDS);
    }


   /**
    ***************************************************************************
    **
    **  Gets several CABAC field values.
    **
    **  The values of the specified fields are extracted from the top or
    **  bottom CABAC input registers.
    **
    **  @param  cabac  The CABAC ID: 0 = top; 1 = bottom
    **
    **  @param  first  The ID of the first field
    **
    **  @param  count  The number of fields to get
    **
    **  @return  The array of field values
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public int[] get(int cabac, int first, int count)
        throws REBException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        int base = getBase(cabac) + CABAC_OFF_READ;
        checkField(first, count);
        int[] regs = new int[NUM_CABAC_REGS], value = new int[count];
        read(base, regs);
        for (int j = 0; j < count; j++) {
            int beg = FIELD_STARTS[first + j];
            int end = beg + FIELD_LENGTHS[first + j];
            int mask = (1 << (end - beg)) - 1;
            int regBeg = beg >> 5;
            int regEnd = (end - 1) >> 5, bitEnd = (32 - end) & 0x1f;
            if (regEnd == regBeg) {
                value[j] = (regs[regBeg] >> bitEnd) & mask;
            }
            else {
                value[j] = ((regs[regBeg] << (32 - bitEnd))
                              | (regs[regEnd] >>> bitEnd)) & mask;
            }
        }

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Sets one CABAC field value.
    **
    **  A single field is inserted into the correct location in the top or
    **  bottom CABAC's output registers.  Since the output registers are
    **  write-only, this can be done in a reasonable way only if the output
    **  registers were initially primed with the contents of the input
    **  registers, which were in turn obtained by fetching the CABAC's
    **  contents.
    **
    **  @param  cabac  The CABAC ID: 0 = top; 1 = bottom
    **
    **  @param  field  The field ID
    **
    **  @param  value  The value to be set
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void set(int cabac, int field, int value) throws REBException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        int base = getBase(cabac);
        checkField(field);
        int beg = FIELD_STARTS[field];
        int end = beg + FIELD_LENGTHS[field];
        int mask = (1 << (end - beg)) - 1;
        int regBeg = beg >> 5;
        int regEnd = (end - 1) >> 5, bitEnd = (32 - end) & 0x1f;
        if (regEnd == regBeg) {
            updateReg(base, regBeg, mask << bitEnd, value << bitEnd);
        }
        else {
            updateReg(base, regBeg, mask >> (32 - bitEnd),
                      value >> (32 - bitEnd));
            updateReg(base, regEnd, mask << bitEnd, value << bitEnd);
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets all CABAC field values.
    **
    **  The values for all fields of the top or bottom CABAC are set into its
    **  output registers, completely overwriting their contents.
    **
    **  @param  cabac  The CABAC ID: 0 = top; 1 = bottom
    **
    **  @param  value  The 21-element array of values to be set
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void set(int cabac, int[] value) throws REBException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        int base = getBase(cabac) + CABAC_OFF_WRITE;
        int[] regs = new int[NUM_CABAC_REGS];
        for (int j = 0; j < regs.length; j++) {
            regs[j] = 0;
        }
        for (int j = 0; j < NUM_CABAC_FLDS; j++) {
            int beg = FIELD_STARTS[j];
            int end = beg + FIELD_LENGTHS[j];
            int val = value[j] & ((1 << (end - beg)) - 1);
            int regBeg = beg >> 5;
            int regEnd = (end - 1) >> 5, bitEnd = (32 - end) & 0x1f;
            if (regEnd == regBeg) {
                regs[regBeg] |= val << bitEnd;
            }
            else {
                regs[regBeg] |= val >> (32 - bitEnd);
                regs[regEnd] |= val << bitEnd;
            }
        }
        write(base, regs);
    }


   /**
    ***************************************************************************
    **
    **  Sets several CABAC field values.
    **
    **  The specified fields are inserted into the correct locations in the
    **  top or bottom CABAC's output registers.  Since the output registers
    **  are write-only, this can be done in a reasonable way only if the
    **  output registers were initially primed with the contents of the input
    **  registers, which were in turn obtained by fetching the CABAC's
    **  contents.
    **
    **  @param  cabac  The CABAC ID: 0 = top; 1 = bottom
    **
    **  @param  first  The ID of the first field
    **
    **  @param  count  The number of fields to set
    **
    **  @param  value  The array of values to be set
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void set(int cabac, int first, int count, int[] value)
        throws REBException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        int base = getBase(cabac);
        checkField(first, count);
        int[] regs = new int[NUM_CABAC_REGS], masks = new int[NUM_CABAC_REGS];
        for (int j = 0; j < regs.length; j++) {
            regs[j] = 0;
            masks[j] = 0;
        }
        for (int j = 0; j < count; j++) {
            int beg = FIELD_STARTS[first + j];
            int end = beg + FIELD_LENGTHS[first + j];
            int mask = (1 << (end - beg)) - 1;
            int val = value[j] & mask;
            int regBeg = beg >> 5;
            int regEnd = (end - 1) >> 5, bitEnd = (32 - end) & 0x1f;
            if (regEnd == regBeg) {
                regs[regBeg] |= val << bitEnd;
                masks[regBeg] |= mask << bitEnd;
            }
            else {
                regs[regBeg] |= val >> (32 - bitEnd);
                regs[regEnd] |= val << bitEnd;
                masks[regBeg] |= mask >> (32 - bitEnd);
                masks[regEnd] |= mask << bitEnd;
            }
        }
        for (int j = 0; j < regs.length; j++) {
            if (masks[j] == 0) continue;
            if (masks[j] == -1) {
                write(base + CABAC_OFF_WRITE + j, regs[j]);
            }
            else {
                updateReg(base, j, masks[j], regs[j]);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a CABAC stripe number for validity.
    **
    **  @param  stripe  The stripe number
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    private void checkStripe(int stripe) throws REBException
    {
        if (stripe < 0 || stripe >= NUM_CABAC_STRIPES) {
            throw new REBException("Invalid CABAC stripe number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the base register address for the CABAC.
    **
    **  @param  cabac  The number of the CABAC
    **
    **  @return  The CABAC base address
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    private int getBase(int cabac) throws REBException
    {
        if (cabac != CABAC_ID_BOTTOM && cabac != CABAC_ID_TOP) {
            throw new REBException("Invalid CABAC ID");
        }

        return REG_CABAC + (cabac == CABAC_ID_BOTTOM ? CABAC_OFF_BOTTOM
                                                     : CABAC_OFF_TOP);
    }


   /**
    ***************************************************************************
    **
    **  Checks a CABAC field ID for validity.
    **
    **  @param  field  The field ID
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    private void checkField(int field) throws REBException
    {
        if (field < 0 || field >= NUM_CABAC_FLDS) {
            throw new REBException("Invalid CABAC field number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a range of CABAC field IDs for validity.
    **
    **  @param  first  The ID of the first field
    **
    **  @param  count  The number of fields
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    private void checkField(int first, int count) throws REBException
    {
        if (first < 0 || count < 0 || first + count > NUM_CABAC_FLDS) {
            throw new REBException("Invalid CABAC field number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Updates a CABAC set register using its corresponding get register.
    **
    **  @param  base   The CABAC base address
    **
    **  @param  reg    The register number
    **
    **  @param  mask   A mask specifying which bits of value are to be used
    **
    **  @param  value  The value to use
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    private void updateReg(int base, int reg, int mask, int value)
        throws REBException
    {
        write(base + CABAC_OFF_WRITE + reg,
              (read(base + CABAC_OFF_READ + reg) & ~mask) | (value & mask));
    }

}
