package org.lsst.ccs.drivers.reb;

/**
 ******************************************************************************
 **
 **  ASPIC access routines.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class Aspic extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    public final static int
        REG_ASPIC       = 0xb00000,
        REG_ASPIC_LOAD  = REG_ASPIC,
        REG_ASPIC_RESET = REG_ASPIC + 1,
        REG_ASPIC_READ  = REG_ASPIC + 0x11,
        NUM_STRIPS      = 3,
        SIDE_BOTTOM     = 0,
        SIDE_TOP        = 1,
        NUM_SIDES       = 2,
        RC_MASK         = 0x0f,
        GAIN_POSN       = 4,
        GAIN_INDEX      = 0,
        RC_INDEX        = 1,
        MODE_TM         = 0x01,
        MODE_AF1        = 0x02,
        ADDR_GAIN_RC    = 0,
        ADDR_CLAMP      = 1,
        ADDR_MODES      = 2,
        VALUE_MASK      = 0xff,
        ADDR_MASK       = 0x03,
        ADDR_POSN       = 16,
        SIDE_MASK       = 0x03,
        SIDE_POSN       = 24,
        STRIP_MASK      = 0x07,
        STRIP_POSN      = 26,
        WRITE_FUNC      = 0x00800000;
 

   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public Aspic()
    {
        super();
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  reg  The associated register client object
    **
    ***************************************************************************
    */
    public Aspic(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Writes ASPIC gain and RC.
    **
    **  @param  strips  The mask of strips to write
    **
    **  @param  sides   The mask of sides (bottom and top) to write
    **
    **  @param  gain    The gain value
    **
    **  @param  rc      The RC value
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void writeGainRc(int strips, int sides, int gain, int rc)
        throws REBException
    {
        writeParam(strips, sides, ADDR_GAIN_RC,
                   (rc & RC_MASK) | (gain << GAIN_POSN));
    }


   /**
    ***************************************************************************
    **
    **  Writes ASPIC clamp.
    **
    **  @param  strips  The mask of strips to write
    **
    **  @param  sides   The mask of sides (bottom and top) to write
    **
    **  @param  clamp   The clamp value
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void writeClamp(int strips, int sides, int clamp) throws REBException
    {
        writeParam(strips, sides, ADDR_CLAMP, clamp);
    }


   /**
    ***************************************************************************
    **
    **  Writes ASPIC special modes.
    **
    **  @param  strips  The mask of strips to write
    **
    **  @param  sides   The mask of sides (bottom and top) to write
    **
    **  @param  modes   The modes value
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void writeModes(int strips, int sides, int modes) throws REBException
    {
        writeParam(strips, sides, ADDR_MODES, modes);
    }


   /**
    ***************************************************************************
    **
    **  Reads ASPIC gain and RC.
    **
    **  @param  strips  The mask of strips to read
    **
    **  @param  side    The side (bottom or top) to read
    **
    **  @return  Three-element array of two-element arrays (gain [0] and
    **           RC [1]), one per strip
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public int[][] readGainRc(int strips, int side) throws REBException
    {
        int[] value = readParam(strips, side, ADDR_GAIN_RC);
        return new int[][]{{value[0] >> GAIN_POSN, value[0] & RC_MASK},
                           {value[1] >> GAIN_POSN, value[1] & RC_MASK},
                           {value[2] >> GAIN_POSN, value[2] & RC_MASK}};
    }


   /**
    ***************************************************************************
    **
    **  Reads ASPIC clamp.
    **
    **  @param  strips  The mask of strips to read
    **
    **  @param  side    The side (bottom or top) to read
    **
    **  @return  Three-element array of values, one per strip
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public int[] readClamp(int strips, int side) throws REBException
    {
        return readParam(strips, side, ADDR_CLAMP);
    }


   /**
    ***************************************************************************
    **
    **  Reads ASPIC special modes.
    **
    **  @param  strips  The mask of strips to read
    **
    **  @param  side    The side (bottom or top) to read
    **
    **  @return  Three-element array of values, one per strip
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public int[] readModes(int strips, int side) throws REBException
    {
        return readParam(strips, side, ADDR_MODES);
    }


   /**
    ***************************************************************************
    **
    **  Resets ASPIC registers.
    **
    **  @param  strip  The number of the strip to reset
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void reset(int strip) throws REBException
    {
        checkVersion(VERSION_4, VERSION_CURR);
        write(REG_ASPIC_RESET, strip);
    }


   /**
    ***************************************************************************
    **
    **  Writes an ASPIC parameter.
    **
    **  @param  strips  The mask of strips to write
    **
    **  @param  sides   The mask of sides (bottom and top) to write
    **
    **  @param  addr    The register address
    **
    **  @param  value   The value to write
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void writeParam(int strips, int sides, int addr, int value)
        throws REBException
    {
        checkVersion(VERSION_4, VERSION_CURR);
        write(REG_ASPIC_LOAD,
              (value & VALUE_MASK) | ((sides & SIDE_MASK) << SIDE_POSN)
                | ((strips & STRIP_MASK) << STRIP_POSN) | WRITE_FUNC);
    }


   /**
    ***************************************************************************
    **
    **  Reads an ASPIC parameter.
    **
    **  @param  strips  The mask of strips to read
    **
    **  @param  side    The side (bottom or top) to read
    **
    **  @param  addr    The register address
    **
    **  @return  The array of read values, one per strip
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public int[] readParam(int strips, int side, int addr)
        throws REBException
    {
        checkVersion(VERSION_4, VERSION_CURR);
        write(REG_ASPIC_LOAD, (((1 << side) & SIDE_MASK) << SIDE_POSN)
                                | ((strips & STRIP_MASK) << STRIP_POSN));
        int[] value = new int[NUM_STRIPS];
        for (int j = 0; j < value.length; j++, strips >>= 1) {
            value[j] = ((strips & 1) == 0) ? 0 : read(REG_ASPIC_READ + j);
        }

        return value;
    }

}
