package org.lsst.ccs.drivers.reb;

import java.io.IOException;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;

/**
 ******************************************************************************
 **
 **  Program to test the Java REB register access routines
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class TestReg {

    /*
    **  Private fields
    */
    private final BaseSet bset = new BaseSet();
    private final SequencerUtils seq = new SequencerUtils(bset);
    private final StatusSet sset = new StatusSet(bset);
    private final TempAdcs tmp = new TempAdcs(bset);
    private final PowerAdcs pwr = new PowerAdcs(bset);
    private final BoardDacs dac = new BoardDacs(bset);
    private int rebId = -1;
    private boolean contActive;
    private BlockingQueue<Integer> consQueue = new ArrayBlockingQueue<>(1);


   /**
    ***************************************************************************
    **
    **  Enumeration for register set identifiers.
    **
    ***************************************************************************
    */
    public enum RsetId {

        STATUS(BaseSet.RSET_STATUS),
        TIMEBASE(BaseSet.RSET_TIME_BASE),
        SEQUENCER(BaseSet.RSET_SEQUENCER),
        POWERADCS(BaseSet.RSET_POWER_ADCS),
        TEMPADCS(BaseSet.RSET_TEMP_ADCS);

        int value;

        RsetId(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }


   /**
    ***************************************************************************
    **
    **  Inner class for detecting console input while reading continuously.
    **
    ***************************************************************************
    */
    class ConsThread extends Thread {

        @Override
        public void run()
        {
            while (true) {
                take(consQueue);
                while (contActive) {
                    awaitTerminal();
                    if (contActive) {
                        clearTerminal();
                    }
                    contActive = false;
                }
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public TestReg()
    {
        ConsThread cons = new ConsThread();
        cons.setDaemon(true);
        cons.start();
    }


   /**
    ***************************************************************************
    **
    **  Connects to a REB.
    **
    ***************************************************************************
    */
    @Command(name="connect", description="Connect to a REB")
    public void connect(@Argument(name="id", description="REB ID")
                        int id) throws REBException
    {
        rebId = id;
        bset.open(rebId);
    }


   /**
    ***************************************************************************
    **
    **  Connects to a REB.
    **
    ***************************************************************************
    */
    @Command(name="connect", description="Connect to a REB")
    public void connect(@Argument(name="id", description="REB ID")
                        int id,
                        @Argument(name="ifcname",
                                  description="Network interface name")
                        String ifcName) throws REBException
    {
        rebId = id;
        bset.open(rebId, ifcName);
    }


   /**
    ***************************************************************************
    **
    **  Disconnects from a REB.
    **
    ***************************************************************************
    */
    @Command(name="disconnect", description="Disconnect from a REB")
    public void disconnect() throws REBException
    {
        bset.close();
        rebId = -1;
    }


   /**
    ***************************************************************************
    **
    **  Shows connection parameters.
    **
    ***************************************************************************
    */
    @Command(name="show", description="Show connection parameters")
    public String show()
    {
        boolean connected = false;
        int version = -1;
        try {
             bset.checkOpen();
             connected = true;
        }
        catch (REBException e) {
        }
        try {
            version = bset.getVersion();
        }
        catch (REBException e) {
        }
        return   "Connection parameters:" +
               "\n  Connected  = " + connected +
               "\n  REB id     = " + rebId +
               "\n  Version    = " + version;
    }


   /**
    ***************************************************************************
    **
    **  Shows register contents.
    **
    ***************************************************************************
    */
    @Command(name="read", description="Show one register")
    public String read(@Argument(name="address",
                                 description="Register address to read")
                       int address) throws REBException
    {
        return String.format("%06x: %08x", address, bset.read(address));
    }


   /**
    ***************************************************************************
    **
    **  Shows register contents.
    **
    ***************************************************************************
    */
    @Command(name="read", description="Show one or more registers")
    public String read(@Argument(name="address",
                                 description="First register address to read")
                       int address,
                       @Argument(name="count",
                                 description="Number of registers to read")
                       int count) throws REBException
    {
        int[] values = new int[count];
        bset.read(address, values);
        StringBuilder result = new StringBuilder(9 * (count + (count + 3) / 4));
        for (int j = 0; j < count; j++, address++) {
            if ((j & 3) == 0) {
                if (j != 0) result.append("\n");
                result.append(String.format("%06x:", address));
            }
            result.append(String.format(" %08x", values[j]));
        }

        return result.toString();
    }


   /**
    ***************************************************************************
    **
    **  Writes to a register.
    **
    ***************************************************************************
    */
    @Command(name="write", description="Write a register")
    public void write(@Argument(name="address",
                                description="Register address to write")
                      int address,
                      @Argument(name="value", description="Value to write")
                      int value) throws REBException
    {
        bset.write(address, value);
    }


   /**
    ***************************************************************************
    **
    **  Updates a register.
    **
    ***************************************************************************
    */
    @Command(name="update", description="Update a register")
    public void update(@Argument(name="address",
                                 description="Address of register to update")
                       int address,
                       @Argument(name="mask",
                                 description="Mask of bits to update")
                       int mask,
                       @Argument(name="value", description="New value to write")
                       int value) throws REBException
    {
        bset.update(address, mask, value);
    }


   /**
    ***************************************************************************
    **
    **  Shows a pair of registers.
    **
    ***************************************************************************
    */
    @Command(name="readlong", description="Show a pair of registers")
    public String readLong(@Argument(name="address",
                                     description="Address of first register to read")
                           int address) throws REBException
    {
        return String.format("%06x: %016x\n", address, bset.readLong(address));
    }


   /**
    ***************************************************************************
    **
    **  Writes a pair of registers.
    **
    ***************************************************************************
    */
    @Command(name="writelong", description="Write a pair of registers")
    public void writeLong(@Argument(name="address",
                                    description="Address of first register to write")
                          int address,
                          @Argument(name="value", description="Value to write")
                          long value) throws REBException
    {
        bset.writeLong(address, value);
    }


   /**
    ***************************************************************************
    **
    **  Enables a register set.
    **
    ***************************************************************************
    */
    @Command(name="enable", description="Enable a register set")
    public void enable(@Argument(name="regset",
                                 description="Register set name")
                       RsetId regset) throws REBException
    {
        bset.enable(regset.getValue());
    }


   /**
    ***************************************************************************
    **
    **  Disables a register set.
    **
    ***************************************************************************
    */
    @Command(name="disable", description="Disable a register set")
    public void disable(@Argument(name="regset",
                                 description="Register set name")
                        RsetId regset) throws REBException
    {
        bset.disable(regset.getValue());
    }


   /**
    ***************************************************************************
    **
    **  Shows a trigger time.
    **
    ***************************************************************************
    */
    @Command(name="trigtime", description="Show a trigger time")
    public String trigtime(@Argument(name="regset",
                                     description="Register set name")
                           RsetId regset) throws REBException
    {
        long time = bset.getTriggerTime(regset.getValue());
        GregorianCalendar tm = new GregorianCalendar();
        tm.setTimeInMillis(time);

        return String.format("Time = %tY-%<tm-%<td %<tH:%<tM:%<tS.%<tL", tm);
    }


   /**
    ***************************************************************************
    **
    **  Shows the time.
    **
    ***************************************************************************
    */
    @Command(name="time", description="Show the time")
    public String time() throws REBException
    {
        long time = bset.getTime();
        GregorianCalendar tm = new GregorianCalendar();
        tm.setTimeInMillis(time);

        return String.format("Time = %tY-%<tm-%<td %<tH:%<tM:%<tS.%<tL", tm);
    }


   /**
    ***************************************************************************
    **
    **  Sets the time.
    **
    ***************************************************************************
    */
    @Command(name="time", description="Set the time")
    public String setTime(@Argument(name="time", description="The time to set")
                          String time) throws REBException
    {
        String result = "\033[A";    // Move cursor up
        if (time.equals(".")) {
            bset.setTime();
        }
        else {
            SimpleDateFormat fmt = new SimpleDateFormat();
            fmt.setLenient(true);
            fmt.applyPattern("y-M-d H:m:s");
            ParsePosition pos = new ParsePosition(0);
            Date date = fmt.parse(time, pos);
            if (date == null) {
                result = "Expected format is yyyy-mm-dd hh:mm:ss";
            }
            else {
                bset.setTime(date.getTime());
            }
        }

        return result;
    }


   /**
    ***************************************************************************
    **
    **  Shows the time base.
    **
    ***************************************************************************
    */
    @Command(name="timebase", description="Show the time base")
    public String timebase() throws REBException
    {
        return "Value = " + bset.getTimeRaw();
    }


   /**
    ***************************************************************************
    **
    **  Sets the time base.
    **
    ***************************************************************************
    */
    @Command(name="timebase", description="Set the time base")
    public void timebase(@Argument(name="time", description="The value to set")
                         long time) throws REBException
    {
        bset.setTimeRaw(time);
    }


   /**
    ***************************************************************************
    **
    **  Loads a test program into the sequencer.
    **
    ***************************************************************************
    */
    @Command(name="loadtest", description="Load sequencer with test program")
    public void loadTest(@Argument(name="totTime",
                                   description="The total time per cycle")
                         int totTime,
                         @Argument(name="ncycle",
                                   description="The number of cycles")
                         int ncycle) throws REBException
    {
        loadTest(totTime, ncycle, 10);
    }


   /**
    ***************************************************************************
    **
    **  Loads a test program into the sequencer.
    **
    ***************************************************************************
    */
    @Command(name="loadtest", description="Load sequencer with test program")
    public void loadTest(@Argument(name="totTime",
                                   description="The total time per cycle")
                         int totTime,
                         @Argument(name="ncycle",
                                   description="The number of cycles")
                         int ncycle,
                         @Argument(name="ontime",
                                   description="The time the ADC is on per cycle")
                         int onTime) throws REBException
    {
        seq.writeLines(0, new int[]{0});          // Default state
        seq.writeLines(1, new int[]{0x1000, 0});  // ADC trigger on, then off
        seq.writeTimes(1, new int[]{onTime, totTime - onTime, 0});
        if (seq.getVersion() == Sequencer.VERSION_0) {
            seq.writeStack(new int[]{0x10000000 | (ncycle & 0x03ffffff)});
            seq.writeDataSource(Sequencer.SEQ_SRC_FPGA_PTN);
        }
        else {
            seq.writeProgExec(0, 1, ncycle);
            seq.writeProgEnd(1);
            seq.writeDataSource(Sequencer.SEQ_SRC_PATTERN);
        }
        seq.writeSliceCount(ncycle);
    }


   /**
    ***************************************************************************
    **
    **  Loads a command into the sequencer.
    **
    ***************************************************************************
    */
    @Command(name="loadcommand", description="Load command into sequencer")
    public void loadCommand(@Argument(name="command",
                                      description="The command to load")
                            String command) throws REBException
    {
        seq.loadCommand(command);
    }


   /**
    ***************************************************************************
    **
    **  Loads a file of commands into the sequencer.
    **
    ***************************************************************************
    */
    @Command(name="loadfile", description="Load command file into sequencer")
    public String loadFile(@Argument(name="filename",
                                     description="The name of the file to load")
                           String fileName) throws REBException, IOException
    {
        int nSlice = seq.loadFile(fileName);
        return "Slice count = " + nSlice;
    }


   /**
    ***************************************************************************
    **
    **  Shows the slice count register.
    **
    ***************************************************************************
    */
    @Command(name="slice", description="Show the slice count")
    public String sliceCount() throws REBException
    {
        return "Slice count = " + seq.readSliceCount();
    }


   /**
    ***************************************************************************
    **
    **  Sets the slice count register.
    **
    ***************************************************************************
    */
    @Command(name="slice", description="Set the slice count")
    public void sliceCount(@Argument(name="count",
                                     description="The value to set")
                           int count) throws REBException
    {
        seq.writeSliceCount(count);
    }


   /**
    ***************************************************************************
    **
    **  Shows the data source register.
    **
    ***************************************************************************
    */
    @Command(name="source", description="Show the data source")
    public String dataSource() throws REBException
    {
        return "Data source = " + seq.readDataSource();
    }


   /**
    ***************************************************************************
    **
    **  Sets the data source register.
    **
    ***************************************************************************
    */
    @Command(name="source", description="Set the data source")
    public void dataSource(@Argument(name="value",
                                     description="The value to set")
                           int value) throws REBException
    {
        seq.writeDataSource(value);
    }


   /**
    ***************************************************************************
    **
    **  Shows the stripe selection register.
    **
    ***************************************************************************
    */
    @Command(name="stripe", description="Show the stripe selection")
    public String stripeSelect() throws REBException
    {
        return "Stripe selection = " + seq.readStripeSelect();
    }


   /**
    ***************************************************************************
    **
    **  Sets the stripe selection register.
    **
    ***************************************************************************
    */
    @Command(name="stripe", description="Set the stripe selection")
    public void stripeSelect(@Argument(name="value",
                                       description="The value to set")
                             int value) throws REBException
    {
        seq.writeStripeSelect(value);
    }


   /**
    ***************************************************************************
    **
    **  Shows the contents of the status block.
    **
    ***************************************************************************
    */
    @Command(name="status", description="Show status block contents")
    public String showStatus() throws REBException
    {
        Status status = sset.readStatus();
        StringBuilder result = new StringBuilder();
        result.append("Status Block:\n");
        result.append(String.format("  Version               = %08x  ",
                                    status.getVersion()));
        result.append(String.format("  Link Status           = %08x\n",
                                    status.getLinkStatus()));
        result.append(String.format("  Cell error count      = %-8s  ",
                                    status.getCellErrorCount()));
        result.append(String.format("  Link down count       = %s\n",
                                    status.getLinkDownCount()));
        result.append(String.format("  Link error count      = %-8s  ",
                                    status.getLinkErrorCount()));
        result.append(String.format("  VC buffer status      = %08x\n",
                                    status.getVcBufferStatus()));
        int[] counts = status.getVcRxCounts();
        result.append("  VC rcve counts        =");
        for (int j = 0; j < counts.length; j++) {
            result.append(String.format(" %s", counts[j]));
        }
        result.append("\n");
        counts = status.getVcTxCounts();
        result.append("  VC xmit counts        =");
        for (int j = 0; j < counts.length; j++) {
            result.append(String.format(" %s", counts[j]));
        }
        result.append("\n");
        result.append(String.format("  Sent image count      = %-8s  ",
                                    status.getSentCount()));
        result.append(String.format("  Discarded image count = %s\n",
                                    status.getDiscCount()));
        result.append(String.format("  Truncated image count = %-8s  ",
                                    status.getTruncCount()));
        result.append(String.format("  Image format          = %s",
                                    status.getFormat()));

        return result.toString();
    }


   /**
    ***************************************************************************
    **
    **  Shows all the board temperatures.
    **
    ***************************************************************************
    */
    @Command(name="temperature", description="Show all board temperatures")
    public String temperature() throws REBException
    {
        return formatAdcs(tmp.readAdcs(), 0);
    }


   /**
    ***************************************************************************
    **
    **  Shows a board temperatures.
    **
    ***************************************************************************
    */
    @Command(name="temperature", description="Show a board temperature")
    public String temperature(@Argument(name="first",
                                        description="The temperature to show")
                              int first) throws REBException
    {
        float values[] = new float[1];
        values[0] = tmp.readAdc(first);

        return formatAdcs(values, first);
    }


   /**
    ***************************************************************************
    **
    **  Shows board temperatures.
    **
    ***************************************************************************
    */
    @Command(name="temperature", description="Show board temperatures")
    public String temperature(@Argument(name="first",
                                        description="The first temperature to show")
                              int first,
                              @Argument(name="count",
                                        description="The number of temperatures to show")
                              int count) throws REBException
    {
        return formatAdcs(tmp.readAdcs(first, count), first);
    }


   /**
    ***************************************************************************
    **
    **  Shows all the board power values.
    **
    ***************************************************************************
    */
    @Command(name="power", description="Show all board power values")
    public String power() throws REBException
    {
        return formatAdcs(pwr.readAdcs(), 0);
    }


   /**
    ***************************************************************************
    **
    **  Shows a board power values.
    **
    ***************************************************************************
    */
    @Command(name="power", description="Show a board power value")
    public String power(@Argument(name="first",
                                  description="The power value to show")
                        int first) throws REBException
    {
        float values[] = new float[1];
        values[0] = pwr.readAdc(first);

        return formatAdcs(values, first);
    }


   /**
    ***************************************************************************
    **
    **  Shows board power values.
    **
    ***************************************************************************
    */
    @Command(name="power", description="Show board power values")
    public String power(@Argument(name="first",
                                  description="The first power value to show")
                        int first,
                        @Argument(name="count",
                                  description="The number of power values to show")
                        int count) throws REBException
    {
        return formatAdcs(pwr.readAdcs(first, count), first);
    }


   /**
    ***************************************************************************
    **
    **  Sets a DAC value.
    **
    ***************************************************************************
    */
    @Command(name="dac", description="Set a DAC value")
    public void dac(@Argument(name="number",
                              description="The DAC number (0 or 1)")
                    int number,
                    @Argument(name="chan",
                              description="The channel number (0 - 7)")
                    int chan,
                    @Argument(name="value", description="The value to set")
                    int value) throws REBException
    {
        dac.set(number, chan, value);
    }


   /**
    ***************************************************************************
    **
    **  Continuously read a set of registers.
    **
    ***************************************************************************
    */
    @Command(name="contread", description="Continuously read registers")
    public String contRead(@Argument(name="address",
                                     description="The first register to read")
                           int address,
                           @Argument(name="count",
                                     description="The number of registers to read")
                           int count) throws REBException
    {
        int[] values = new int[count];
        int nRead = 0;
        long startTime = System.currentTimeMillis();
        System.out.println("Press any key to terminate...");
        contActive = true;
        consQueue.offer(0);
        try {
            while (contActive) {
                bset.read(address, values);
                nRead += count;
            }
        }
        finally {
            double rate = (1000.0 * nRead)
                            / (System.currentTimeMillis() - startTime);
            return String.format("Read rate = %.4g Hz", rate);
        }
    }


   /**
    ***************************************************************************
    **
    **  Continuously reads the board temperatures.
    **
    ***************************************************************************
    */
    @Command(name="conttemp", description="Continuously read temperatures")
    public String contTemp() throws REBException
    {
        int nRead = 0;
        long startTime = System.currentTimeMillis();
        System.out.println("Press any key to terminate...");
        contActive = true;
        consQueue.offer(0);
        try {
            while (contActive) {
                tmp.readAdcs();
                nRead++;
            }
        }
        finally {
            double rate = (1000.0 * nRead)
                            / (System.currentTimeMillis() - startTime);
            return String.format("Read rate = %.4g Hz", rate);
        }
    }


   /**
    ***************************************************************************
    **
    **  Continuously read the board power values.
    **
    ***************************************************************************
    */
    @Command(name="contpower", description="Continuously read power values")
    public String contPower() throws REBException
    {
        int nRead = 0;
        long startTime = System.currentTimeMillis();
        System.out.println("Press any key to terminate...");
        contActive = true;
        consQueue.offer(0);
        try {
            while (contActive) {
                pwr.readAdcs();
                nRead++;
            }
        }
        finally {
            double rate = (1000.0 * nRead)
                            / (System.currentTimeMillis() - startTime);
            return String.format("Read rate = %.4g Hz", rate);
        }
    }


   /**
    ***************************************************************************
    **
    **  Formats ADC values.
    **
    ***************************************************************************
    */
    private static String formatAdcs(float[] values, int index)
    {
        StringBuilder result = new StringBuilder();
        for (int j = 0; j < values.length; j++) {
            if ((j & 3) == 0) {
                result.append(j == 0 ? "ADCs:" : "\n     ");
            }
            result.append(String.format("  %2s: %6g", j + index, values[j]));
        }

        return result.toString();
    }


   /**
    ***************************************************************************
    **
    **  Waits for an item to arrive on a queue.
    **
    ***************************************************************************
    */
    private static void take(BlockingQueue queue)
    {
        while (true) {
            try {
                queue.take();
                return;
            }
            catch (InterruptedException e) {
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Waits for terminal input.
    **
    ***************************************************************************
    */
    private static void awaitTerminal()
    {
        while (true) {
            try {
                if (System.in.available() > 0) return;
                try {
                    Thread.sleep(50);
                }
                catch (InterruptedException e) {
                }
            }
            catch (IOException e) {
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Clears terminal input.
    **
    ***************************************************************************
    */
    private static void clearTerminal()
    {
        try {
            while (System.in.available() > 0) {
                System.in.read();
            }
        }
        catch (IOException e) {
        }
    }

}
