package org.lsst.ccs.drivers.reb;

import java.nio.ByteBuffer;

/**
 ***************************************************************************
 **
 **  Image data
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class Image extends ImageMetadata {

   /**
    ***************************************************************************
    **
    **  Private data.
    **
    ***************************************************************************
    */
    private long mdata;        // Reference to the C++ metadata object
    private ByteBuffer data;   // The image data


   /**
    ***************************************************************************
    **
    **  Constructors.
    **
    ***************************************************************************
    */
    public Image()
    {
        super();
    }

    public Image(long tag, int length, int format, int schema, int version,
                 int address, int cluster, int element, long mdata)
    {
        super(tag, length, format, schema, version, address, cluster, element);
        this.mdata = mdata;
    }

    public Image(long tag, int length, int format, int schema, int version,
                 int address, int cluster, int element, ByteBuffer data)
    {
        super(tag, length, format, schema, version, address, cluster, element);
        this.data = data;
    }


   /**
    ***************************************************************************
    **
    **  Gets the image pixel data.
    **
    **  @return  The image's pixel data if it has been successfully read, or
    **           null if it hasn't been. The data returned is a read-only
    **           wrapper around the image data to assure the user cannot
    **           change the state of the image.
    **
    ***************************************************************************
    */
    public ByteBuffer getData()
    {
        return (data != null && data.limit() > 0) ? data.asReadOnlyBuffer()
                                                  : null;
    }


   /**
    ***************************************************************************
    **
    **  Gets the image metadata.
    **
    **  @return  The image's metadata
    **
    ***************************************************************************
    */
    public ImageMetadata getMetadata()
    {
        return new ImageMetadata(tag, length, format, schema, version, address,
                                 cluster, element);
    }


   /**
    ***************************************************************************
    **
    **  Deletes the metadata reference.
    **
    ***************************************************************************
    */
    public void deleteMetadataRef()
    {
        if (mdata != 0) {
            ImageClient.deleteImageMetadataRef(this);
        }
    }


   /**
    ***************************************************************************
    **
    **  Performs finalization.
    **
    **  Allocated non-java memory (in this case the C++ image metadata object)
    **  has to be freed.
    **
    ***************************************************************************
    */
    @Override
    public void finalize() throws Throwable
    {
        super.finalize();
        deleteMetadataRef();
    }

}
