package org.lsst.ccs.drivers.reb;

/**
 ***************************************************************************
 **
 **  Base register set access routines.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class BaseSet extends RegClient {

   /**
    ***************************************************************************
    **
    **  Public constants.
    **
    ***************************************************************************
    */
    public final static int
        RSET_STATUS       = 0,
        RSET_TIME_BASE    = 1,
        RSET_SEQUENCER    = 2,
        RSET_POWER_ADCS   = 3,
        RSET_TEMP_ADCS    = 4,

        REG_SCHEMA        = 0,
        REG_VERSION       = 1,
        REG_ID            = 2,
        REG_TIME_BASE     = 4,
        REG_STATE         = 8,
        REG_TRIGGER       = 9,
        REG_TRIG_TIME     = 10,

        REG_SN_REB_START  = 0x800000,
        REG_SN_REB_VALUE  = 0x800001,
        REG_SN_DREB_START = 0x800010,
        REG_SN_DREB_VALUE = 0x800011,

        REG_DCDC_SYNC     = 0x900000,

        VERSION_CURR      = -3,
        VERSION_UNSET     = -2,
        VERSION_UNKNOWN   = -1,
        VERSION_0         = 0,
        VERSION_1         = 1,
        VERSION_2         = 2,
        VERSION_3         = 3,

        CLOCK_PERIOD_0    = 5,      // Nanoseconds
        CLOCK_PERIOD_1    = 10;     // Nanoseconds

    public final static long
        SN_READ_OKAY      = 0x0001000000000000L,
        SN_READ_ERROR     = 0x0002000000000000L,
        SN_READ_TIMEOUT   = 0x0004000000000000L,
        SN_VALUE_MASK     = 0x0000ffffffffffffL;

   /**
    ***************************************************************************
    **
    **  Private data.
    **
    ***************************************************************************
    */
    private final static int WAIT_TIMEOUT = 1000;
    private int version = VERSION_UNSET;
    private int handleInst;


   /**
    ***************************************************************************
    **
    **  Constructors.
    **
    ***************************************************************************
    */
    public BaseSet()
    {
        super();
    }

    public BaseSet(RegClient reg)
    {
        handle = reg.handle;
    }


   /**
    ***************************************************************************
    **
    **  Gets the firmware version.
    **
    **  @exception  REBException if the firmware version is not recognized
    **
    ***************************************************************************
    */
    public int getVersion() throws REBException
    {
        try {
            checkOpen();
        }
        catch (REBException e) {
            version = VERSION_UNSET;
            throw e;
        }
        if (version == VERSION_UNSET || handleInst != handle.instance) {
            handleInst = handle.instance;
            int schema = read(REG_SCHEMA);
            int fwVersion = read(REG_VERSION);
            if (schema == 0x00f1de00) {
                version = VERSION_0;
            }
            else if (schema == 0x00000001) {
                version = VERSION_1;
            }
            else if (schema == 0x00000002) {
                version = VERSION_2;
            }
            else if (schema == 0x00000003) {
                version = VERSION_3;
            }
            else {
                version = VERSION_UNKNOWN;
                throw new REBException("Unknown firmware version");
            }
        }

        return version;
    }


   /**
    ***************************************************************************
    **
    **  Checks for a particular firmware version.
    **
    **  @param  version  The version to check against
    **
    **  @exception  REBException  if the firmware version is not recognized
    **                            or doesn't match the specified version
    **
    ***************************************************************************
    */
    public void checkVersion(int version) throws REBException
    {
        if (getVersion() != version) {
            throw new REBException("Incompatible firmware version");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks for a particular firmware version range.
    **
    **  @param  lowvers   The lowest valid version
    **
    **  @param  highvers  The highest valid version.  Can be VERSION_CURR to
    **                    cause this check to be skipped.
    **
    **  @exception  REBException  if the firmware version is not recognized
    **                            or doesn't match the specified versions
    **
    ***************************************************************************
    */
    public void checkVersion(int lowvers, int highvers) throws REBException
    {
        if (getVersion() < lowvers
              || (highvers != VERSION_CURR && getVersion() > highvers)) {
            throw new REBException("Incompatible firmware version");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks for not a particular firmware version.
    **
    **  @param  version  The version to check against
    **
    **  @exception  REBException  if the firmware version is not recognized
    **                            or matches the specified version
    **
    ***************************************************************************
    */
    public void checkNotVersion(int version) throws REBException
    {
        if (getVersion() == version) {
            throw new REBException("Incompatible firmware version");
        }
    }


   /**
    ***************************************************************************
    **
    **  Enables the time-base.
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void enable() throws REBException
    {
        enable(RSET_TIME_BASE);
    }


   /**
    ***************************************************************************
    **
    **  Disables the time-base.
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void disable() throws REBException
    {
        disable(RSET_TIME_BASE);
    }


   /**
    ***************************************************************************
    **
    **  Gets the raw time-base value.
    **
    **  @return  The time-base value as clock ticks
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public long getTimeRaw() throws REBException
    {
        return readLong(REG_TIME_BASE);
    }


   /**
    ***************************************************************************
    **
    **  Gets the time-base value as millisecond Unix time.
    **
    **  @return  The time-base value as millisecond Unix time
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public long getTime() throws REBException
    {
        return getPeriod() * getTimeRaw() / 1000000;
    }


   /**
    ***************************************************************************
    **
    **  Sets the raw time-base value.
    **
    **  @param  time  The time-base value as clock ticks
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void setTimeRaw(long time) throws REBException
    {
        writeLong(REG_TIME_BASE, time);
    }


   /**
    ***************************************************************************
    **
    **  Sets the time-base value as millisecond Unix time.
    **
    **  @param  time  The time-base value as millisecond Unix time
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void setTime(long time) throws REBException
    {
        setTimeRaw(1000000 * time / getPeriod());
    }


   /**
    ***************************************************************************
    **
    **  Sets the time-base value to the current system time.
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void setTime() throws REBException
    {
        setTime(System.currentTimeMillis());
    }


   /**
    ***************************************************************************
    **
    **  Enables a register set.
    **
    **  @param  regSet  The register set identifier
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void enable(int regSet) throws REBException
    {
        //        update(REG_TRIGGER, 1<< regSet, 1 << regSet);
        //        write(REG_TRIGGER, read(REG_STATE) | (1 << regSet));
        write(REG_TRIGGER,
              (read(REG_STATE) & ((1 << RSET_STATUS) | (1 << RSET_TIME_BASE)))
                | (1 << regSet));
    }


   /**
    ***************************************************************************
    **
    **  Disables a register set.
    **
    **  @param  regSet  The register set identifier
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void disable(int regSet) throws REBException
    {
        //        update(REG_TRIGGER, 1<< regSet, 0);
        //        write(REG_TRIGGER, read(REG_STATE) & ~(1 << regSet));
        write(REG_TRIGGER,
              (read(REG_STATE) & ((1 << RSET_STATUS) | (1 << RSET_TIME_BASE)))
                & ~(1 << regSet));
    }


   /**
    ***************************************************************************
    **
    **  Waits for a register set to become disabled.
    **
    **  This generally indicates the completion of a read operation
    **
    **  @param  regSet  The register set identifier
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void waitDone(int regSet) throws REBException
    {
        long limit = System.currentTimeMillis() + WAIT_TIMEOUT;
        boolean busy = true;
        while (busy && System.currentTimeMillis() < limit) {
            busy = (read(REG_STATE) & (1 << regSet)) != 0;
        }
        if (busy) {
            throw new REBException("Completion wait timeout");
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the raw trigger time for a register set.
    **
    **  @param  regSet  The register set identifier
    **
    **  @return  The trigger time as clock ticks
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public long getTriggerTimeRaw(int regSet) throws REBException
    {
        return readLong(REG_TRIG_TIME + 2 * regSet);
    }


   /**
    ***************************************************************************
    **
    **  Gets the trigger time for a register set as Unix millisecond time.
    **
    **  @param  regSet  The register set identifier
    **
    **  @return  The trigger time as Unix millisecond time
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public long getTriggerTime(int regSet) throws REBException
    {
        return getPeriod() * getTriggerTimeRaw(regSet) / 1000000;
    }


   /**
    ***************************************************************************
    **
    **  Gets the REB serial number.
    **
    **  @return  The 48-bit REB serial number
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public long getRebSN() throws REBException
    {
        checkVersion(VERSION_3, VERSION_CURR);
        write(REG_SN_REB_START, 0);
        long value = readLong(REG_SN_REB_VALUE);
        if ((value & SN_READ_OKAY) == 0) {
            String problem = (value & SN_READ_ERROR) != 0 ? "error" : "timeout";
            throw new REBException("Serial number read " + problem);
        }
        return value & SN_VALUE_MASK;
    }


   /**
    ***************************************************************************
    **
    **  Gets the DREB serial number.
    **
    **  @return  The 48-bit DREB serial number
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public long getDrebSN() throws REBException
    {
        checkVersion(VERSION_3, VERSION_CURR);
        write(REG_SN_DREB_START, 0);
        long value = readLong(REG_SN_DREB_VALUE);
        if ((value & SN_READ_OKAY) == 0) {
            String problem = (value & SN_READ_ERROR) != 0 ? "error" : "timeout";
            throw new REBException("Serial number read " + problem);
        }
        return value & SN_VALUE_MASK;
    }


   /**
    ***************************************************************************
    **
    **  Sets the DC/DC synchronization.
    **
    **  @param  enable  If true, enable synchronization; otherwise disable it
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public void setDcdcSync(boolean enable) throws REBException
    {
        checkVersion(VERSION_3, VERSION_CURR);
        write(REG_DCDC_SYNC, enable ? 1 : 0);
    }


   /**
    ***************************************************************************
    **
    **  Gets the DC/DC synchronization.
    **
    **  @return  Whether DC/DC synchronization is in effect
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    public boolean isDcdcSync() throws REBException
    {
        checkVersion(VERSION_3, VERSION_CURR);
        return (read(REG_DCDC_SYNC) & 1) != 0;
    }


   /**
    ***************************************************************************
    **
    **  Gets the clock period.
    **
    **  @return  The board's clock period in 10 ns units
    **
    **  @exception  REBException 
    **
    ***************************************************************************
    */
    private int getPeriod() throws REBException
    {
        return getVersion() == VERSION_0 ? CLOCK_PERIOD_0 : CLOCK_PERIOD_1;
    }

}
