package org.lsst.ccs.drivers.rcm;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;

/**
 ***************************************************************************
 **
 **  Program to test the Java image client interface routines
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestImage implements ImageClient.Listener {

    /*
    **  Private fields
    */
    private final static String CUP = "\033[A", NL = "\n";
    private final ImageClient imc = new ImageClient();
    private final BaseSet reg = new BaseSet();
    private int rcmId = -1, imgLength = -1, imgOffset = 0;
    private boolean listening, showTime, reuseImage = true;
    private Image image = new Image();


   /**
    ***************************************************************************
    **
    **  Enumeration for on/off.
    **
    ***************************************************************************
    */
    public enum OnOff {

        ON, OFF;

    }


   /**
    ***************************************************************************
    **
    **  Connects to an RCM.
    **
    ***************************************************************************
    */
    @Command(name="connect", description="Connect to an RCM")
    public void connect(@Argument(name="id", description="RCM ID")
                        int id) throws RcmException
    {
        imc.open(id);
        reg.open(id);
        rcmId = id;
    }


   /**
    ***************************************************************************
    **
    **  Connects to an RCM.
    **
    ***************************************************************************
    */
    @Command(name="connect", description="Connect to an RCM")
    public void connect(@Argument(name="id", description="RCM ID")
                        int id,
                        @Argument(name="ifcname",
                                  description="Network interface name")
                        String ifcName) throws RcmException
    {
        imc.open(id, ifcName);
        reg.open(id, ifcName);
        rcmId = id;
    }


   /**
    ***************************************************************************
    **
    **  Disconnects from an RCM.
    **
    ***************************************************************************
    */
    @Command(name="disconnect", description="Disconnect from an RCM")
    public void disconnect() throws RcmException
    {
        rcmId = -1;
        imc.close();
        reg.close();
    }


   /**
    ***************************************************************************
    **
    **  Turns image data buffer re-use on or off.
    **
    ***************************************************************************
    */
    @Command(name="reuse", description="Enable or disable image buffer re-use")
    public void reuse(@Argument(name="action", description="Action to take")
                         OnOff action)
    {
        reuseImage = (action.ordinal() == 0);
    }


   /**
    ***************************************************************************
    **
    **  Turns image acquisition time display on or off.
    **
    ***************************************************************************
    */
    @Command(name="time", description="Enable or disable time display")
    public void time(@Argument(name="action", description="Action to take")
                         OnOff action)
    {
        showTime = (action.ordinal() == 0);
    }


   /**
    ***************************************************************************
    **
    **  Turns the listener on or off.
    **
    ***************************************************************************
    */
    @Command(name="listen", description="Enable or disable the listener")
    public String listen(@Argument(name="action", description="Action to take")
                         OnOff action)
    {
        String result = checkConnect();
        if (result != null) return result;
        if (action.ordinal() == 0) {
            listening = true;
            imc.setListener(this, reuseImage ? image : null);
        }
        else {
            listening = false;
            imc.clearListener();
        }

        return CUP;
    }


   /**
    ***************************************************************************
    **
    **  Sets the internal image display region.
    **
    ***************************************************************************
    */
    @Command(name="region", description="Set the image display region")
    public void region(@Argument(name="length",
                                 description="The region length (pixels)")
                       int length)
    {
        imgLength = length;
    }


   /**
    ***************************************************************************
    **
    **  Sets the internal image display region.
    **
    ***************************************************************************
    */
    @Command(name="region", description="Set the image display region")
    public void region(@Argument(name="length",
                                 description="The region length (pixels)")
                       int length,
                       @Argument(name="offset",
                                 description="The offset to the region (pixels)")
                       int offset)
    {
        imgLength = length;
        imgOffset = offset;
    }


   /**
    ***************************************************************************
    **
    **  Displays parameters.
    **
    ***************************************************************************
    */
    @Command(name="show", description="Display parameters")
    public String show()
    {
        return String.format(  "RCM ID       = %-8s  Image length = %s" +
                             "\nImage offset = %-8s  Listener on  = %s" +
                             "\nShow time    = %-8s  Reuse image  = %s",
                             rcmId, imgLength, imgOffset, listening, showTime,
                             reuseImage);
    }


   /**
    ***************************************************************************
    **
    **  Waits for an image to arrive.
    **
    ***************************************************************************
    */
    @Command(name="wait", description="Wait for an image to arrive")
    public String awaitImage() throws RcmException
    {
        String result = checkConnect();
        if (result != null) return result;
        result = checkListen();
        if (result != null) return result;
        image = imc.awaitImage(reuseImage ? image : null);
        if (showTime) {
            long cTime = reg.getTime();
            long tTime = reg.getTriggerTime(BaseSet.RSET_SEQUENCER);
            result = "Elapsed time = " + (cTime - tTime) + " msec";
        }
        String text = formatMetadata(image, "Got metadata", false);

        return (result == null) ? text : result + NL + text;
    }


   /**
    ***************************************************************************
    **
    **  Reads an image.
    **
    ***************************************************************************
    */
    @Command(name="read", description="Read an image")
    public String read() throws RcmException
    {
        String result = checkConnect();
        if (result != null) return result;
        result = checkListen();
        if (result != null) return result;
        long start = System.currentTimeMillis();
        boolean imgOkay = imc.readImage(image);
        if (showTime) {
            result = "Elapsed time = " + (System.currentTimeMillis() - start)
                       + " msec";
        }
        if (!imgOkay) {
            String text = "Error reading image data";
            result = (result == null) ? text : result + NL + text;
        }

        return (result == null) ? CUP : result;
    }


   /**
    ***************************************************************************
    **
    **  Displays image data.
    **
    ***************************************************************************
    */
    @Command(name="display", description="Display image data")
    public String display(@Argument(name="length",
                                    description="The display region length (pixels)")
                       int length)
    {
        String text = formatImage(image, 0, length);
        return (text == null) ? CUP : text;
    }


   /**
    ***************************************************************************
    **
    **  Displays image data.
    **
    ***************************************************************************
    */
    @Command(name="display", description="Display image data")
    public String display(@Argument(name="length",
                                    description="The display region length (pixels)")
                          int length,
                          @Argument(name="offset",
                                    description="The offset to the display region (pixels)")
                          int offset)
    {
        String text = formatImage(image, offset, length);
        return (text == null) ? CUP : text;
    }


   /**
    ***************************************************************************
    **
    **  Displays image metadata.
    **
    ***************************************************************************
    */
    @Command(name="metadata", description="Display image metadata")
    public String metadata()
    {
        return formatMetadata(image, "Metadata", true);
    }


   /**
    ***************************************************************************
    **
    **  Handles each received image.
    **
    **  @param  img   The received image
    **
    ***************************************************************************
    */
    @Override
    public void processImage(Image img)
    {
        image = img;
        if (showTime) {
            try {
                long cTime = reg.getTime();
                long tTime = reg.getTriggerTime(BaseSet.RSET_SEQUENCER);
                System.out.println("Elapsed time = " + (cTime - tTime)
                                     + " msec");
            }
            catch (RcmException e) {            
            }
        }
        System.out.println(formatMetadata(image, "Got image", true));
        if (image.getData() != null) {
            String text = formatImage(image, imgOffset, imgLength);
            if (text != null) {
                System.out.println(text);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Formats the metadata for an image.
    **
    **  @param  img        The image whose metadata is to be displayed
    **
    **  @param  caption    A caption to be displayed before the metadata
    **
    **  @param  showValid  If true, display whether or not the image contains
    **                     valid data
    **
    **  @return  A string containing the formatted metadata
    **
    ***************************************************************************
    */
    private String formatMetadata(Image img, String caption, boolean showValid)
    {
        StringBuilder blanks = new StringBuilder(caption.length());
        for (int j = 0; j < caption.length(); j++) {
            blanks.append(" ");
        }
        return String.format("%s: tag = %016x, length = %s, format = %s," + NL
                               + "%s  schema = %08x,  version = %08x, "
                               + "address = %s," + NL
                               + "%s  cluster = %s, element = %s%s",
                             caption, image.getTag(), image.getLength(),
                             image.getFormat(), blanks, image.getSchema(),
                             image.getVersion(), image.getAddress(),
                             blanks, image.getCluster(), image.getElement(),
                             showValid ? image.getData() != null ? ", valid"
                                       : ", invalid" : "");
    }


   /**
    ***************************************************************************
    **
    **  Formats the contents of an image.
    **
    **  @param  img     The image to be displayed
    **
    **  @param  offset  The offset to the first pixel to display
    **
    **  @param  count   The number of pixels to display.  If zero, display up
    **                  to the stored length.  If negative, display nothing.
    **
    **  @return  The forrmatted image contents
    **
    ***************************************************************************
    */
    private String formatImage(Image img, int offset, int count)
    {
        if (count < 0) {
            return null;
        }
        ByteBuffer buff = img.getData();
        if (buff == null) {
            return "Image contains no valid data";
        }

        offset = (offset < 0) ? 0 : offset;
        int iCount = img.getLength();
        if (count == 0 || count + offset > iCount) {
            count = iCount - offset;
        }
        if (count <= 0) {
            return null;
        }

        StringBuilder text = new StringBuilder();
        buff.order(ByteOrder.LITTLE_ENDIAN);
        buff.position(4 * offset);
        for (int j = 0; j < count; j++) {
            if ((j & 0x03) == 0) {
                if (j != 0) {
                    text.append(NL);
                }
                text.append(String.format("%08x:", offset + j));
            }
            text.append(String.format(" %08x", buff.getInt()));
        }

        return text.toString();
    }


   /**
    ***************************************************************************
    **
    **  Checks that an RCM is connected.
    **
    ***************************************************************************
    */
    private String checkConnect()
    {
        return (imc != null) ? null : "No RCM connected";
    }


   /**
    ***************************************************************************
    **
    **  Checks that the listener is not active.
    **
    ***************************************************************************
    */
    private String checkListen()
    {
        return !listening ? null : "Listener is active";
    }

}
