package org.lsst.ccs.drivers.rcm;

/**
 ***************************************************************************
 **
 **  RCM status block access routines.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class StatusSet extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants.
    **
    ***************************************************************************
    */
    public final static int
        REG_STATUS_0      = 0x800000,
        REG_STATUS        = 0xa00000,
        OFF_STAT_VERSION  = 0,
        OFF_STAT_LINKSTAT = 1,
        OFF_STAT_CERR_CNT = 2,
        OFF_STAT_LDWN_CNT = 3,
        OFF_STAT_LERR_CNT = 4,
        OFF_STAT_BUFFSTAT = 5,
        OFF_STAT_RX_CNTS  = 6,
        OFF_STAT_TX_CNTS  = 10,
        NUM_STAT_REG      = 14;
 
   /**
    ***************************************************************************
    **
    **  Private data.
    **
    ***************************************************************************
    */
    private int baseAddr;


   /**
    ***************************************************************************
    **
    **  Constructors.
    **
    ***************************************************************************
    */
    public StatusSet()
    {
        super();
    }

    public StatusSet(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Enables the status block.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    @Override
    public void enable() throws RcmException
    {
        enable(RSET_STATUS);
    }


   /**
    ***************************************************************************
    **
    **  Disables the status block.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    @Override
    public void disable() throws RcmException
    {
        disable(RSET_STATUS);
    }


   /**
    ***************************************************************************
    **
    **  Gets the status block trigger time.
    **
    **  @return  The Unix millisecond time of the last status block trigger
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public long getTriggerTime() throws RcmException
    {
        return getTriggerTime(RSET_STATUS);
    }


   /**
    ***************************************************************************
    **
    **  Reads the status block.
    **
    **  @return  The contents of the status block, as a Status object
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public Status readStatus() throws RcmException
    {
        Status status = new Status();
        int[] regs = new int[NUM_STAT_REG];
        read(getVersion() >= VERSION_3 ? REG_STATUS : REG_STATUS_0, regs);
        status.version = regs[OFF_STAT_VERSION];
        status.linkStat = regs[OFF_STAT_LINKSTAT];
        status.cerrCount = regs[OFF_STAT_CERR_CNT];
        status.ldwnCount = regs[OFF_STAT_LDWN_CNT];
        status.lerrCount = regs[OFF_STAT_LERR_CNT];
        status.buffStat = regs[OFF_STAT_BUFFSTAT];
        for (int j = 0; j < status.rxCount.length; j++)
            status.rxCount[j] = regs[OFF_STAT_RX_CNTS + j];
        for (int j = 0; j < status.txCount.length; j++)
            status.txCount[j] = regs[OFF_STAT_TX_CNTS + j];

        return status;
    }

}
