package org.lsst.ccs.drivers.rcm;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;

/**
 ***************************************************************************
 **
 **  Sequencer utility routines.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class SequencerUtils extends Sequencer {

   /**
    ***************************************************************************
    **
    **  Private constants
    **
    ***************************************************************************
    */
    private final static int 
        SEQTYPE_REGISTER = 0,
        SEQTYPE_OUTPUT   = 1,
        SEQTYPE_TIMES    = 2,
        SEQTYPE_PROGRAM  = 3,
        SEQTYPE_PROGEXEC = 4,
        SEQTYPE_PROGJUMP = 5,
        SEQTYPE_PROGSUBE = 6,
        SEQTYPE_PROGEND  = 7,
        SEQTYPE_SLICE    = 8,
        SEQTYPE_SOURCE   = 9,
        SEQTYPE_STACK    = 10,
        SEQTYPE_BEB      = 11;

    private final static Map<String, Integer> seqTypes = new HashMap<>();
    static {
        seqTypes.put("register", SEQTYPE_REGISTER);
        seqTypes.put("output",   SEQTYPE_OUTPUT);
        seqTypes.put("times",    SEQTYPE_TIMES);
        seqTypes.put("program",  SEQTYPE_PROGRAM);
        seqTypes.put("progexec", SEQTYPE_PROGEXEC);
        seqTypes.put("progjump", SEQTYPE_PROGJUMP);
        seqTypes.put("progsube", SEQTYPE_PROGSUBE);
        seqTypes.put("progend",  SEQTYPE_PROGEND);
        seqTypes.put("slice",    SEQTYPE_SLICE);
        seqTypes.put("source",   SEQTYPE_SOURCE);
        seqTypes.put("stack",    SEQTYPE_STACK);
        seqTypes.put("beb",      SEQTYPE_BEB);
    }

    private final static Map<Integer, Integer> seqMinArgs = new HashMap<>();
    static {
        seqMinArgs.put(SEQTYPE_REGISTER, 3);
        seqMinArgs.put(SEQTYPE_OUTPUT,   3);
        seqMinArgs.put(SEQTYPE_TIMES,    3);
        seqMinArgs.put(SEQTYPE_PROGRAM,  3);
        seqMinArgs.put(SEQTYPE_PROGEXEC, 4);
        seqMinArgs.put(SEQTYPE_PROGJUMP, 4);
        seqMinArgs.put(SEQTYPE_PROGSUBE, 2);
        seqMinArgs.put(SEQTYPE_PROGEND,  2);
        seqMinArgs.put(SEQTYPE_SLICE,    2);
        seqMinArgs.put(SEQTYPE_SOURCE,   2);
        seqMinArgs.put(SEQTYPE_STACK,    3);
        seqMinArgs.put(SEQTYPE_BEB,      2);
    }

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private int stkSize;


   /**
    ***************************************************************************
    **
    **  Constructors.
    **
    ***************************************************************************
    */
    public SequencerUtils()
    {
        super();
    }

    public SequencerUtils(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Carries out a sequencer load command.
    **
    **  @param  cmnd  A sequencer command
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void loadCommand(String cmnd) throws RcmException
    {
        loadCommand(cmnd, 0);
    }


   /**
    ***************************************************************************
    **
    **  Carries out a sequencer load command.
    **
    **  @param  cmnd     A sequencer command
    **
    **  @param  lineNum  The line number associated with the command, or zero
    **                   if there is no line number
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void loadCommand(String cmnd, int lineNum) throws RcmException
    {
        boolean thrown = false;
        List<Integer> args = new ArrayList<>();
        String word = "";

        try {
            Scanner scan = new Scanner(cmnd);
            int type = -1, count = 0;
            args.clear();
            while (scan.hasNext()) {
                word = scan.next();
                if (count == 0) {
                    if (word.startsWith("#")) break;
                    Integer iType = seqTypes.get(word);
                    if (iType == null) {
                        thrown = true;
                        throwException("Invalid operation (" + word + ")",
                                       lineNum);
                    }
                    type = iType;
                }
                else {
                    args.add((int)(long)Long.decode(word));
                }
                count++;
            }
            if (count == 0) return;
            int minArgs = seqMinArgs.get(type);
            if (count < minArgs) {
                thrown = true;
                throwException("Too few arguments (" + (count - 1) + ")",
                               lineNum);
            }
            if (minArgs != 3 && count > minArgs) {
                thrown = true;
                throwException("Too many arguments (" + (count - 1) + ")",
                               lineNum);
            }
            int arg0 = args.get(0);
            if (minArgs == 2) {
                if (type == SEQTYPE_PROGSUBE) {
                    writeProgEndSubr(arg0);
                }
                else if (type == SEQTYPE_PROGEND) {
                    writeProgEnd(arg0);
                }
                else if (type == SEQTYPE_SLICE) {
                    writeSliceCount(arg0);
                }
                else if (type == SEQTYPE_SOURCE) {
                    writeDataSource(arg0);
                }
                else if (type == SEQTYPE_BEB) {
                    writeBebSelect(arg0);
                }
            }
            else {
                int[] values = new int[count - 2];
                for (int j = 0; j < values.length; j++) {
                    values[j] = args.get(j + 1);
                }
                if (type == SEQTYPE_REGISTER) {
                    write(arg0, values);
                }
                else if (type == SEQTYPE_OUTPUT) {
                    writeLines(arg0, values);
                }
                else if (type == SEQTYPE_TIMES) {
                    writeTimes(arg0, values);
                }
                else if (type == SEQTYPE_PROGRAM) {
                    writeProgram(arg0, values);
                }
                else if (type == SEQTYPE_PROGEXEC) {
                    writeProgExec(arg0, values[0], values[1]);
                }
                else if (type == SEQTYPE_PROGJUMP) {
                    writeProgJump(arg0, values[0], values[1]);
                }
                else if (type == SEQTYPE_STACK) {
                    writeStack(arg0, values);
                    if (arg0 + values.length > stkSize) {
                        stkSize = arg0 + values.length;
                        writeStackSize(stkSize);
                    }
                }
            }
        }
        catch (NumberFormatException e) {
            throwException("Invalid integer (" + word + ")", lineNum);
        }
        catch (RcmException e) {
            if (thrown) {
                throw e;
            }
            else {
                throwException(e.getMessage(), lineNum);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Loads the sequencer from a file.
    **
    **  @param  fileName  The name of the file containing sequencer commands
    **
    **  @return  The number of slices generated by the sequencer program
    **
    **  @exception  RcmException 
    **  @exception  IOException 
    **
    ***************************************************************************
    */
    public int loadFile(String fileName) throws RcmException, IOException
    {
        BufferedReader rdr = null;
        int lineNum = 0;
        stkSize = 0;
        clearCache();

        try {
            rdr = new BufferedReader(new FileReader(fileName));
            while (true) {
                String line = rdr.readLine();
                if (line == null) break;
                lineNum++;
                loadCommand(line, lineNum);
            }
        }
        finally {
            if (rdr != null) {
                try {
                    rdr.close();
                }
                catch (IOException e) {
                }
            }
        }

        return getCacheSliceCount();
    }


   /**
    ***************************************************************************
    **
    **  Throws a parsing error exception.
    **
    **  @param  text     The text message for the exception
    **
    **  @param  lineNum  The line number where the error occurred, or zero if
    **                   no line number is available
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private void throwException(String text, int lineNum) throws RcmException
    {
        if (lineNum > 0) {
            throw new RcmException(text + " at line " + lineNum);
        }
        else {
            throw new RcmException(text);
        }
    }

}
