package org.lsst.ccs.drivers.rcm;

/**
 ***************************************************************************
 **
 **  RCM sequencer access routines.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class Sequencer extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants.
    **
    ***************************************************************************
    */
    public final static int
        REG_SEQ_LINES    = 0x100000,
        REG_SEQ_TIMES    = 0x200000,
        REG_SEQ_STACK    = 0x300000,
        REG_SEQ_PROGRAM  = 0x300000,
        REG_SEQ_STEP     = 0x310000,
        REG_SEQ_STOP     = 0x320000,
        REG_SEQ_SSIZE    = 0x400000,
        REG_SEQ_NSLICE   = 0x400005,
        REG_SEQ_SOURCE   = 0x400006,
        REG_SEQ_STRIPE   = 0x400007,
        REG_SEQ_BEB      = 0x400007,
        REG_SEQ_STEP_0   = 0x500000,
        REG_SEQ_STOP_0   = 0x600000,
        SEQ_MAX_FUNC_0   = 4,
        SEQ_MAX_FUNC     = 16,
        SEQ_MAX_SLICE    = 16,
        SEQ_MAX_PROGRAM  = 1024,
        SEQ_MAX_STACK    = 4096,
        SEQ_SRC_FPGA_PTN = 0,
        SEQ_SRC_BEB_PTN  = 0x1beb,
        SEQ_SRC_BEB_ADC  = 0x0beb,
        SEQ_SRC_ADC      = 0,
        SEQ_SRC_PATTERN  = 1,
        SEQ_OPC_EXECUTE  = 1,
        SEQ_OPC_JUMP     = 2,
        SEQ_OPC_END_SUBR = 3,
        SEQ_OPC_END_PROG = 4,
        SEQ_PRG_V_OPCODE = 28,
        SEQ_PRG_V_FUNC   = 24,
        SEQ_PRG_M_FUNC   = SEQ_MAX_FUNC - 1,
        SEQ_PRG_V_LOOP   = 23,
        SEQ_PRG_M_EXCCNT = 0x7fffff,
        SEQ_PRG_V_SUBADD = 18,
        SEQ_PRG_M_SUBADD = 0x3ff,
        SEQ_PRG_M_SUBCNT = 0x1ffff,
        SEQ_STK_V_FUNC   = 28,
        SEQ_STK_M_FUNC   = SEQ_MAX_FUNC_0 - 1,
        SEQ_STK_V_LOOP   = 27,
        SEQ_STK_M_COUNT  = 0x7ffffff,
        MAX_SUBR_LEVEL   = 16;

   /**
    ***************************************************************************
    **
    **  Fields.
    **
    ***************************************************************************
    */
    protected int[][] lines = new int[SEQ_MAX_FUNC][SEQ_MAX_SLICE];
    protected int[][] times = new int[SEQ_MAX_FUNC][SEQ_MAX_SLICE];
    protected int[] prog  = new int[SEQ_MAX_PROGRAM];
    protected int[] stack = new int[SEQ_MAX_STACK];
    protected int stackSize;
        

   /**
    ***************************************************************************
    **
    **  Constructors.
    **
    ***************************************************************************
    */
    public Sequencer(RegClient reg)
    {
        super(reg);
    }

    public Sequencer()
    {
        super();
    }


   /**
    ***************************************************************************
    **
    **  Enables the sequencer.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    @Override
    public void enable() throws RcmException
    {
        enable(RSET_SEQUENCER);
    }


   /**
    ***************************************************************************
    **
    **  Disables the sequencer.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    @Override
    public void disable() throws RcmException
    {
        disable(RSET_SEQUENCER);
    }


   /**
    ***************************************************************************
    **
    **  Gets the sequencer trigger time.
    **
    **  @return  The Unix millisecond time of the sequencer trigger
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public long getTriggerTime() throws RcmException
    {
        return getTriggerTime(RSET_SEQUENCER);
    }


   /**
    ***************************************************************************
    **
    **  Writes an output lines value.
    **
    **  @param  func   The function number
    **
    **  @param  slice  The time slice number
    **
    **  @param  value  The output lines value
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeLines(int func, int slice, int value)
        throws RcmException
    {
        checkFunc(func);
        checkSlice(slice);
        write(REG_SEQ_LINES | (func<< 4) | slice, value);
        lines[func][slice] = value;
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of output lines values.
    **
    **  @param  func    The function number
    **
    **  @param  values  The array of output lines values
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeLines(int func, int[] values) throws RcmException
    {
        writeLines(func, values, 0, values.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of output lines values.
    **
    **  @param  func    The function number
    **
    **  @param  values  The array of output lines values
    **
    **  @param  offset  The offset in values of the first value to write
    **
    **  @param  count   The number of values to write
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeLines(int func, int[] values, int offset, int count)
        throws RcmException
    {
        checkFunc(func);
        checkSlice(count - 1);
        write(REG_SEQ_LINES | (func << 4), values, offset, count);
        System.arraycopy(values, offset, lines[func], 0, count);
    }


   /**
    ***************************************************************************
    **
    **  Writes a time slice value.
    **
    **  @param  func   The function number
    **
    **  @param  slice  The time slice number
    **
    **  @param  value  The time slice value
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeTimes(int func, int slice, int value)
        throws RcmException
    {
        checkFunc(func);
        checkSlice(slice);
        write(REG_SEQ_TIMES | (func << 4) | slice, value);
        times[func][slice] = value;
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of time slice values.
    **
    **  @param  func    The function number
    **
    **  @param  values  The array of time slice values
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeTimes(int func, int[] values) throws RcmException
    {
        writeTimes(func, values, 0, values.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of time slice values.
    **
    **  @param  func    The function number
    **
    **  @param  values  The array of time slice values
    **
    **  @param  offset  The offset in values of the first value to write
    **
    **  @param  count   The number of values to write
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeTimes(int func, int[] values, int offset, int count)
        throws RcmException
    {
        checkFunc(func);
        checkSlice(count - 1);
        write(REG_SEQ_TIMES | (func << 4), values, offset, count);
        System.arraycopy(values, offset, times[func], 0, count);
    }


   /**
    ***************************************************************************
    **
    **  Writes a stack value.
    **
    **  @param  addr   The stack address
    **
    **  @param  value  The value to write
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeStack(int addr, int value) throws RcmException
    {
        checkVersion(VERSION_0);
        checkStackAddr(addr);
        write(REG_SEQ_STACK + addr, value);
        stack[addr] = value;
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of stack values.
    **
    **  @param  addr    The first stack address
    **
    **  @param  values  The array of values to write
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeStack(int addr, int[] values) throws RcmException
    {
        writeStack(addr, values, 0, values.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of stack values.
    **
    **  @param  addr    The first stack address
    **
    **  @param  values  The array of stack values to write
    **
    **  @param  offset  The offset in values of the first value to write
    **
    **  @param  count   The number of values to write
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeStack(int addr, int[] values, int offset, int count)
        throws RcmException
    {
        checkVersion(VERSION_0);
        checkStackAddr(addr, count);
        write(REG_SEQ_STACK + addr, values, offset, count);
        System.arraycopy(values, offset, stack, addr, count);
    }


   /**
    ***************************************************************************
    **
    **  Writes a complete set of stack values, setting the stack size.
    **
    **  @param  values  The array of stack values to write
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeStack(int[] values) throws RcmException
    {
        writeStack(values, 0, values.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes a complete set of stack values, setting the stack size.
    **
    **  @param  values  The array of stack values to write
    **
    **  @param  offset  The offset in values of the first item to write
    **
    **  @param  count   The number of values to write
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeStack(int[] values, int offset, int count)
        throws RcmException
    {
        checkVersion(VERSION_0);
        checkStackAddr(0, count);
        write(REG_SEQ_STACK, values, offset, count);
        write(REG_SEQ_SSIZE, count);
        System.arraycopy(values, offset, stack, 0, count);
        stackSize = count;
    }


   /**
    ***************************************************************************
    **
    **  Writes the stack size.
    **
    **  @param  value  The value to write to the stack size register
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeStackSize(int value) throws RcmException
    {
        checkVersion(VERSION_0);
        write(REG_SEQ_SSIZE, value);
        stackSize = value;
    }


   /**
    ***************************************************************************
    **
    **  Writes a program word.
    **
    **  @param  addr   The program address
    **
    **  @param  value  The value to write
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeProgram(int addr, int value) throws RcmException
    {
        checkNotVersion(VERSION_0);
        checkProgAddr(addr);
        write(REG_SEQ_PROGRAM + addr, value);
        prog[addr] = value;
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of program words.
    **
    **  @param  addr    The first program address
    **
    **  @param  values  The array of values to write
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeProgram(int addr, int[] values) throws RcmException
    {
        writeProgram(addr, values, 0, values.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of program words.
    **
    **  @param  addr    The first program address
    **
    **  @param  values  The array of program words to write
    **
    **  @param  offset  The offset in values of the first value to write
    **
    **  @param  count   The number of values to write
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeProgram(int addr, int[] values, int offset, int count)
        throws RcmException
    {
        checkNotVersion(VERSION_0);
        checkProgAddr(addr, count);
        write(REG_SEQ_PROGRAM + addr, values, offset, count);
        System.arraycopy(values, offset, prog, addr, count);
    }


   /**
    ***************************************************************************
    **
    **  Writes an execute program word.
    **
    **  @param  addr   The program address
    **
    **  @param  func   The ID of the function to execute
    **
    **  @param  count  The number of times to execute the function
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeProgExec(int addr, int func, int count)
        throws RcmException
    {
        checkFunc(func);
        int value = count & SEQ_PRG_M_EXCCNT;
        if (count != -1 && count != value) {
            throw new RcmException("Invalid repetition count");
        }
        value |= (SEQ_OPC_EXECUTE << SEQ_PRG_V_OPCODE)
                   | (func << SEQ_PRG_V_FUNC)
                   | (count == -1 ? (1 << SEQ_PRG_V_LOOP) : 0);
        writeProgram(addr, value);
    }


   /**
    ***************************************************************************
    **
    **  Writes a jump program word.
    **
    **  @param  addr      The program address
    **
    **  @param  subaddr   The address of the subroutine to jump to
    **
    **  @param  count     The number of times to run the subroutine
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeProgJump(int addr, int subaddr, int count)
        throws RcmException
    {
        checkProgAddr(subaddr);
        int value = count & SEQ_PRG_M_SUBCNT;
        if (count != value) {
            throw new RcmException("Invalid repetition count");
        }
        value |= (SEQ_OPC_JUMP << SEQ_PRG_V_OPCODE)
                   | (subaddr << SEQ_PRG_V_SUBADD);
        writeProgram(addr, value);
    }


   /**
    ***************************************************************************
    **
    **  Writes an end subroutine word.
    **
    **  @param  addr  The program address
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeProgEndSubr(int addr) throws RcmException
    {
        writeProgram(addr, SEQ_OPC_END_SUBR << SEQ_PRG_V_OPCODE);
    }


   /**
    ***************************************************************************
    **
    **  Writes an end program word.
    **
    **  @param  addr  The program address
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeProgEnd(int addr) throws RcmException
    {
        writeProgram(addr, SEQ_OPC_END_PROG << SEQ_PRG_V_OPCODE);
    }


   /**
    ***************************************************************************
    **
    **  Writes the slice count.
    **
    **  @param  value  The slice count value to write
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeSliceCount(int value) throws RcmException
    {
        write(REG_SEQ_NSLICE, value);
    }


   /**
    ***************************************************************************
    **
    **  Reads the slice count.
    **
    **  @return  The slice count value read
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public int readSliceCount() throws RcmException
    {
        return read(REG_SEQ_NSLICE);
    }


   /**
    ***************************************************************************
    **
    **  Writes the data source register.
    **
    **  @param  value  The value to write
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeDataSource(int value) throws RcmException
    {
        write(REG_SEQ_SOURCE, value);
    }


   /**
    ***************************************************************************
    **
    **  Reads the data source register.
    **
    **  @return  The slice count value read
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public int readDataSource() throws RcmException
    {
        return read(REG_SEQ_SOURCE);
    }


   /**
    ***************************************************************************
    **
    **  Writes the stripe selection register.
    **
    **  @param  value  The value to write
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeStripeSelect(int value) throws RcmException
    {
        checkVersion(VERSION_3, VERSION_CURR);
        write(REG_SEQ_STRIPE, value);
    }


   /**
    ***************************************************************************
    **
    **  Reads the stripe selection register.
    **
    **  @return  The slice count value read
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public int readStripeSelect() throws RcmException
    {
        checkVersion(VERSION_3, VERSION_CURR);
        return read(REG_SEQ_STRIPE);
    }


   /**
    ***************************************************************************
    **
    **  Writes the BEB selector register.
    **
    **  @param  value  The value to write
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeBebSelect(int value) throws RcmException
    {
        checkVersion(VERSION_0);
        write(REG_SEQ_BEB, value);
    }


   /**
    ***************************************************************************
    **
    **  Sends the step command.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void sendStep() throws RcmException
    {
        int stepAddr = getVersion() == VERSION_0 ? REG_SEQ_STEP_0 : REG_SEQ_STEP;
        write(stepAddr, 0);
    }


   /**
    ***************************************************************************
    **
    **  Sends the stop command.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void sendStop() throws RcmException
    {
        int stopAddr = getVersion() == VERSION_0 ? REG_SEQ_STOP_0 : REG_SEQ_STOP;
        write(stopAddr, 0);
    }


   /**
    ***************************************************************************
    **
    **  Clears the sequencer program cached copy.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void clearCache() throws RcmException
    {
        for (int j = 0; j < lines.length; j++) {
            for (int k = 0; k < lines[j].length; k++) {
                lines[j][k] = 0;
                times[j][k] = 0;
            }
        }
        if (getVersion() == VERSION_0) {
            for (int j = 0; j < stack.length; j++) {
                stack[j] = 0;
            }
            stackSize = 0;
        }
        else {
            for (int j = 0; j < prog.length; j++) {
                prog[j] = 0;
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the slice count of the sequencer program cached copy.
    **
    **  @return  The number of slices generated by the whole program
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public int getCacheSliceCount() throws RcmException
    {
        int maxFunc = getVersion() == VERSION_0 ? SEQ_MAX_FUNC_0 : SEQ_MAX_FUNC;
        int[] funcSlice = new int[maxFunc];
        funcSlice[0] = 0;
        for (int j = 1; j < maxFunc; j++) {
            int nSlice = 0;
            boolean prevAdc = false;
            for (int k = 0; k < SEQ_MAX_SLICE && times[j][k] != 0; k++) {
                boolean adc = (lines[j][k] & 0x1000) != 0;
                if (adc & !prevAdc) {
                    nSlice++;
                }
                prevAdc = adc;
            }
            funcSlice[j] = nSlice;
        }
        if (getVersion() == VERSION_0) {
            int nSlice = 0;
            for (int j = 0; j < stackSize; j++) {
                int func = (stack[j] >> SEQ_STK_V_FUNC) & SEQ_STK_M_FUNC;
                int count = stack[j] & SEQ_STK_M_COUNT;
                if (funcSlice[func] != 0
                      && (stack[j] & (1 << SEQ_STK_V_LOOP)) != 0) {
                    throw new RcmException("Data-producing infinite loop");
                }
                nSlice += count * funcSlice[func];
            }
            return nSlice;
        }
        else {
            return getProgSlices(0, 0, funcSlice);
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of slices generated by a sequencer program routine.
    **
    **  @param  addr       The program address
    **
    **  @param  level      The subroutine nesting level
    **
    **  @param  funcSlice  The array of slice counts for each defined function
    **
    **  @return  The number of slices generated by the routine
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private int getProgSlices(int addr, int level, int[] funcSlice)
        throws RcmException
    {
        int nSlice = 0;

        for (; addr < SEQ_MAX_PROGRAM; addr++) {
            int cmnd = prog[addr];

            switch (cmnd >>> SEQ_PRG_V_OPCODE) {

            case SEQ_OPC_EXECUTE:
                int func = (cmnd >> SEQ_PRG_V_FUNC) & SEQ_PRG_M_FUNC;
                if (funcSlice[func] != 0
                      && (cmnd & (1 << SEQ_PRG_V_LOOP)) != 0) {
                    throw new RcmException("Data-producing infinite loop");
                }
                nSlice += (cmnd & SEQ_PRG_M_EXCCNT) * funcSlice[func];
                break;

            case SEQ_OPC_JUMP:
                if (level >= MAX_SUBR_LEVEL) {
                    throw new RcmException("Maximum subroutine level exceeded");
                }
                int sAddr = (cmnd >> SEQ_PRG_V_SUBADD) & SEQ_PRG_M_SUBADD;
                int count = cmnd & SEQ_PRG_M_SUBCNT;
                int slice = getProgSlices(sAddr, level + 1, funcSlice);
                nSlice += (slice >= 0 ? count * slice : -1 ^ slice);
                if (slice < 0) {
                    return nSlice ^ (level == 0 ? 0 : -1);
                }
                break;

            case SEQ_OPC_END_SUBR:
                if (level == 0) {
                    throw new RcmException("Subroutine trailer at top level");
                }
                return nSlice;

            case SEQ_OPC_END_PROG:
                return nSlice ^ (level == 0 ? 0 : -1);

            default:
                throw new RcmException("Invalid sequencer opcode ("
                                         + (cmnd >>> SEQ_PRG_V_OPCODE) + ")");

            }
        }

        if (level == 0) {
            throw new RcmException("Missing end of program");
        }
        else {
            throw new RcmException("Missing subroutine trailer");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a function number.
    **
    **  @param  func  The function ID
    **
    **  @exception  RcmException if function ID is invalid
    **
    ***************************************************************************
    */
    private void checkFunc(int func) throws RcmException
    {
        int maxFunc = getVersion() == VERSION_0
                        ? SEQ_MAX_FUNC_0 : SEQ_MAX_FUNC;
        if (func < 0 || func >= maxFunc) {
            throw new RcmException("Invalid function number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a time slice number.
    **
    **  @param  slice  The time slice number
    **
    **  @exception  RcmException if slice number is invalid
    **
    ***************************************************************************
    */
    private void checkSlice(int slice) throws RcmException
    {
        if (slice < 0 || slice >= SEQ_MAX_SLICE) {
            throw new RcmException("Invalid time slice number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a stack address.
    **
    **  @param  addr  The stack address
    **
    **  @exception  RcmException if stack address is invalid
    **
    ***************************************************************************
    */
    private void checkStackAddr(int addr) throws RcmException
    {
        if (addr < 0 || addr >= SEQ_MAX_STACK) {
            throw new RcmException("Invalid stack address");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a range of stack addresses.
    **
    **  @param  addr   The first stack address
    **
    **  @param  count  The number of stack addresses
    **
    **  @exception  RcmException if stack address range is invalid
    **
    ***************************************************************************
    */
    private void checkStackAddr(int addr, int count) throws RcmException
    {
        if (addr < 0 || count <= 0 || addr + count > SEQ_MAX_STACK) {
            throw new RcmException("Invalid stack address");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a program address.
    **
    **  @param  addr  The program address
    **
    **  @exception  RcmException if program address id invalid
    **
    ***************************************************************************
    */
    private void checkProgAddr(int addr) throws RcmException
    {
        if (addr < 0 || addr >= SEQ_MAX_PROGRAM) {
            throw new RcmException("Invalid program address");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a range of program addresses.
    **
    **  @param  addr   The first program address
    **
    **  @param  count  The number of program addresses
    **
    **  @exception  RcmException if program address range is invalid
    **
    ***************************************************************************
    */
    private void checkProgAddr(int addr, int count) throws RcmException
    {
        if (addr < 0 || count <= 0 || addr + count > SEQ_MAX_PROGRAM) {
            throw new RcmException("Invalid program address");
        }
    }

}
