package org.lsst.ccs.drivers.rcm;

/**
 ***************************************************************************
 **
 **  CABAC access routines.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class Cabac extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants.
    **
    ***************************************************************************
    */
    public final static int
        REG_CABAC         = 0x500000,
        CABAC_OFF_LOAD    = 0,
        CABAC_OFF_FETCH   = 1,
        CABAC_OFF_RESET   = 2,
        CABAC_OFF_LOAD_1  = 5,
        CABAC_OFF_RESET_1 = 6,
        CABAC_OFF_FETCH_1 = 7,
        CABAC_OFF_BOTTOM  = 0x10,
        CABAC_OFF_TOP     = 0x20,
        CABAC_OFF_WRITE   = 0,
        CABAC_OFF_READ    = 0x100,
        NUM_CABAC_REGS    = 5,
        CABAC_ID_BOTTOM   = 0,
        CABAC_ID_TOP      = 1,
        NUM_CABAC_PAIRS   = 3,
        NUM_CABACS        = 6;

    public final static int
        CABAC_FLD_OD_EXP0 = 0,
        CABAC_FLD_OD_EXP1 = 1,
        CABAC_FLD_OD_RDO0 = 2,
        CABAC_FLD_OD_RDO1 = 3,
        CABAC_FLD_PCLK0   = 4,
        CABAC_FLD_PCLK1   = 5,
        CABAC_FLD_PCLK2   = 6,
        CABAC_FLD_PCLK3   = 7,
        CABAC_FLD_SCLK0   = 8,
        CABAC_FLD_SCLK1   = 9,
        CABAC_FLD_SCLK2   = 10,
        CABAC_FLD_RG      = 11,
        CABAC_FLD_GD      = 12,
        CABAC_FLD_OG      = 13,
        CABAC_FLD_RD      = 14,
        CABAC_FLD_SPH     = 15,
        CABAC_FLD_SPL     = 16,
        CABAC_FLD_MUX0    = 17,
        CABAC_FLD_MUX1    = 18,
        CABAC_FLD_PULSE   = 19,
        CABAC_FLD_MUXE    = 20,
        NUM_CABAC_FLDS    = 21;
        
    public final static int[]
        FIELD_LENGTHS  = {  8,   8,   8,   8,   8,   8,   8,   8,
                            8,   8,   8,   8,   8,   8,   8,   8,  
                            8,   4,   4,   1,   1},
        FIELD_STARTS   = {  0,   8,  16,  24,  32,  40,  48,  56,
                           64,  72,  80,  88,  96, 104, 112, 120,
                          128, 136, 140, 144, 145},
        FIELD_STARTS_1 = {  2,  10,  18,  26,  34,  42,  50,  58,
                           66,  74,  82,  90,  98, 106, 114, 122,
                          130, 138, 142,   0,   1};


   /**
    ***************************************************************************
    **
    **  Constructors.
    **
    ***************************************************************************
    */
    public Cabac()
    {
        super();
    }

    public Cabac(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Loads a CABAC pair from the registers.
    **
    **  @param  pair  The number of the pair to be loaded
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void load(int pair) throws RcmException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        checkPair(pair);
        write(REG_CABAC + CABAC_OFF_LOAD, pair);
    }


   /**
    ***************************************************************************
    **
    **  Resets a CABAC pair.
    **
    **  @param  pair  The number of the pair to be reset
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void reset(int pair) throws RcmException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        checkPair(pair);
        write(REG_CABAC + CABAC_OFF_RESET, pair);
    }


   /**
    ***************************************************************************
    **
    **  Fetches the contents of a CABAC pair to the registers.
    **
    **  @param  pair  The number of the pair to be fetched
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void fetch(int pair) throws RcmException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        checkPair(pair);
        write(REG_CABAC + CABAC_OFF_FETCH, pair);
    }


   /**
    ***************************************************************************
    **
    **  Sets one CABAC field value.
    **
    **  @param  cabac  The number of the CABAC
    **
    **  @param  field  The field ID
    **
    **  @param  value  The value to be set
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void set(int cabac, int field, int value) throws RcmException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        int base = getBase(cabac);
        checkField(field);
        int beg = FIELD_STARTS[field];
        int end = beg + FIELD_LENGTHS[field];
        int mask = (1 << (end - beg)) - 1;
        int regBeg = beg >> 5;
        int regEnd = (end - 1) >> 5, bitEnd = (32 - end) & 0x1f;
        if (regEnd == regBeg) {
            updateReg(base, regBeg, mask << bitEnd, value << bitEnd);
        }
        else {
            updateReg(base, regBeg, mask >> (32 - bitEnd),
                      value >> (32 - bitEnd));
            updateReg(base, regEnd, mask << bitEnd, value << bitEnd);
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets all CABAC field values.
    **
    **  @param  cabac  The number of the CABAC
    **
    **  @param  value  The array of values to be set
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void set(int cabac, int[] value) throws RcmException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        int base = getBase(cabac) + CABAC_OFF_WRITE;
        int[] regs = new int[NUM_CABAC_REGS];
        for (int j = 0; j < regs.length; j++) {
            regs[j] = 0;
        }
        for (int j = 0; j < NUM_CABAC_FLDS; j++) {
            int beg = FIELD_STARTS[j];
            int end = beg + FIELD_LENGTHS[j];
            int val = value[j] & ((1 << (end - beg)) - 1);
            int regBeg = beg >> 5;
            int regEnd = (end - 1) >> 5, bitEnd = (32 - end) & 0x1f;
            if (regEnd == regBeg) {
                regs[regBeg] |= val << bitEnd;
            }
            else {
                regs[regBeg] |= val >> (32 - bitEnd);
                regs[regEnd] |= val << bitEnd;
            }
        }
        write(base, regs);
    }


   /**
    ***************************************************************************
    **
    **  Sets several CABAC field values.
    **
    **  @param  cabac  The number of the CABAC
    **
    **  @param  first  The ID of the first field
    **
    **  @param  count  The number of fields to set
    **
    **  @param  value  The array of values to be set
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void set(int cabac, int first, int count, int[] value)
        throws RcmException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        int base = getBase(cabac);
        checkField(first, count);
        int[] regs = new int[NUM_CABAC_REGS], masks = new int[NUM_CABAC_REGS];
        for (int j = 0; j < regs.length; j++) {
            regs[j] = 0;
            masks[j] = 0;
        }
        for (int j = 0; j < count; j++) {
            int beg = FIELD_STARTS[first + j];
            int end = beg + FIELD_LENGTHS[first + j];
            int mask = (1 << (end - beg)) - 1;
            int val = value[j] & mask;
            int regBeg = beg >> 5;
            int regEnd = (end - 1) >> 5, bitEnd = (32 - end) & 0x1f;
            if (regEnd == regBeg) {
                regs[regBeg] |= val << bitEnd;
                masks[regBeg] |= mask << bitEnd;
            }
            else {
                regs[regBeg] |= val >> (32 - bitEnd);
                regs[regEnd] |= val << bitEnd;
                masks[regBeg] |= mask >> (32 - bitEnd);
                masks[regEnd] |= mask << bitEnd;
            }
        }
        for (int j = 0; j < regs.length; j++) {
            if (masks[j] == 0) continue;
            if (masks[j] == -1) {
                write(base + CABAC_OFF_WRITE + j, regs[j]);
            }
            else {
                updateReg(base, j, masks[j], regs[j]);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets one CABAC field value.
    **
    **  @param  cabac  The number of the CABAC
    **
    **  @param  field  The field ID
    **
    **  @return  The value of the field
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public int get(int cabac, int field) throws RcmException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        int base = getBase(cabac) + CABAC_OFF_READ;
        checkField(field);
        int beg = FIELD_STARTS[field];
        int end = beg + FIELD_LENGTHS[field];
        int mask = (1 << (end - beg)) - 1;
        int regBeg = beg >> 5;
        int regEnd = (end - 1) >> 5, bitEnd = (32 - end) & 0x1f;
        if (regEnd == regBeg) {
            return (read(base + regBeg) >> bitEnd) & mask;
        }
        else {
            return ((read(base + regBeg) << (32 - bitEnd))
                      | (read(base + regEnd) >>> bitEnd)) & mask;
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets all CABAC field values.
    **
    **  @param  cabac  The number of the CABAC
    **
    **  @return  The array of field values
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public int[] get(int cabac) throws RcmException
    {
        return get(cabac, 0, NUM_CABAC_FLDS);
    }


   /**
    ***************************************************************************
    **
    **  Gets several CABAC field values.
    **
    **  @param  cabac  The number of the CABAC
    **
    **  @param  field  The ID of the first field
    **
    **  @param  count  The number of fields to get
    **
    **  @return  The array of field values
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public int[] get(int cabac, int first, int count)
        throws RcmException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        int base = getBase(cabac) + CABAC_OFF_READ;
        checkField(first, count);
        int[] regs = new int[NUM_CABAC_REGS], value = new int[count];
        read(base, regs);
        for (int j = 0; j < count; j++) {
            int beg = FIELD_STARTS[first + j];
            int end = beg + FIELD_LENGTHS[first + j];
            int mask = (1 << (end - beg)) - 1;
            int regBeg = beg >> 5;
            int regEnd = (end - 1) >> 5, bitEnd = (32 - end) & 0x1f;
            if (regEnd == regBeg) {
                value[j] = regs[regBeg] >> bitEnd;
            }
            else {
                value[j] = ((regs[regBeg] << (32 - bitEnd))
                              | (regs[regEnd] >>> bitEnd)) & mask;
            }
        }

        return value;
    }


   /**
    ***************************************************************************
    **
    **  Reads all CABAC input registers in raw format.
    **
    **  @param  cabac  The number of the CABAC
    **
    **  @return  The array of values read from the registers
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public int[] readRaw(int cabac) throws RcmException
    {
        checkVersion(VERSION_2, VERSION_CURR);
        int base = getBase(cabac) + CABAC_OFF_READ;
        int[] regs = new int[NUM_CABAC_REGS];
        read(base, regs);

        return regs;
    }


   /**
    ***************************************************************************
    **
    **  Checks a CABAC pair value for validity.
    **
    **  @param  pair  The number of the pair
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private void checkPair(int pair) throws RcmException
    {
        if (pair < 0 || pair >= NUM_CABAC_PAIRS) {
            throw new RcmException("Invalid CABAC pair number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the base register address for the CABAC.
    **
    **  @param  cabac  The number of the CABAC
    **
    **  @return  The CABAC base address
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private int getBase(int cabac) throws RcmException
    {
        if (cabac != CABAC_ID_BOTTOM && cabac != CABAC_ID_TOP) {
            throw new RcmException("Invalid CABAC ID");
        }

        return REG_CABAC + (cabac == CABAC_ID_BOTTOM ? CABAC_OFF_BOTTOM
                                                     : CABAC_OFF_TOP);
    }


   /**
    ***************************************************************************
    **
    **  Checks a CABAC field ID for validity.
    **
    **  @param  field  The field ID
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private void checkField(int field) throws RcmException
    {
        if (field < 0 || field >= NUM_CABAC_FLDS) {
            throw new RcmException("Invalid CABAC field number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a range of CABAC field IDs for validity.
    **
    **  @param  first  The ID of the first field
    **
    **  @param  count  The number of fields
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private void checkField(int first, int count) throws RcmException
    {
        if (first < 0 || count < 0 || first + count > NUM_CABAC_FLDS) {
            throw new RcmException("Invalid CABAC field number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Updates a CABAC set register using its corresponding get register.
    **
    **  @param  base   The CABAC base address
    **
    **  @param  reg    The register number
    **
    **  @param  mask   A mask specifying which bits of value are to be used
    **
    **  @param  value  The value to use
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private void updateReg(int base, int reg, int mask, int value)
        throws RcmException
    {
        write(base + CABAC_OFF_WRITE + reg,
              (read(base + CABAC_OFF_READ + reg) & ~mask) | (value & mask));
    }

}
