package org.lsst.ccs.drivers.rcm;

/**
 ***************************************************************************
 **
 **  Routines for setting the on-board DACs.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class BoardDacs extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    public final static int
        REG_BOARD_DACS   = 0x400000,
        REG_BOARD_DACS_0 = 0x500001,
        DACS_SET         = 0,
        DACS_LOAD        = 1,
        DACS_INCR        = 0x10,
        DAC_CLOCK_RAILS  = 0,
        DAC_CS_GATE      = 1,
        NUM_BOARD_DACS   = 2,
        NUM_BOARD_DACS_0 = 1,
        CHAN_SCLK_L      = 0,
        CHAN_SCLK_H      = 1,
        CHAN_RG_L        = 2,
        CHAN_RG_H        = 3,
        CHAN_PCLK_L      = 4,
        CHAN_PCLK_H      = 5,
        CHAN_HEATER_1    = 6,
        CHAN_HEATER_2    = 7,
        CHAN_CSGATE_1    = 0,
        CHAN_CSGATE_2    = 1,
        CHAN_CSGATE_3    = 2,
        NUM_CLOCKS       = 6,
        NUM_HEATERS      = 2,
        NUM_CSGATES      = 3,
        NUM_DAC_CHANNELS = 8;
    

   /**
    ***************************************************************************
    **
    **  Constructors.
    **
    ***************************************************************************
    */
    public BoardDacs()
    {
        super();
    }

    public BoardDacs(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Loads the DAC values.
    **
    **  All the values previously set in the DAC are loaded into its output
    **  registers
    **
    **  @param  dac  The DAC number
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void load(int dac) throws RcmException
    {
        int base = getBase(dac);
        if (getVersion() == VERSION_0) return;
        write(base + DACS_LOAD, 0);
    }


   /**
    ***************************************************************************
    **
    **  Sets a DAC value.
    **
    **  The supplied value is set into the DAC, but not loaded into the DAC's
    **  output register until the load method is called
    **
    **  @param  dac    The DAC number
    **
    **  @param  chan   The channel number within the DAC
    **
    **  @param  value  The value to be set
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void set(int dac, int chan, int value) throws RcmException
    {
        int base = getBase(dac);
        checkChannel(chan);
        write(base + DACS_SET, (chan << 12) | (value & 0x0fff));
    }


   /**
    ***************************************************************************
    **
    **  Sets multiple DAC values.
    **
    **  The supplied values are set into the DAC, but not loaded into the DAC's
    **  output registers until the load method is called
    **
    **  @param  dac    The DAC number
    **
    **  @param  chan   The first channel number to set within the DAC
    **
    **  @param  value  An array containing the values to be set
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void set(int dac, int chan, int[] value) throws RcmException
    {
        int base = getBase(dac);
        checkChannel(chan, value.length);
        for (int j = 0; j < value.length; j++) {
            write(base + DACS_SET, ((chan + j) << 12) | (value[j] & 0x0fff));
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the DAC base register address.
    **
    **  @param  dac  The DAC number
    **
    **  @return  The base address of the DAC
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private int getBase(int dac) throws RcmException
    {
        int base, nDacs;
        if (getVersion() == VERSION_0) {
            base = REG_BOARD_DACS_0;
            nDacs = NUM_BOARD_DACS_0;
        }
        else {
            base = REG_BOARD_DACS;
            nDacs = NUM_BOARD_DACS;
        }
        if (dac < 0 || dac >= nDacs) {
            throw new RcmException("Invalid DAC number");
        }

        return base + dac * DACS_INCR;
    }


   /**
    ***************************************************************************
    **
    **  Checks a DAC channel number for validity.
    **
    **  @param  chan  The channel number
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private void checkChannel(int chan) throws RcmException
    {
        if (chan < 0 || chan >= NUM_DAC_CHANNELS) {
            throw new RcmException("Invalid DAC channel number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a DAC channel number range for validity.
    **
    **  @param  chan   The first channel number
    **
    **  @param  count  The number of channels
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private void checkChannel(int chan, int count) throws RcmException
    {
        if (chan < 0 || count <= 0 || chan + count > NUM_DAC_CHANNELS) {
            throw new RcmException("Invalid DAC channel number");
        }
    }

}
