package org.lsst.ccs.drivers.rcm;

import java.io.IOException;
import java.io.PrintStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import jline.console.ConsoleReader;
import org.lsst.ccs.utilities.sa.CmndProcess;

/**
 ***************************************************************************
 **
 **  Program to test the Java image client interface routines
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestImage implements CmndProcess.Dispatch, ImageClient.Listener {

    private final static int
        CMD_CONNECT     = 0,
        CMD_DISPLAY     = 1,
        CMD_REGION      = 2,
        CMD_LISTEN      = 3,
        CMD_WAIT        = 4,
        CMD_READ        = 5,
        CMD_METADATA    = 6,
        CMD_DISCONN     = 7,
        NUM_CMDS        = 8;

    private final static String[] helpConnect = {
        "Connect to an RCM",
        "connect <id> [<ifc>]",
        "id   The id of the RCM",
        "ifc  The name of the network interface to use",
    };

    private final static String[] helpDisconn = {
        "Disconnect from an RCM",
        "disconnect",
    };

    private final static String[] helpListen = {
        "Enable or disable the image listener",
        "listen <action> [<newbuff>] [<time>]",
        "action   The action to perform: 'on' to enable; 'off' to disable",
        "newbuff  If present (any value), use a new image buffer each time",
        "time     If present (any value), show image acquisition times",
    };

    private final static String[] helpWait = {
        "Wait for an image to arrive",
        "wait [<newbuff>] [<time>]",
        "newbuff  If present (any value), allocate a new image buffer",
        "time     If present (any value), show image acquisition time",
    };

    private final static String[] helpRead = {
        "Read image data",
        "read [<time>]",
        "time  If present (any value), show image read time",
    };

    private final static String[] helpDisplay = {
        "Display selected data from the last image received",
        "display <count> [<offset>]",
        "count   The number of pixels to display",
        "offset  The offset of the first pixel to display (default 0)",
    };

    private final static String[] helpMetadata = {
        "Display the metadata from the last image received",
        "metadata",
    };

    private final static String[] helpRegion = {
        "Set the region to be displayed automatically from each received image",
        "region [<count>] [<offset>]",
        "count   The number of pixels to display (initially 0)",
        "offset  The offset of the first pixel to display (initially 0)",
    };

    private final static CmndProcess.Lookup onoffNames;
    static {
        onoffNames = new CmndProcess.Lookup(2);
        onoffNames.add("on",  1);
        onoffNames.add("off", 0);
    }

    private final static CmndProcess.Command cmnd;
    static {
        cmnd = new CmndProcess.Command(NUM_CMDS);
        cmnd.add("connect",  CMD_CONNECT,  helpConnect,  "Is");
        cmnd.add("disconn",  CMD_DISCONN,  helpDisconn,  "");
        cmnd.add("listen",   CMD_LISTEN,   helpListen,   "Kss",  onoffNames);
        cmnd.add("wait",     CMD_WAIT,     helpWait,     "ss");
        cmnd.add("read",     CMD_READ,     helpRead,     "s");
        cmnd.add("display",  CMD_DISPLAY,  helpDisplay,  "Ii");
        cmnd.add("metadata", CMD_METADATA, helpMetadata, "");
        cmnd.add("region",   CMD_REGION,   helpRegion,   "ii");
    }

    private final static PrintStream out = System.out;
    private final CmndProcess proc = new CmndProcess();
    private final ImageClient imc = new ImageClient();
    private final RegClient reg = new RegClient();
    private final BaseSet bset = new BaseSet(reg);
    private int rcmId = 0, imgLength = -1, imgOffset = 0;
    private boolean imgOkay, listening = false, listenTime = false;
    private Image image = new Image();


/**
 ***************************************************************************
 **
 **  Main program.
 **
 ***************************************************************************
 */
    public static void main(String[] args)
    {
        (new TestImage()).run();
        System.exit(0);
    }


/**
 ***************************************************************************
 **
 **  Runs the test.
 **
 **  Loops reading and processing each new typed command line.
 **
 ***************************************************************************
 */
    public void run()
    {
        proc.add(this, cmnd);

        try {
            ConsoleReader reader = new ConsoleReader();
            while (true) {
                String line = reader.readLine(">> ");
                if (line == null || !proc.process(line)) break;
            }
        }
        catch (IOException e) {
            out.println(e);
        }
    }


/**
 ***************************************************************************
 **
 **  Dispatches a command for processing.
 **
 **  @param  code   The command function code
 **
 **  @param  found  A bit mask indicating which command arguments are
 **                 present
 **
 **  @param  args   The array of command arguments
 **
 ***************************************************************************
 */
    @Override
    public boolean dispatch(int code, int found, Object[] args)
    {
        try {
            switch (code) {
            case CMD_CONNECT:
                procConnect(found, args); break;
            case CMD_DISCONN:
                procDisconn(found, args); break;
            case CMD_LISTEN:
                procListen(found, args); break;
            case CMD_WAIT:
                procWait(found, args); break;
            case CMD_READ:
                procRead(found, args); break;
            case CMD_DISPLAY:
                procDisplay(found, args); break;
            case CMD_METADATA:
                procMetadata(found, args); break;
            case CMD_REGION:
                procRegion(found, args); break;
            default:
                out.println("Command not fully implemented");
            }
        }
        catch (RcmException e) {
            out.println(e);
        }

        return true;
    }


/**
 ***************************************************************************
 **
 **  Processes the CONNECT command.
 **
 ***************************************************************************
 */
    private void procConnect(int found, Object[] args) throws RcmException
    {
        rcmId = (Integer)args[0];
        if ((found & 0x02) == 0) {
            imc.open(rcmId);
            reg.open(rcmId);
        }
        else {
            String ifc = (String)args[1];
            imc.open(rcmId, ifc);
            reg.open(rcmId, ifc);
        }
    }


/**
 ***************************************************************************
 **
 **  Processes the DISCONNECT command.
 **
 ***************************************************************************
 */
    private void procDisconn(int found, Object[] args) throws RcmException
    {
        imc.close();
        reg.close();
    }


/**
 ***************************************************************************
 **
 **  Processes the LISTEN command.
 **
 ***************************************************************************
 */
    private void procListen(int found, Object[] args)
    {
        if (!checkConnect()) return;
        if ((Integer)args[0] != 0) {
            listening = true;
            listenTime = (found & 0x04) != 0;
            imc.setListener(this, (found & 0x02) != 0 ? null : image);
        }
        else {
            listening = false;
            imc.clearListener();
        }
    }


/**
 ***************************************************************************
 **
 **  Processes the WAIT command.
 **
 ***************************************************************************
 */
    private void procWait(int found, Object[] args) throws RcmException
    {
        if (!checkConnect() || !checkListen()) return;
        image = imc.awaitImage((found & 0x01) != 0 ? null : image);
        if ((found & 0x02) != 0) {
            long currTime = bset.getTime();
            long trigTime = bset.getTriggerTime(BaseSet.RSET_SEQUENCER);
            out.format("Elapsed time = %s msec\n", currTime - trigTime);
        }
        displayMetadata(image, "Got metadata", false);
    }


/**
 ***************************************************************************
 **
 **  Processes the READ command.
 **
 ***************************************************************************
 */
    private void procRead(int found, Object[] args) throws RcmException
    {
        if (!checkConnect() || !checkListen()) return;
        long start = System.currentTimeMillis();
        imgOkay = imc.readImage(image);
        if (found != 0) {
            out.format("Elapsed time = %s msec\n",
                       System.currentTimeMillis() - start);
        }
        if (!imgOkay) {
            out.println("Error reading image data");
        }
    }


/**
 ***************************************************************************
 **
 **  Processes the DISPLAY command.
 **
 ***************************************************************************
 */
    private void procDisplay(int found, Object[] args)
    {
        int offset = ((found & 0x02) != 0) ? (Integer)args[1] : 0;
        displayImage(image, offset, (Integer)args[0]);
    }


/**
 ***************************************************************************
 **
 **  Processes the METADATA command.
 **
 ***************************************************************************
 */
    private void procMetadata(int found, Object[] args)
    {
        displayMetadata(image, "Metadata", true);
    }


/**
 ***************************************************************************
 **
 **  Processes the REGION command.
 **
 ***************************************************************************
 */
    private void procRegion(int found, Object[] args)
    {
        if (found == 0) {
            out.format("Display region: length = %s, offset = %s\n",
                       imgLength, imgOffset);
        }
        else {
            if ((found & 0x01) != 0) {
                imgLength = (Integer)args[0];
            }
            if ((found & 0x02) != 0) {
                imgOffset = (Integer)args[1];
            }
        }
    }


/**
 ***************************************************************************
 **
 **  Handles each received image.
 **
 **  @param  img   The received image
 **
 ***************************************************************************
 */
    @Override
    public void processImage(Image img)
    {
        image = img;
        if (listenTime) {
            try {
                long currTime = bset.getTime();
                long trigTime = bset.getTriggerTime(BaseSet.RSET_SEQUENCER);
                out.format("Elapsed time = %s msec\n", currTime - trigTime);
            }
            catch (RcmException e) {            
            }
        }
        displayMetadata(image, "Got image", true);
        if (image.getData() != null) {
            displayImage(image, imgOffset, imgLength);
        }
    }


/**
 ***************************************************************************
 **
 **  Displays the metadata for an image.
 **
 **  @param  img        The image whose metadata is to be displayed
 **
 **  @param  caption    A caption to be displayed before the metadata
 **
 **  @param  showValid  If true, display whether or not the image contains
 **                     valid data
 **
 ***************************************************************************
 */
    private void displayMetadata(Image img, String caption, boolean showValid)
    {
        StringBuilder blanks = new StringBuilder(caption.length());
        for (int j = 0; j < caption.length(); j++) {
            blanks.append(" ");
        }
        out.format("%s: tag = %016x, length = %s, schema = %08x,\n"
                   + "%s  version = %08x, address = %s, cluster = %s, "
                   + "element = %s%s\n",
                   caption, image.getTag(), image.getLength(),
                   image.getSchema(), blanks, image.getVersion(),
                   image.getAddress(), image.getCluster(), image.getElement(),
                   showValid ? image.getData() != null ? ", valid"
                             : ", invalid" : "");
    }


/**
 ***************************************************************************
 **
 **  Displays the contents of an image.
 **
 **  @param  img     The image to be displayed
 **
 **  @param  offset  The offset to the first pixel to display
 **
 **  @param  count   The number of pixels to display.  If zero, display up
 **                  to the stored length.  If negative, display nothing.
 **
 ***************************************************************************
 */
    private void displayImage(Image img, int offset, int count)
    {
        if (count < 0) return;
        ByteBuffer buff = img.getData();
        if (buff == null) {
            out.println("Image contains no valid data");
            return;
        }

        offset = (offset < 0) ? 0 : offset;
        int iCount = img.getLength() / 4;
        if (count == 0 || count + offset > iCount) {
            count = iCount - offset;
        }
        if (count <= 0) return;

        buff.order(ByteOrder.LITTLE_ENDIAN);
        buff.position(4 * offset);
        for (int j = 0; j < count; j++) {
            if ((j & 0x03) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format("%08x:", offset + j);
            }
            out.format(" %08x", buff.getInt());
        }
        out.println();
    }


/**
 ***************************************************************************
 **
 **  Checks that an RCM is connected.
 **
 ***************************************************************************
 */
    private boolean checkConnect()
    {
        boolean okay = (imc != null);
        if (!okay) {
            out.println("No RCM connected");
        }
        return okay;
    }


/**
 ***************************************************************************
 **
 **  Checks that the listener is not active.
 **
 ***************************************************************************
 */
    private boolean checkListen()
    {
        boolean okay = !listening;
        if (!okay) {
            out.println("Listener is active");
        }
        return okay;
    }

}
