package org.lsst.ccs.drivers.rcm;

/**
 ***************************************************************************
 **
 **  Temperature ADCs reading routines.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TempAdcs extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    public final static int
        REG_TEMP_ADCS   = 0x600010,
        ADC_DREB_T1     = 0,
        ADC_DREB_T2     = 1,
        ADC_REB_T1      = 2,
        ADC_REB_T2      = 3,
        ADC_REB_T3      = 4,
        ADC_REB_T4      = 5,
        ADC_REB_T5      = 6,
        ADC_REB_T6      = 7,
        ADC_REB_T7      = 8,
        ADC_REB_T8      = 9,
        ADC_REB_T9      = 10,
        NUM_TEMP_REGS   = 11;

    public final static float
        TEMP_SCALE      = 1 / 128F;
 

   /**
    ***************************************************************************
    **
    **  Constructors.
    **
    ***************************************************************************
    */
    public TempAdcs()
    {
        super();
    }

    public TempAdcs(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Enables the temperature ADC reading.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    @Override
    public void enable() throws RcmException
    {
        checkNotVersion(VERSION_0);
        enable(RSET_TEMP_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Waits for the data to be available.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void waitDone() throws RcmException
    {
        checkNotVersion(VERSION_0);
        waitDone(RSET_TEMP_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Gets the time of the read enable.
    **
    **  @return  The Unix millisecond time of the read enable
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public long getTriggerTime() throws RcmException
    {
        checkVersion(VERSION_2);
        return getTriggerTime(RSET_TEMP_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Reads one temperature value.
    **
    **  @param  adc  The number of the ADC to read
    **
    **  @return  The ADC value (Celsius)
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public float readAdc(int adc) throws RcmException
    {
        if (adc < 0 || adc >= NUM_TEMP_REGS) {
            throw new RcmException("Invalid temperature ADC number");
        }
        enable();
        waitDone();

        return TEMP_SCALE * read(REG_TEMP_ADCS + adc);
    }


   /**
    ***************************************************************************
    **
    **  Reads all the temperature values.
    **
    **  @return  The array of ADC values (Celsius)
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public float[] readAdcs() throws RcmException
    {
        return readAdcs(0, NUM_TEMP_REGS);
    }


   /**
    ***************************************************************************
    **
    **  Reads a range of temperature values.
    **
    **  @param  first  The number of the first ADC to read
    **
    **  @param  count  The number of ADCs to read
    **
    **  @return  The array of ADC values (Celsius)
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public float[] readAdcs(int first, int count) throws RcmException
    {
        if (first < 0 || count < 0 || first + count > NUM_TEMP_REGS) {
            throw new RcmException("Invalid temperature ADC range");
        }
        enable();
        waitDone();
        int[] rawData = new int[count];
        float[] data = new float[count];
        read(REG_TEMP_ADCS + first, rawData);
        for (int j = 0; j < count; j++) {
            data[j] = TEMP_SCALE * rawData[j];
        }

        return data;
    }

}
