package org.lsst.ccs.drivers.rcm;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Scanner;

/**
 ***************************************************************************
 **
 **  Sequencer utility routines.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class SequencerUtils extends Sequencer {

   /**
    ***************************************************************************
    **
    **  Private constants
    **
    ***************************************************************************
    */
    private final static int 
        SEQTYPE_REGISTER = 0,
        SEQTYPE_OUTPUT   = 1,
        SEQTYPE_TIMES    = 2,
        SEQTYPE_PROGRAM  = 3,
        SEQTYPE_PROGEXEC = 4,
        SEQTYPE_PROGJUMP = 5,
        SEQTYPE_PROGSUBE = 6,
        SEQTYPE_PROGEND  = 7,
        SEQTYPE_SLICE    = 8,
        SEQTYPE_SOURCE   = 9,
        SEQTYPE_STACK    = 10,
        SEQTYPE_BEB      = 11,
        MAX_SUBR_LEVEL   = 16;

    private final static HashMap<String, Integer> seqTypes;
    static {
        seqTypes = new HashMap<String,Integer>();
        seqTypes.put("register", SEQTYPE_REGISTER);
        seqTypes.put("output",   SEQTYPE_OUTPUT);
        seqTypes.put("times",    SEQTYPE_TIMES);
        seqTypes.put("program",  SEQTYPE_PROGRAM);
        seqTypes.put("progexec", SEQTYPE_PROGEXEC);
        seqTypes.put("progjump", SEQTYPE_PROGJUMP);
        seqTypes.put("progsube", SEQTYPE_PROGSUBE);
        seqTypes.put("progend",  SEQTYPE_PROGEND);
        seqTypes.put("slice",    SEQTYPE_SLICE);
        seqTypes.put("source",   SEQTYPE_SOURCE);
        seqTypes.put("stack",    SEQTYPE_STACK);
        seqTypes.put("beb",      SEQTYPE_BEB);
    }

    private final static HashMap<Integer, Integer> seqMinArgs;
    static {
        seqMinArgs = new HashMap<Integer,Integer>();
        seqMinArgs.put(SEQTYPE_REGISTER, 3);
        seqMinArgs.put(SEQTYPE_OUTPUT,   3);
        seqMinArgs.put(SEQTYPE_TIMES,    3);
        seqMinArgs.put(SEQTYPE_PROGRAM,  3);
        seqMinArgs.put(SEQTYPE_PROGEXEC, 4);
        seqMinArgs.put(SEQTYPE_PROGJUMP, 4);
        seqMinArgs.put(SEQTYPE_PROGSUBE, 2);
        seqMinArgs.put(SEQTYPE_PROGEND,  2);
        seqMinArgs.put(SEQTYPE_SLICE,    2);
        seqMinArgs.put(SEQTYPE_SOURCE,   2);
        seqMinArgs.put(SEQTYPE_STACK,    3);
        seqMinArgs.put(SEQTYPE_BEB,      2);
    }

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private int stkSize;


   /**
    ***************************************************************************
    **
    **  Constructors.
    **
    ***************************************************************************
    */
    public SequencerUtils()
    {
        super();
    }

    public SequencerUtils(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Carries out a sequencer load command.
    **
    **  @param  cmnd  A sequencer command
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void loadCommand(String cmnd) throws RcmException
    {
        loadCommand(cmnd, 0);
    }


   /**
    ***************************************************************************
    **
    **  Carries out a sequencer load command.
    **
    **  @param  cmnd     A sequencer command
    **
    **  @param  lineNum  The line number associated with the command, or zero
    **                   if there is no line number
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void loadCommand(String cmnd, int lineNum) throws RcmException
    {
        boolean thrown = false;
        ArrayList<Integer> args = new ArrayList<Integer>();
        String word = "";

        try {
            Scanner scan = new Scanner(cmnd);
            int type = -1, count = 0;
            args.clear();
            while (scan.hasNext()) {
                word = scan.next();
                if (count == 0) {
                    if (word.startsWith("#")) break;
                    Integer iType = seqTypes.get(word);
                    if (iType == null) {
                        thrown = true;
                        throwException("Invalid operation (" + word + ")",
                                       lineNum);
                    }
                    type = iType;
                }
                else {
                    args.add((int)(long)Long.decode(word));
                }
                count++;
            }
            if (count == 0) return;
            int minArgs = seqMinArgs.get(type);
            if (count < minArgs) {
                thrown = true;
                throwException("Too few arguments (" + (count - 1) + ")",
                               lineNum);
            }
            if (minArgs != 3 && count > minArgs) {
                thrown = true;
                throwException("Too many arguments (" + (count - 1) + ")",
                               lineNum);
            }
            int arg0 = args.get(0);
            if (minArgs == 2) {
                if (type == SEQTYPE_PROGSUBE) {
                    writeProgEndSubr(arg0);
                }
                else if (type == SEQTYPE_PROGEND) {
                    writeProgEnd(arg0);
                }
                else if (type == SEQTYPE_SLICE) {
                    writeSliceCount(arg0);
                }
                else if (type == SEQTYPE_SOURCE) {
                    writeDataSource(arg0);
                }
                else if (type == SEQTYPE_BEB) {
                    writeBebSelect(arg0);
                }
            }
            else {
                int[] values = new int[count - 2];
                for (int j = 0; j < values.length; j++) {
                    values[j] = args.get(j + 1);
                }
                if (type == SEQTYPE_REGISTER) {
                    write(arg0, values);
                }
                else if (type == SEQTYPE_OUTPUT) {
                    writeLines(arg0, values);
                }
                else if (type == SEQTYPE_TIMES) {
                    writeTimes(arg0, values);
                }
                else if (type == SEQTYPE_PROGRAM) {
                    writeProgram(arg0, values);
                }
                else if (type == SEQTYPE_PROGEXEC) {
                    writeProgExec(arg0, values[0], values[1]);
                }
                else if (type == SEQTYPE_PROGJUMP) {
                    writeProgJump(arg0, values[0], values[1]);
                }
                else if (type == SEQTYPE_STACK) {
                    writeStack(arg0, values);
                    if (arg0 + values.length > stkSize) {
                        stkSize = arg0 + values.length;
                        writeStackSize(stkSize);
                    }
                }
            }
        }
        catch (NumberFormatException e) {
            throwException("Invalid integer (" + word + ")", lineNum);
        }
        catch (RcmException e) {
            if (thrown) {
                throw e;
            }
            else {
                throwException(e.getMessage(), lineNum);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Loads the sequencer from a file.
    **
    **  @param  fileName  The name of the file containing sequencer commands
    **
    **  @return  The number of slices generated by the sequencer program
    **
    **  @exception  RcmException 
    **  @exception  IOException 
    **
    ***************************************************************************
    */
    public int loadFile(String fileName) throws RcmException, IOException
    {
        BufferedReader rdr = null;
        int lineNum = 0;
        stkSize = 0;
        clearCache();

        try {
            rdr = new BufferedReader(new FileReader(fileName));
            while (true) {
                String line = rdr.readLine();
                if (line == null) break;
                lineNum++;
                loadCommand(line, lineNum);
            }
        }
        finally {
            if (rdr != null) {
                try {
                    rdr.close();
                }
                catch (IOException e) {
                }
            }
        }

        return getCacheSliceCount();
    }


   /**
    ***************************************************************************
    **
    **  Clears the sequencer program cached copy.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void clearCache() throws RcmException
    {
        for (int j = 0; j < lines.length; j++) {
            for (int k = 0; k < lines[j].length; k++) {
                lines[j][k] = 0;
                times[j][k] = 0;
            }
        }
        if (getVersion() == VERSION_0) {
            for (int j = 0; j < stack.length; j++) {
                stack[j] = 0;
            }
            stackSize = 0;
        }
        else {
            for (int j = 0; j < prog.length; j++) {
                prog[j] = 0;
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the slice count of the sequencer program cached copy.
    **
    **  @return  The number of slices generated by the whole program
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public int getCacheSliceCount() throws RcmException
    {
        int maxFunc = getVersion() == VERSION_0 ? SEQ_MAX_FUNC_0 : SEQ_MAX_FUNC;
        int[] funcSlice = new int[maxFunc];
        funcSlice[0] = 0;
        for (int j = 1; j < maxFunc; j++) {
            int nSlice = 0;
            boolean prevAdc = false;
            for (int k = 0; k < SEQ_MAX_SLICE && times[j][k] != 0; k++) {
                boolean adc = (lines[j][k] & 0x1000) != 0;
                if (adc & !prevAdc) {
                    nSlice++;
                }
                prevAdc = adc;
            }
            funcSlice[j] = nSlice;
        }
        if (getVersion() == VERSION_0) {
            int nSlice = 0;
            for (int j = 0; j < stackSize; j++) {
                int func = (stack[j] >> SEQ_STK_V_FUNC) & SEQ_STK_M_FUNC;
                int count = stack[j] & SEQ_STK_M_COUNT;
                if (funcSlice[func] != 0
                      && (stack[j] & (1 << SEQ_STK_V_LOOP)) != 0) {
                    throw new RcmException("Data-producing infinite loop");
                }
                nSlice += count * funcSlice[func];
            }
            return nSlice;
        }
        else {
            return getProgSlices(0, 0, funcSlice);
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of slices generated by a sequencer program routine.
    **
    **  @param  addr       The program address
    **
    **  @param  level      The subroutine nesting level
    **
    **  @param  funcSlice  The array of slice counts for each defined function
    **
    **  @return  The number of slices generated by the routine
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private int getProgSlices(int addr, int level, int[] funcSlice)
        throws RcmException
    {
        int nSlice = 0;

        for (; addr < SEQ_MAX_PROGRAM; addr++) {
            int cmnd = prog[addr];

            switch (cmnd >>> SEQ_PRG_V_OPCODE) {

            case SEQ_OPC_EXECUTE:
                int func = (cmnd >> SEQ_PRG_V_FUNC) & SEQ_PRG_M_FUNC;
                if (funcSlice[func] != 0
                      && (cmnd & (1 << SEQ_PRG_V_LOOP)) != 0) {
                    throw new RcmException("Data-producing infinite loop");
                }
                nSlice += (cmnd & SEQ_PRG_M_EXCCNT) * funcSlice[func];
                break;

            case SEQ_OPC_JUMP:
                if (level >= MAX_SUBR_LEVEL) {
                    throw new RcmException("Maximum subroutine level exceeded");
                }
                int sAddr = (cmnd >> SEQ_PRG_V_SUBADD) & SEQ_PRG_M_SUBADD;
                int count = cmnd & SEQ_PRG_M_SUBCNT;
                int slice = getProgSlices(sAddr, level + 1, funcSlice);
                nSlice += (slice >= 0 ? count * slice : -1 ^ slice);
                if (slice < 0) {
                    return nSlice ^ (level == 0 ? 0 : -1);
                }
                break;

            case SEQ_OPC_END_SUBR:
                if (level == 0) {
                    throw new RcmException("Subroutine trailer at top level");
                }
                return nSlice;

            case SEQ_OPC_END_PROG:
                return nSlice ^ (level == 0 ? 0 : -1);

            default:
                throw new RcmException("Invalid sequencer opcode ("
                                         + (cmnd >>> SEQ_PRG_V_OPCODE) + ")");

            }
        }

        if (level == 0) {
            throw new RcmException("Missing end of program");
        }
        else {
            throw new RcmException("Missing subroutine trailer");
        }
    }


   /**
    ***************************************************************************
    **
    **  Throws a parsing error exception.
    **
    **  @param  text     The text message for the exception
    **
    **  @param  lineNum  The line number where the error occurred, or zero if
    **                   no line number is available
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private void throwException(String text, int lineNum) throws RcmException
    {
        if (lineNum > 0) {
            throw new RcmException(text + " at line " + lineNum);
        }
        else {
            throw new RcmException(text);
        }
    }

}
