package org.lsst.ccs.drivers.rcm;

/**
 ***************************************************************************
 **
 **  Base register set access routines.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class BaseSet extends RegClient {

   /**
    ***************************************************************************
    **
    **  Public constants.
    **
    ***************************************************************************
    */
    public final static int
        RSET_STATUS     = 0,
        RSET_TIME_BASE  = 1,
        RSET_SEQUENCER  = 2,
        RSET_POWER_ADCS = 3,
        RSET_TEMP_ADCS  = 4,
        REG_SCHEMA      = 0,
        REG_VERSION     = 1,
        REG_ID          = 2,
        REG_TIME_BASE   = 4,
        REG_STATE       = 8,
        REG_TRIGGER     = 9,
        REG_TRIG_TIME   = 10,
        VERSION_UNSET   = -2,
        VERSION_UNKNOWN = -1,
        VERSION_0       = 0,
        VERSION_1       = 1,
        VERSION_2       = 2,
        CLOCK_PERIOD_0  = 5,      // Nanoseconds
        CLOCK_PERIOD_1  = 10;     // Nanoseconds
 
   /**
    ***************************************************************************
    **
    **  Private data.
    **
    ***************************************************************************
    */
    private final static int WAIT_TIMEOUT = 1000;
    private int version = VERSION_UNSET;
    private int handleInst;


   /**
    ***************************************************************************
    **
    **  Constructors.
    **
    ***************************************************************************
    */
    public BaseSet()
    {
        super();
    }

    public BaseSet(RegClient reg)
    {
        handle = reg.handle;
    }


   /**
    ***************************************************************************
    **
    **  Gets the firmware version.
    **
    **  @exception  RcmException if the firmware version is not recognized
    **
    ***************************************************************************
    */
    public int getVersion() throws RcmException
    {
        try {
            checkOpen();
        }
        catch (RcmException e) {
            version = VERSION_UNSET;
            throw e;
        }
        if (version == VERSION_UNSET || handleInst != handle.instance) {
            handleInst = handle.instance;
            int schema = read(REG_SCHEMA);
            int fwVersion = read(REG_VERSION);
            if (schema == 0x00f1de00) {
                version = VERSION_0;
            }
            else if (schema == 0x00000001) {
                version = VERSION_1;
            }
            else if (schema == 0x00000002) {
                version = VERSION_2;
            }
            else {
                version = VERSION_UNKNOWN;
                throw new RcmException("Unknown firmware version");
            }
        }

        return version;
    }


   /**
    ***************************************************************************
    **
    **  Checks for a particular firmware version.
    **
    **  @param  version  The version to check against
    **
    **  @exception  RcmException  if the firmware version is not recognized
    **                            or doesn't match the specified version
    **
    ***************************************************************************
    */
    public void checkVersion(int version) throws RcmException
    {
        if (getVersion() != version) {
            throw new RcmException("Incompatible firmware version");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks for not a particular firmware version.
    **
    **  @param  version  The version to check against
    **
    **  @exception  RcmException  if the firmware version is not recognized
    **                            or matches the specified version
    **
    ***************************************************************************
    */
    public void checkNotVersion(int version) throws RcmException
    {
        if (getVersion() == version) {
            throw new RcmException("Incompatible firmware version");
        }
    }


   /**
    ***************************************************************************
    **
    **  Enables the time-base.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void enable() throws RcmException
    {
        enable(RSET_TIME_BASE);
    }


   /**
    ***************************************************************************
    **
    **  Disables the time-base.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void disable() throws RcmException
    {
        disable(RSET_TIME_BASE);
    }


   /**
    ***************************************************************************
    **
    **  Gets the raw time-base value.
    **
    **  @return  The time-base value as clock ticks
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public long getTimeRaw() throws RcmException
    {
        return readLong(REG_TIME_BASE);
    }


   /**
    ***************************************************************************
    **
    **  Gets the time-base value as millisecond Unix time.
    **
    **  @return  The time-base value as millisecond Unix time
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public long getTime() throws RcmException
    {
        return getPeriod() * getTimeRaw() / 1000000;
    }


   /**
    ***************************************************************************
    **
    **  Sets the raw time-base value.
    **
    **  @param  time  The time-base value as clock ticks
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void setTimeRaw(long time) throws RcmException
    {
        writeLong(REG_TIME_BASE, time);
    }


   /**
    ***************************************************************************
    **
    **  Sets the time-base value as millisecond Unix time.
    **
    **  @param  time  The time-base value as millisecond Unix time
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void setTime(long time) throws RcmException
    {
        setTimeRaw(1000000 * time / getPeriod());
    }


   /**
    ***************************************************************************
    **
    **  Sets the time-base value to the current system time.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void setTime() throws RcmException
    {
        setTime(System.currentTimeMillis());
    }


   /**
    ***************************************************************************
    **
    **  Enables a register set.
    **
    **  @param  regSet  The register set identifier
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void enable(int regSet) throws RcmException
    {
        //        update(REG_TRIGGER, 1<< regSet, 1 << regSet);
        //        write(REG_TRIGGER, read(REG_STATE) | (1 << regSet));
        write(REG_TRIGGER,
              (read(REG_STATE) & ((1 << RSET_STATUS) | (1 << RSET_TIME_BASE)))
                | (1 << regSet));
    }


   /**
    ***************************************************************************
    **
    **  Disables a register set.
    **
    **  @param  regSet  The register set identifier
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void disable(int regSet) throws RcmException
    {
        //        update(REG_TRIGGER, 1<< regSet, 0);
        //        write(REG_TRIGGER, read(REG_STATE) & ~(1 << regSet));
        write(REG_TRIGGER,
              (read(REG_STATE) & ((1 << RSET_STATUS) | (1 << RSET_TIME_BASE)))
                & ~(1 << regSet));
    }


   /**
    ***************************************************************************
    **
    **  Waits for a register set to become disabled.
    **
    **  This generally indicates the completion of a read operation
    **
    **  @param  regSet  The register set identifier
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void waitDone(int regSet) throws RcmException
    {
        long limit = System.currentTimeMillis() + WAIT_TIMEOUT;
        boolean busy = true;
        while (busy && System.currentTimeMillis() < limit) {
            busy = (read(REG_STATE) & (1 << regSet)) != 0;
        }
        if (busy) {
            throw new RcmException("Completion wait timeout");
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the raw trigger time for a register set.
    **
    **  @param  regSet  The register set identifier
    **
    **  @return  The trigger time as clock ticks
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public long getTriggerTimeRaw(int regSet) throws RcmException
    {
        return readLong(REG_TRIG_TIME + 2 * regSet);
    }


   /**
    ***************************************************************************
    **
    **  Gets the trigger time for a register set as Unix millisecond time.
    **
    **  @param  regSet  The register set identifier
    **
    **  @return  The trigger time as Unix millisecond time
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public long getTriggerTime(int regSet) throws RcmException
    {
        return getPeriod() * getTriggerTimeRaw(regSet) / 1000000;
    }


   /**
    ***************************************************************************
    **
    **  Gets the clock period.
    **
    **  @return  The board's clock period in 10 ns units
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private int getPeriod() throws RcmException
    {
        return getVersion() == VERSION_0 ? CLOCK_PERIOD_0 : CLOCK_PERIOD_1;
    }

}
