package org.lsst.ccs.drivers.rcm;

import java.io.IOException;
import java.io.PrintStream;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.GregorianCalendar;
import jline.console.ConsoleReader;
import org.lsst.ccs.utilities.sa.CmndProcess;

/**
 ***************************************************************************
 **
 **  Program to test the Java RCM register access routines
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestReg implements CmndProcess.Dispatch {

    /*
    **  Command codes
    */
    private final static int
        CMD_CONNECT     = 0,
        CMD_SHOW        = 1,
        CMD_READ        = 2,
        CMD_WRITE       = 3,
        CMD_UPDATE      = 4,
        CMD_READLONG    = 5,
        CMD_WRITELONG   = 6,
        CMD_ENABLE      = 7,
        CMD_DISABLE     = 8,
        CMD_TRIGTIME    = 9,
        CMD_TIMEBASE    = 10,
        CMD_LOADTEST    = 11,
        CMD_LOADCMND    = 12,
        CMD_LOADFILE    = 13,
        CMD_STATUS      = 14,
        CMD_TIME        = 15,
        CMD_DISCONN     = 16,
        CMD_TEMP        = 17,
        CMD_POWER       = 18,
        CMD_DAC         = 19,
        NUM_CMDS        = 20;

    /*
    **  Help text
    */
    private final static String[] helpConnect = {
        "Connect to an RCM",
        "connect <id>",
        "id    The id of the RCM",
    };

    private final static String[] helpDisconn = {
        "Disconnect from an RCM",
        "disconnect",
    };

    private final static String[] helpShow = {
        "Show the current connection parameters",
        "show",
    };

    private final static String[] helpRead = {
        "Read and display registers from the RCM",
        "read <address> [<count>]",
        "address  The address of the first register to read",
        "count    The number of registers to read (default 1)",
    };

    private final static String[] helpReadlong = {
        "Read and display a register pair from the RCM",
        "read <address>",
        "address  The address of the register pair to read",
    };

    private final static String[] helpWrite = {
        "Write to registers on the RCM",
        "write <address> <value1> [<value2>]... [<value8>]",
        "address  The address of the first register to write",
        "valuen   The value(s) to write",
    };

    private final static String[] helpWritelong = {
        "Write to a register pair on the RCM",
        "write <address> <value>",
        "address  The address of the register pair",
        "value    The value to write",
    };

    private final static String[] helpUpdate = {
        "Update a register on the RCM",
        "update <address> <mask> <value>",
        "address  The address of the register to update",
        "mask     The mask of bits to change",
        "value    The value to write under the mask",
    };

    private final static String[] helpEnable = {
        "Enable a register set on the RCM",
        "enable <regset>",
        "regset  The name or number of the register set",
    };

    private final static String[] helpDisable = {
        "Disable a register set on the RCM",
        "disable <regset>",
        "regset  The name or number of the register set",
    };

    private final static String[] helpTrigtime = {
        "Display the trigger time for a register set on the RCM",
        "trigtime <regset> <raw>",
        "regset  The name or number of the register set",
        "raw     If present (value ignored), displays the raw value",
    };

    private final static String[] helpTime = {
        "Set or show the time on the RCM, as time of day",
        "time [<value>]",
        "value  If present (yyyy-mm-dd hh:mm:ss or '.' for current time),",
        "       the time is set; otherwise it is shown",
    };

    private final static String[] helpTimebase = {
        "Set or show the time on the RCM, as a raw value",
        "timebase [<value>]",
        "value  If present (long), the time is set; otherwise it is shown",
    };

    private final static String[] helpLoadtest = {
        "Load the RCM sequencer with a basic test sequence",
        "loadtest <tottime> <ncycle> [<ontime>]",
        "tottime  Time for each ADC on/off cycle (10 ns ticks)",
        "ncycle   The number of on/off cycles to perform",
        "ontime   Time the ADC trigger is on (10 ns ticks, default 10)",
    };

    private final static String[] helpLoadcmnd = {
        "Load a command into the RCM sequencer",
        "loadcmnd command",
        "command  A sequencer text command",
    };

    private final static String[] helpLoadfile = {
        "Load the RCM sequencer from a file of commands",
        "loadfile file",
        "file  The name of a file containing sequencer text commands",
    };

    private final static String[] helpStatus = {
        "Display the contents of the RCM status block",
        "status",
    };

    private final static String[] helpTemp = {
        "Display board temperatures",
        "temp [first] [count]",
        "first  The first temperature to read (default 0)",
        "count  The number of temperatures to read (default 1 or all)",
    };

    private final static String[] helpPower = {
        "Display board power (voltage and current) values",
        "power [first] [count]",
        "first  The first power value to read(default 0)",
        "count  The number of power values to read (default 1 or all)",
    };

    private final static String[] helpDac = {
        "Set a REB board DAC value",
        "dac number chan value",
        "number  The number of the DAC to set (0 or 1)",
        "chan    The number of the channel to set (0 - 7)",
        "value   The value to set",
    };

    /*
    **  Lookup tables
    */
    private final static CmndProcess.Lookup rsetNames;
    static {
        rsetNames = new CmndProcess.Lookup(5);
        rsetNames.add("status",    BaseSet.RSET_STATUS);
        rsetNames.add("timebase",  BaseSet.RSET_TIME_BASE);
        rsetNames.add("sequencer", BaseSet.RSET_SEQUENCER);
        rsetNames.add("poweradcs", BaseSet.RSET_POWER_ADCS);
        rsetNames.add("tempadcs",  BaseSet.RSET_TEMP_ADCS);
    }

    /*
    **  Command definitions
    */
    private final static CmndProcess.Command cmnd;
    static {
        cmnd = new CmndProcess.Command(NUM_CMDS);
        cmnd.add("connect",    CMD_CONNECT,   helpConnect,   "I");
        cmnd.add("disconnect", CMD_DISCONN,   helpDisconn,   "");
        cmnd.add("show",       CMD_SHOW,      helpShow,      "");
        cmnd.add("read",       CMD_READ,      helpRead,      "Ii");
        cmnd.add("write",      CMD_WRITE,     helpWrite,     "IIiiiiiii");
        cmnd.add("update",     CMD_UPDATE,    helpUpdate,    "III");
        cmnd.add("readlong",   CMD_READLONG,  helpReadlong,  "I");
        cmnd.add("writelong",  CMD_WRITELONG, helpWritelong, "IL");
        cmnd.add("enable",     CMD_ENABLE,    helpEnable,    "J",
                 rsetNames);
        cmnd.add("disable",    CMD_DISABLE,   helpDisable,   "J",
                 rsetNames);
        cmnd.add("trigtime",   CMD_TRIGTIME,  helpTrigtime,  "Js",
                 rsetNames);
        cmnd.add("time",       CMD_TIME,      helpTime,      "s");
        cmnd.add("timebase",   CMD_TIMEBASE,  helpTimebase,  "l");
        cmnd.add("loadtest",   CMD_LOADTEST,  helpLoadtest,  "IIi");
        cmnd.add("loadcmnd",   CMD_LOADCMND,  helpLoadcmnd,  "E");
        cmnd.add("loadfile",   CMD_LOADFILE,  helpLoadfile,  "S");
        cmnd.add("status",     CMD_STATUS,    helpStatus,    "");
        cmnd.add("temp",       CMD_TEMP,      helpTemp,      "ii");
        cmnd.add("power",      CMD_POWER,     helpPower,     "ii");
        cmnd.add("dac",        CMD_DAC,       helpDac,       "III");
    }

    /*
    **  Private fields
    */
    private final static PrintStream out = System.out;
    private final CmndProcess proc = new CmndProcess();
    private final BaseSet bset = new BaseSet();
    private final SequencerUtils seq = new SequencerUtils(bset);
    private final StatusSet sset = new StatusSet(bset);
    private final TempAdcs tmp = new TempAdcs(bset);
    private final PowerAdcs pwr = new PowerAdcs(bset);
    private final BoardDacs dac = new BoardDacs(bset);
    private int rcmId = -1;


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        (new TestReg()).run();
        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Runs the test
    **
    ***************************************************************************
    */
    public void run()
    {
        proc.add(this, cmnd);

        try {
            ConsoleReader reader = new ConsoleReader();
            while (true) {
                String line = reader.readLine(">> ");
                if (line == null) break;
                if (!proc.process(line)) break;
            }
        }
        catch (IOException e) {
            out.println(e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Dispatches a command
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int code, int found, Object[] args)
    {
        try {
            switch (code) {
            case CMD_CONNECT:
                procConnect(found, args); break;
            case CMD_DISCONN:
                procDisconn(found, args); break;
            case CMD_SHOW:
                procShow(found, args); break;
            case CMD_READ:
                procRead(found, args); break;
            case CMD_WRITE:
                procWrite(found, args); break;
            case CMD_UPDATE:
                procUpdate(found, args); break;
            case CMD_READLONG:
                procReadLong(found, args); break;
            case CMD_WRITELONG:
                procWriteLong(found, args); break;
            case CMD_ENABLE:
                procEnable(found, args); break;
            case CMD_DISABLE:
                procDisable(found, args); break;
            case CMD_TRIGTIME:
                procTrigtime(found, args); break;
            case CMD_TIME:
                procTime(found, args); break;
            case CMD_TIMEBASE:
                procTimebase(found, args); break;
            case CMD_LOADTEST:
                procLoadtest(found, args); break;
            case CMD_LOADCMND:
                procLoadcmnd(found, args); break;
            case CMD_LOADFILE:
                procLoadfile(found, args); break;
            case CMD_STATUS:
                procStatus(found, args); break;
            case CMD_TEMP:
                procTemp(found, args); break;
            case CMD_POWER:
                procPower(found, args); break;
            case CMD_DAC:
                procDac(found, args); break;
            default:
                out.println("Command not fully implemented");
            }
        }
        catch (RcmException e) {
            out.println(e);
        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Processes the CONNECT command
    **
    ***************************************************************************
    */
    private void procConnect(int found, Object[] args) throws RcmException
    {
        rcmId = (Integer)args[0];
        bset.open(rcmId);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DISCONNECT command
    **
    ***************************************************************************
    */
    private void procDisconn(int found, Object[] args) throws RcmException
    {
        bset.close();
        rcmId = -1;
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOW command
    **
    ***************************************************************************
    */
    private void procShow(int found, Object[] args)
    {
        boolean connected = false;
        int version = -1;
        try {
             bset.checkOpen();
             connected = true;
        }
        catch (RcmException e) {
        }
        try {
            version = bset.getVersion();
        }
        catch (RcmException e) {
        }
        out.println("Connection parameters:");
        out.format("  Connected  = %s\n", connected);
        out.format("  RCM id     = %s\n", rcmId);
        out.format("  Version    = %s\n", version);
    }


   /**
    ***************************************************************************
    **
    **  Processes the READ command
    **
    ***************************************************************************
    */
    private void procRead(int found, Object[] args) throws RcmException
    {
        int address = (Integer)args[0];
        if ((found & 0x02) == 0)
            out.format("%06x: %08x\n", address, bset.read(address));
        else {
            int count = (Integer)args[1];
            int[] values = new int[count];
            bset.read(address, values);
            for (int j = 0; j < count; j++, address++) {
                if ((j & 3) == 0) {
                    if (j != 0) out.println();
                    out.format("%06x:", address);
                }
                out.format(" %08x", values[j]);
            }
            if (count > 0) out.println();
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the WRITE command
    **
    ***************************************************************************
    */
    private void procWrite(int found, Object[] args) throws RcmException
    {
        int address = (Integer)args[0];
        int[] values = new int[8];
        int count = 0;
        for (int j = 1; j <= 8; j++)
            if ((found & (1 << j)) != 0)
                values[count++] = (Integer)args[j];
        if (count == 1)
            bset.write(address, values[0]);
        else {
            bset.write(address, values);
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the UPDATE command
    **
    ***************************************************************************
    */
    private void procUpdate(int found, Object[] args) throws RcmException
    {
        bset.update((Integer)args[0], (Integer)args[1], (Integer)args[2]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the READLONG command
    **
    ***************************************************************************
    */
    private void procReadLong(int found, Object[] args) throws RcmException
    {
        int address = (Integer)args[0];
        out.format("%06x: %016x\n", address, bset.readLong(address));
    }


   /**
    ***************************************************************************
    **
    **  Processes the WRITELONG command
    **
    ***************************************************************************
    */
    private void procWriteLong(int found, Object[] args) throws RcmException
    {
        bset.writeLong((Integer)args[0], (Long)args[1]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the ENABLE command
    **
    ***************************************************************************
    */
    private void procEnable(int found, Object[] args) throws RcmException
    {
        bset.enable((Integer)args[0]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the DISABLE command
    **
    ***************************************************************************
    */
    private void procDisable(int found, Object[] args) throws RcmException
    {
        bset.disable((Integer)args[0]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the TRIGTIME command
    **
    ***************************************************************************
    */
    private void procTrigtime(int found, Object[] args) throws RcmException
    {
        if ((found & 0x02) != 0) {
            long time = bset.getTriggerTimeRaw((Integer)args[0]);
            out.println("Value = " + time);
        }
        else {
            long time = bset.getTriggerTime((Integer)args[0]);
            GregorianCalendar tm = new GregorianCalendar();
            tm.setTimeInMillis(time);
            out.format("Time = %tY-%<tm-%<td %<tH:%<tM:%<tS.%<tL\n", tm);
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the TIME command
    **
    ***************************************************************************
    */
    private void procTime(int found, Object[] args) throws RcmException
    {
        if (found == 0) {
            long time = bset.getTime();
            GregorianCalendar tm = new GregorianCalendar();
            tm.setTimeInMillis(time);
            out.format("Time = %tY-%<tm-%<td %<tH:%<tM:%<tS.%<tL\n", tm);
        }
        else {
            String time = (String)args[0];
            if (time.equals(".")) {
                bset.setTime();
            }
            else {
                SimpleDateFormat fmt = new SimpleDateFormat();
                fmt.setLenient(true);
                fmt.applyPattern("y-M-d H:m:s");
                ParsePosition pos = new ParsePosition(0);
                Date date = fmt.parse(time, pos);
                if (date == null) {
                    out.println("Expected format is yyyy-mm-dd hh:mm:ss");
                }
                else {
                    bset.setTime(date.getTime());
                }
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the TIMEBASE command
    **
    ***************************************************************************
    */
    private void procTimebase(int found, Object[] args) throws RcmException
    {
        if (found == 0) {
            out.println("Value = " + bset.getTimeRaw());
        }
        else {
            bset.setTimeRaw((Long)args[0]);
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the LOADTEST command
    **
    ***************************************************************************
    */
    private void procLoadtest(int found, Object[] args) throws RcmException
    {
        int totTime = (Integer)args[0],
            ncycle = (Integer)args[1],
            onTime = (found & 0x04) != 0 ? (Integer)args[2] : 10;

        seq.writeLines(0, new int[]{0});          // Default state
        seq.writeLines(1, new int[]{0x1000, 0});  // ADC trigger on, then off
        seq.writeTimes(1, new int[]{onTime, totTime - onTime, 0});
        if (seq.getVersion() == Sequencer.VERSION_0) {
            seq.writeStack(new int[]{0x10000000 | (ncycle & 0x03ffffff)});
            seq.writeDataSource(Sequencer.SEQ_SRC_FPGA_PTN);
        }
        else {
            seq.writeProgExec(0, 1, ncycle);
            seq.writeProgEnd(1);
            seq.writeDataSource(Sequencer.SEQ_SRC_PATTERN);
        }
        seq.writeSliceCount(ncycle);
    }


   /**
    ***************************************************************************
    **
    **  Processes the LOADCMND command
    **
    ***************************************************************************
    */
    private void procLoadcmnd(int found, Object[] args) throws RcmException
    {
        seq.loadCommand((String)args[0]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the LOADFILE command
    **
    ***************************************************************************
    */
    private void procLoadfile(int found, Object[] args) throws RcmException
    {
        try {
            int nSlice = seq.loadFile((String)args[0]);
            out.println("Slice count = " + nSlice);
        }
        catch (IOException e) {
            out.println(e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Processes the STATUS command
    **
    ***************************************************************************
    */
    private void procStatus(int found, Object[] args) throws RcmException
    {
        Status status = sset.readStatus();
        out.println("Status Block:");
        out.format("  Version          = %08x  ", status.getVersion());
        out.format("  Link Status      = %08x\n", status.getLinkStatus());
        out.format("  Cell error count = %-8s  ", status.getCellErrorCount());
        out.format("  Link down count  = %s\n", status.getLinkDownCount());
        out.format("  Link error count = %-8s  ", status.getLinkErrorCount());
        out.format("  VC buffer status = %08x\n", status.getVcBufferStatus());
        int[] counts = status.getVcRxCounts();
        out.format("  VC rcve counts   =");
        for (int j = 0; j < counts.length; j++)
            out.format(" %s", counts[j]);
        out.println();
        counts = status.getVcTxCounts();
        out.format("  VC xmit counts   =");
        for (int j = 0; j < counts.length; j++)
            out.format(" %s", counts[j]);
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Processes the TEMP command
    **
    ***************************************************************************
    */
    private void procTemp(int found, Object[] args) throws RcmException
    {
        int first = 0;
        float[] values = null;
        switch (found & 0x03) {
        case 1:
            values = new float[1];
            values[0] = tmp.readAdc((Integer)args[0]);
            break;
        case 0:
            values = tmp.readAdcs();
            break;
        case 3:
            first = (Integer)args[0];
        case 2:
            values = tmp.readAdcs(first, (Integer)args[1]);
            break;
        }
        for (int j = 0; j < values.length; j++) {
            if ((j & 3) == 0) {
                out.print(j == 0 ? "ADCs:" : "\n     ");
            }
            out.format("  %2s: %6g", j + first, values[j]);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Processes the POWER command
    **
    ***************************************************************************
    */
    private void procPower(int found, Object[] args) throws RcmException
    {
        int first = 0;
        float[] values = null;
        switch (found & 0x03) {
        case 1:
            values = new float[1];
            values[0] = pwr.readAdc((Integer)args[0]);
            break;
        case 0:
            values = pwr.readAdcs();
            break;
        case 3:
            first = (Integer)args[0];
        case 2:
            values = pwr.readAdcs(first, (Integer)args[1]);
            break;
        }
        for (int j = 0; j < values.length; j++) {
            if ((j & 3) == 0) {
                out.print(j == 0 ? "ADCs:" : "\n     ");
            }
            out.format("  %2s: %6g", j + first, values[j]);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Processes the DAC command
    **
    ***************************************************************************
    */
    private void procDac(int found, Object[] args) throws RcmException
    {
        dac.set((Integer)args[0], (Integer)args[1], (Integer)args[2]);
    }

}
