package org.lsst.ccs.drivers.rcm;

/**
 ***************************************************************************
 **
 **  RCM sequencer access routines
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class Sequencer extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    public final static int
        REG_SEQ_LINES    = 0x100000,
        REG_SEQ_TIMES    = 0x200000,
        REG_SEQ_STACK    = 0x300000,
        REG_SEQ_PROGRAM  = 0x300000,
        REG_SEQ_STEP     = 0x310000,
        REG_SEQ_STOP     = 0x320000,
        REG_SEQ_SSIZE    = 0x400000,
        REG_SEQ_NSLICE   = 0x400005,
        REG_SEQ_SOURCE   = 0x400006,
        REG_SEQ_BEB      = 0x400007,
        REG_SEQ_STEP_0   = 0x500000,
        REG_SEQ_STOP_0   = 0x600000,
        SEQ_MAX_FUNC_0   = 4,
        SEQ_MAX_FUNC     = 16,
        SEQ_MAX_SLICE    = 16,
        SEQ_MAX_PROGRAM  = 1024,
        SEQ_MAX_STACK    = 4096,
        SEQ_SRC_FPGA_PTN = 0,
        SEQ_SRC_BEB_PTN  = 0x1beb,
        SEQ_SRC_BEB_ADC  = 0x0beb,
        SEQ_SRC_ADC      = 0,
        SEQ_SRC_PATTERN  = 1,
        SEQ_OPC_EXECUTE  = 1,
        SEQ_OPC_JUMP     = 2,
        SEQ_OPC_END_SUBR = 3,
        SEQ_OPC_END_PROG = 4,
        SEQ_PRG_V_OPCODE = 28,
        SEQ_PRG_V_FUNC   = 24,
        SEQ_PRG_M_FUNC   = SEQ_MAX_FUNC - 1,
        SEQ_PRG_V_LOOP   = 23,
        SEQ_PRG_M_EXCCNT = 0x7fffff,
        SEQ_PRG_V_SUBADD = 18,
        SEQ_PRG_M_SUBADD = 0x3ff,
        SEQ_PRG_M_SUBCNT = 0x1ffff,
        SEQ_STK_V_FUNC   = 28,
        SEQ_STK_M_FUNC   = SEQ_MAX_FUNC_0 - 1,
        SEQ_STK_V_LOOP   = 27,
        SEQ_STK_M_COUNT  = 0x7ffffff;

   /**
    ***************************************************************************
    **
    **  Fields
    **
    ***************************************************************************
    */
    protected int[][] lines = new int[SEQ_MAX_FUNC][SEQ_MAX_SLICE];
    protected int[][] times = new int[SEQ_MAX_FUNC][SEQ_MAX_SLICE];
    protected int[] prog  = new int[SEQ_MAX_PROGRAM];
    protected int[] stack = new int[SEQ_MAX_STACK];
    protected int stackSize;
        

   /**
    ***************************************************************************
    **
    **  Constructors.
    **
    ***************************************************************************
    */
    public Sequencer(RegClient reg)
    {
        super(reg);
    }

    public Sequencer()
    {
        super();
    }


   /**
    ***************************************************************************
    **
    **  Enables the sequencer.
    **
    ***************************************************************************
    */
    @Override
    public void enable() throws RcmException
    {
        enable(RSET_SEQUENCER);
    }


   /**
    ***************************************************************************
    **
    **  Disables the sequencer.
    **
    ***************************************************************************
    */
    @Override
    public void disable() throws RcmException
    {
        disable(RSET_SEQUENCER);
    }


   /**
    ***************************************************************************
    **
    **  Gets the sequencer trigger time.
    **
    ***************************************************************************
    */
    public long getTriggerTime() throws RcmException
    {
        return getTriggerTime(RSET_SEQUENCER);
    }


   /**
    ***************************************************************************
    **
    **  Writes an output lines value.
    **
    ***************************************************************************
    */
    public void writeLines(int func, int slice, int value)
        throws RcmException
    {
        checkFunc(func);
        checkSlice(slice);
        write(REG_SEQ_LINES | (func<< 4) | slice, value);
        lines[func][slice] = value;
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of output lines values
    **
    ***************************************************************************
    */
    public void writeLines(int func, int[] values) throws RcmException
    {
        writeLines(func, values, 0, values.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of output lines values
    **
    ***************************************************************************
    */
    public void writeLines(int func, int[] values, int offset, int count)
        throws RcmException
    {
        checkFunc(func);
        checkSlice(count - 1);
        write(REG_SEQ_LINES | (func << 4), values, offset, count);
        System.arraycopy(values, offset, lines[func], 0, count);
    }


   /**
    ***************************************************************************
    **
    **  Writes a time value
    **
    ***************************************************************************
    */
    public void writeTimes(int func, int slice, int value)
        throws RcmException
    {
        checkFunc(func);
        checkSlice(slice);
        write(REG_SEQ_TIMES | (func << 4) | slice, value);
        times[func][slice] = value;
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of time values
    **
    ***************************************************************************
    */
    public void writeTimes(int func, int[] values) throws RcmException
    {
        writeTimes(func, values, 0, values.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of time values
    **
    ***************************************************************************
    */
    public void writeTimes(int func, int[] values, int offset, int count)
        throws RcmException
    {
        checkFunc(func);
        checkSlice(count - 1);
        write(REG_SEQ_TIMES | (func << 4), values, offset, count);
        System.arraycopy(values, offset, times[func], 0, count);
    }


   /**
    ***************************************************************************
    **
    **  Writes a stack value
    **
    ***************************************************************************
    */
    public void writeStack(int addr, int value) throws RcmException
    {
        checkVersion(VERSION_0);
        checkStackAddr(addr);
        write(REG_SEQ_STACK + addr, value);
        stack[addr] = value;
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of stack values
    **
    ***************************************************************************
    */
    public void writeStack(int addr, int[] values) throws RcmException
    {
        writeStack(addr, values, 0, values.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of stack values
    **
    ***************************************************************************
    */
    public void writeStack(int addr, int[] values, int offset, int count)
        throws RcmException
    {
        checkVersion(VERSION_0);
        checkStackAddr(addr, count);
        write(REG_SEQ_STACK + addr, values, offset, count);
        System.arraycopy(values, offset, stack, addr, count);
    }


   /**
    ***************************************************************************
    **
    **  Writes a complete set of stack values, setting the stack size
    **
    ***************************************************************************
    */
    public void writeStack(int[] values) throws RcmException
    {
        writeStack(values, 0, values.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes a complete set of stack values, setting the stack size
    **
    ***************************************************************************
    */
    public void writeStack(int[] values, int offset, int count)
        throws RcmException
    {
        checkVersion(VERSION_0);
        checkStackAddr(0, count);
        write(REG_SEQ_STACK, values, offset, count);
        write(REG_SEQ_SSIZE, count);
        System.arraycopy(values, offset, stack, 0, count);
        stackSize = count;
    }


   /**
    ***************************************************************************
    **
    **  Writes the stack size
    **
    ***************************************************************************
    */
    public void writeStackSize(int value) throws RcmException
    {
        checkVersion(VERSION_0);
        write(REG_SEQ_SSIZE, value);
        stackSize = value;
    }


   /**
    ***************************************************************************
    **
    **  Writes a program word
    **
    ***************************************************************************
    */
    public void writeProgram(int addr, int value) throws RcmException
    {
        checkNotVersion(VERSION_0);
        checkProgAddr(addr);
        write(REG_SEQ_PROGRAM + addr, value);
        prog[addr] = value;
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of program words
    **
    ***************************************************************************
    */
    public void writeProgram(int addr, int[] values) throws RcmException
    {
        writeProgram(addr, values, 0, values.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes a set of program words
    **
    ***************************************************************************
    */
    public void writeProgram(int addr, int[] values, int offset, int count)
        throws RcmException
    {
        checkNotVersion(VERSION_0);
        checkProgAddr(addr, count);
        write(REG_SEQ_PROGRAM + addr, values, offset, count);
        System.arraycopy(values, offset, prog, addr, count);
    }


   /**
    ***************************************************************************
    **
    **  Writes an execute program word
    **
    ***************************************************************************
    */
    public void writeProgExec(int addr, int func, int count)
        throws RcmException
    {
        checkFunc(func);
        int value = count & SEQ_PRG_M_EXCCNT;
        if (count != -1 && count != value) {
            throw new RcmException("Invalid repetition count");
        }
        value |= (SEQ_OPC_EXECUTE << SEQ_PRG_V_OPCODE)
                   | (func << SEQ_PRG_V_FUNC)
                   | (count == -1 ? (1 << SEQ_PRG_V_LOOP) : 0);
        writeProgram(addr, value);
    }


   /**
    ***************************************************************************
    **
    **  Writes a jump program word
    **
    ***************************************************************************
    */
    public void writeProgJump(int addr, int subaddr, int count)
        throws RcmException
    {
        checkProgAddr(subaddr);
        int value = count & SEQ_PRG_M_SUBCNT;
        if (count != value) {
            throw new RcmException("Invalid repetition count");
        }
        value |= (SEQ_OPC_JUMP << SEQ_PRG_V_OPCODE)
                   | (subaddr << SEQ_PRG_V_SUBADD);
        writeProgram(addr, value);
    }


   /**
    ***************************************************************************
    **
    **  Writes an end subroutine word
    **
    ***************************************************************************
    */
    public void writeProgEndSubr(int addr) throws RcmException
    {
        writeProgram(addr, SEQ_OPC_END_SUBR << SEQ_PRG_V_OPCODE);
    }


   /**
    ***************************************************************************
    **
    **  Writes an end program word
    **
    ***************************************************************************
    */
    public void writeProgEnd(int addr) throws RcmException
    {
        writeProgram(addr, SEQ_OPC_END_PROG << SEQ_PRG_V_OPCODE);
    }


   /**
    ***************************************************************************
    **
    **  Writes the slice count
    **
    ***************************************************************************
    */
    public void writeSliceCount(int value) throws RcmException
    {
        write(REG_SEQ_NSLICE, value);
    }


   /**
    ***************************************************************************
    **
    **  Writes the data source register
    **
    ***************************************************************************
    */
    public void writeDataSource(int value) throws RcmException
    {
        write(REG_SEQ_SOURCE, value);
    }


   /**
    ***************************************************************************
    **
    **  Writes the BEB selector register
    **
    ***************************************************************************
    */
    public void writeBebSelect(int value) throws RcmException
    {
        checkVersion(VERSION_0);
        write(REG_SEQ_BEB, value);
    }


   /**
    ***************************************************************************
    **
    **  Sends the step command
    **
    ***************************************************************************
    */
    public void sendStep() throws RcmException
    {
        int stepAddr = getVersion() == VERSION_0 ? REG_SEQ_STEP_0 : REG_SEQ_STEP;
        write(stepAddr, 0);
    }


   /**
    ***************************************************************************
    **
    **  Sends the stop command
    **
    ***************************************************************************
    */
    public void sendStop() throws RcmException
    {
        int stopAddr = getVersion() == VERSION_0 ? REG_SEQ_STOP_0 : REG_SEQ_STOP;
        write(stopAddr, 0);
    }


   /**
    ***************************************************************************
    **
    **  Checks a function number
    **
    ***************************************************************************
    */
    private void checkFunc(int func) throws RcmException
    {
        int maxFunc = getVersion() == VERSION_0
                        ? SEQ_MAX_FUNC_0 : SEQ_MAX_FUNC;
        if (func < 0 || func >= maxFunc) {
            throw new RcmException("Invalid function number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a time slice number
    **
    ***************************************************************************
    */
    private void checkSlice(int slice) throws RcmException
    {
        if (slice < 0 || slice >= SEQ_MAX_SLICE) {
            throw new RcmException("Invalid time slice number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a stack address
    **
    ***************************************************************************
    */
    private void checkStackAddr(int addr) throws RcmException
    {
        if (addr < 0 || addr >= SEQ_MAX_STACK) {
            throw new RcmException("Invalid stack address");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a range of stack addresses
    **
    ***************************************************************************
    */
    private void checkStackAddr(int addr, int count) throws RcmException
    {
        if (addr < 0 || count <= 0 || addr + count > SEQ_MAX_STACK) {
            throw new RcmException("Invalid stack address");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a program address
    **
    ***************************************************************************
    */
    private void checkProgAddr(int addr) throws RcmException
    {
        if (addr < 0 || addr >= SEQ_MAX_PROGRAM) {
            throw new RcmException("Invalid program address");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a range of program addresses
    **
    ***************************************************************************
    */
    private void checkProgAddr(int addr, int count) throws RcmException
    {
        if (addr < 0 || count <= 0 || addr + count > SEQ_MAX_PROGRAM) {
            throw new RcmException("Invalid program address");
        }
    }

}
