package org.lsst.ccs.drivers.rcm;

/**
 ***************************************************************************
 **
 **  Java interface to the RCM registers
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class RegClient {

   /**
    ***************************************************************************
    **
    **  Inner class to contain a copy of the register access handle
    **
    ***************************************************************************
    */
    class Handle {

        long value;
        int instance;

    }


   /**
    ***************************************************************************
    **
    **  Data fields
    **
    ***************************************************************************
    */
    Handle handle;


   /**
    ***************************************************************************
    **
    **  Static initializer
    **
    ***************************************************************************
    */
    static {
        System.loadLibrary("Rcm");
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public RegClient()
    {
        handle = new Handle();
    }


   /**
    ***************************************************************************
    **
    **  Finalizer
    **
    ***************************************************************************
    */
    @Override
    protected void finalize() throws Throwable
    {
        super.finalize();
        deleteRegClient(handle.value);
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection
    **
    ***************************************************************************
    */
    public void open(int id) throws RcmException
    {
        if (handle.value != 0) {
            throw new RcmException("Connection already open");
        }
        handle.value = newRegClient(id);
        handle.instance++;
    }


   /**
    ***************************************************************************
    **
    **  Closes a connection
    **
    ***************************************************************************
    */
    public void close() throws RcmException
    {
        checkOpen();
        deleteRegClient(handle.value);
        handle.value = 0;
    }


   /**
    ***************************************************************************
    **
    **  Checks that connection is open
    **
    ***************************************************************************
    */
    protected void checkOpen() throws RcmException
    {
        if (handle.value == 0) {
            throw new RcmException("Connection not open");
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads from a register
    **
    ***************************************************************************
    */
    public int read(int address) throws RcmException
    {
        checkOpen();
        return readReg(handle.value, address);
    }


   /**
    ***************************************************************************
    **
    **  Reads from a set of registers
    **
    ***************************************************************************
    */
    public void read(int address, int[] values, int offset, int count)
        throws RcmException
    {
        checkOpen();
        readRegs(handle.value, address, values, offset, count);
    }


   /**
    ***************************************************************************
    **
    **  Reads from a set of registers
    **
    ***************************************************************************
    */
    public void read(int address, int[] values) throws RcmException
    {
        checkOpen();
        readRegs(handle.value, address, values, 0, values.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes to a register
    **
    ***************************************************************************
    */
    public void write(int address, int value) throws RcmException
    {
        checkOpen();
        writeReg(handle.value, address, value);
    }


   /**
    ***************************************************************************
    **
    **  Writes to a set of registers
    **
    ***************************************************************************
    */
    public void write(int address, int[] values, int offset, int count)
        throws RcmException
    {
        checkOpen();
        writeRegs(handle.value, address, values, offset, count);
    }


   /**
    ***************************************************************************
    **
    **  Writes to a set of registers
    **
    ***************************************************************************
    */
    public void write(int address, int[] values) throws RcmException
    {
        checkOpen();
        writeRegs(handle.value, address, values, 0, values.length);
    }


   /**
    ***************************************************************************
    **
    **  Updates a register
    **
    ***************************************************************************
    */
    public int update(int address, int mask, int value) throws RcmException
    {
        checkOpen();
        return updateReg(handle.value, address, mask, value);
    }


   /**
    ***************************************************************************
    **
    **  Reads a register pair as a long value
    **
    ***************************************************************************
    */
    public long readLong(int address) throws RcmException
    {
        checkOpen();
        int[] vals = new int[2];
        readRegs(handle.value, address, vals, 0, vals.length);
        return ((long)vals[1] << 32) | (vals[0] & 0xffffffffL);
    }


   /**
    ***************************************************************************
    **
    **  Writes a long value to a register pair
    **
    ***************************************************************************
    */
    public void writeLong(int address, long value) throws RcmException
    {
        checkOpen();
        int[] vals = {(int)value, (int)(value >> 32)};
        writeRegs(handle.value, address, vals, 0, vals.length);
    }


   /**
    ***************************************************************************
    **
    **  Performs one-time initialization
    **
    ***************************************************************************
    */
    private native static void initSys();


   /**
    ***************************************************************************
    **
    **  Creates a new object
    **
    ***************************************************************************
    */
    private native long newRegClient(int id);


   /**
    ***************************************************************************
    **
    **  Deletes the object
    **
    ***************************************************************************
    */
    private native void deleteRegClient(long handle);


   /**
    ***************************************************************************
    **
    **  Reads from a register
    **
    ***************************************************************************
    */
    private native int readReg(long handle, int address) throws RcmException;


   /**
    ***************************************************************************
    **
    **  Reads from a set of registers
    **
    ***************************************************************************
    */
    private native void readRegs(long handle, int address, int[] values,
                                 int offset, int count)
        throws RcmException;


   /**
    ***************************************************************************
    **
    **  Writes to a register
    **
    ***************************************************************************
    */
    private native void writeReg(long handle, int address, int value)
        throws RcmException;


   /**
    ***************************************************************************
    **
    **  Writes to a set of registers
    **
    ***************************************************************************
    */
    private native void writeRegs(long handle, int address, int[] values,
                                  int offset, int count)
        throws RcmException;


   /**
    ***************************************************************************
    **
    **  Updates a register
    **
    ***************************************************************************
    */
    private native int updateReg(long handle, int address, int mask, int value)
        throws RcmException;

}
