package org.lsst.ccs.drivers.rcm;

/**
 ***************************************************************************
 **
 **  Power ADCs reading routines
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class PowerAdcs extends BaseSet {

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    public final static int
        REG_POWER_ADCS  = 0x600000,
        ADC_V6_VOLTAGE  = 0,
        ADC_V6_CURRENT  = 1,
        ADC_V9_VOLTAGE  = 2,
        ADC_V9_CURRENT  = 3,
        ADC_V24_VOLTAGE = 4,
        ADC_V24_CURRENT = 5,
        ADC_V40_VOLTAGE = 6,
        ADC_V40_CURRENT = 7,
        NUM_POWER_REGS  = 8;

    public final static float
        VOLTAGE_SCALE   = 1 / 640F,
        CURRENT_SCALE   = 1 / 640000F;


   /**
    ***************************************************************************
    **
    **  Constructors.
    **
    ***************************************************************************
    */
    public PowerAdcs()
    {
        super();
    }

    public PowerAdcs(RegClient reg)
    {
        super(reg);
    }


   /**
    ***************************************************************************
    **
    **  Enables the power ADC reading
    **
    ***************************************************************************
    */
    @Override
    public void enable() throws RcmException
    {
        checkNotVersion(VERSION_0);
        enable(RSET_POWER_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Waits for the data to be available
    **
    ***************************************************************************
    */
    public void waitDone() throws RcmException
    {
        checkNotVersion(VERSION_0);
        waitDone(RSET_POWER_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Gets the time of the read enable
    **
    ***************************************************************************
    */
    public long getTriggerTime() throws RcmException
    {
        checkVersion(VERSION_2);
        return getTriggerTime(RSET_POWER_ADCS);
    }


   /**
    ***************************************************************************
    **
    **  Reads one power value
    **
    ***************************************************************************
    */
    public float readAdc(int adc) throws RcmException
    {
        if (adc < 0 || adc >= NUM_POWER_REGS) {
            throw new RcmException("Invalid power ADC number");
        }
        enable();
        waitDone();

        return ((adc & 1) == 0 ? VOLTAGE_SCALE : CURRENT_SCALE)
                 * read(REG_POWER_ADCS + adc);
    }


   /**
    ***************************************************************************
    **
    **  Reads all the power values
    **
    ***************************************************************************
    */
    public float[] readAdcs() throws RcmException
    {
        return readAdcs(0, NUM_POWER_REGS);
    }


   /**
    ***************************************************************************
    **
    **  Reads a range of power values
    **
    ***************************************************************************
    */
    public float[] readAdcs(int first, int count) throws RcmException
    {
        if (first < 0 || count < 0 || first + count > NUM_POWER_REGS) {
            throw new RcmException("Invalid power ADC range");
        }
        enable();
        waitDone();
        int[] rawData = new int[count];
        float[] data = new float[count];
        read(REG_POWER_ADCS + first, rawData);
        for (int j = 0; j < count; j++) {
            data[j] = (((first + j) & 1) == 0 ? VOLTAGE_SCALE : CURRENT_SCALE)
                        * rawData[j];
        }

        return data;
    }

}
