package org.lsst.ccs.drivers.rcm;

/**
 ***************************************************************************
 **
 **  Java interface to the RCM registers.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class RegClient {

   /**
    ***************************************************************************
    **
    **  Inner class to contain a copy of the register access handle.
    **
    ***************************************************************************
    */
    class Handle {

        long value;
        int instance;

    }


   /**
    ***************************************************************************
    **
    **  Data fields.
    **
    ***************************************************************************
    */
    Handle handle;


   /**
    ***************************************************************************
    **
    **  Static initializer.
    **
    ***************************************************************************
    */
    static {
        System.loadLibrary("Rcm");
    }


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    ***************************************************************************
    */
    public RegClient()
    {
        handle = new Handle();
    }


   /**
    ***************************************************************************
    **
    **  Finalizer.
    **
    ***************************************************************************
    */
    @Override
    protected void finalize() throws Throwable
    {
        super.finalize();
        deleteRegClient(handle.value);
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection.
    **
    **  @param  id  The ID of the RCM to connect to
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void open(int id) throws RcmException
    {
        if (handle.value != 0) {
            throw new RcmException("Connection already open");
        }
        handle.value = newRegClient(id);
        handle.instance++;
    }


   /**
    ***************************************************************************
    **
    **  Closes a connection.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void close() throws RcmException
    {
        checkOpen();
        deleteRegClient(handle.value);
        handle.value = 0;
    }


   /**
    ***************************************************************************
    **
    **  Checks that connection is open.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    protected void checkOpen() throws RcmException
    {
        if (handle.value == 0) {
            throw new RcmException("Connection not open");
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads from a register.
    **
    **  @param  address  The address of the register to read
    **
    **  @return  The value contained in the register
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public int read(int address) throws RcmException
    {
        checkOpen();
        return readReg(handle.value, address);
    }


   /**
    ***************************************************************************
    **
    **  Reads from a set of registers.
    **
    **  @param  address  The address of the first register to read
    **
    **  @param  values   An array to receive the register values
    **
    **  @param  offset   The offset to the array element where the first value
    **                   is to be put
    **
    **  @param  count    The number of registers to read
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void read(int address, int[] values, int offset, int count)
        throws RcmException
    {
        checkOpen();
        readRegs(handle.value, address, values, offset, count);
    }


   /**
    ***************************************************************************
    **
    **  Reads from a set of registers.
    **
    **  @param  address  The address of the first register to read
    **
    **  @param  values   An array to receive the register values.  The size
    **                   of the array determines how many registers to read.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void read(int address, int[] values) throws RcmException
    {
        checkOpen();
        readRegs(handle.value, address, values, 0, values.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes to a register.
    **
    **  @param  address  The address of the register to write
    **
    **  @param  value    The value to write to the register
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void write(int address, int value) throws RcmException
    {
        checkOpen();
        writeReg(handle.value, address, value);
    }


   /**
    ***************************************************************************
    **
    **  Writes to a set of registers.
    **
    **  @param  address  The address of the first register to write
    **
    **  @param  values   An array containing the values to write
    **
    **  @param  offset   The offset to the array element containing the first
    **                   value to write
    **
    **  @param  count    The number of registers to write
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void write(int address, int[] values, int offset, int count)
        throws RcmException
    {
        checkOpen();
        writeRegs(handle.value, address, values, offset, count);
    }


   /**
    ***************************************************************************
    **
    **  Writes to a set of registers.
    **
    **  @param  address  The address of the first register to write
    **
    **  @param  values   An array containing the values to write.  The size of
    **                   array is the number of registers to write.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void write(int address, int[] values) throws RcmException
    {
        checkOpen();
        writeRegs(handle.value, address, values, 0, values.length);
    }


   /**
    ***************************************************************************
    **
    **  Updates a register.
    **
    **  @param  address  The address of the register to update
    **
    **  @param  mask     A mask in which set bits indicate which bits of
    **                   value are to be written
    **
    **  @param  value    The value to write to the register
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public int update(int address, int mask, int value) throws RcmException
    {
        checkOpen();
        return updateReg(handle.value, address, mask, value);
    }


   /**
    ***************************************************************************
    **
    **  Reads a register pair as a long value.
    **
    **  @param  address  The address of the first register to read
    **
    **  @return  The value obtained by combining the values as a little-endian
    **           pair
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public long readLong(int address) throws RcmException
    {
        checkOpen();
        int[] vals = new int[2];
        readRegs(handle.value, address, vals, 0, vals.length);
        return ((long)vals[1] << 32) | (vals[0] & 0xffffffffL);
    }


   /**
    ***************************************************************************
    **
    **  Writes a long value to a register pair.
    **
    **  @param  address  The address of the first register to write
    **
    **  @param  value    The value to write, as a little-endian pair
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void writeLong(int address, long value) throws RcmException
    {
        checkOpen();
        int[] vals = {(int)value, (int)(value >> 32)};
        writeRegs(handle.value, address, vals, 0, vals.length);
    }


   /**
    ***************************************************************************
    **
    **  Performs one-time initialization.
    **
    ***************************************************************************
    */
    private native static void initSys();


   /**
    ***************************************************************************
    **
    **  Creates a new register client object.
    **
    **  @param  id  The ID of the RCM to be accessed by the object
    **
    **  @return  The handle for the created object
    **
    ***************************************************************************
    */
    private native long newRegClient(int id);


   /**
    ***************************************************************************
    **
    **  Deletes the object.
    **
    **  @param  handle  The handle of the register client object
    **
    ***************************************************************************
    */
    private native void deleteRegClient(long handle);


   /**
    ***************************************************************************
    **
    **  Reads from a register.
    **
    **  @param  handle   The handle of the register client object
    **
    **  @param  address  The address of the register to read
    **
    **  @return  The value contained in the register
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private native int readReg(long handle, int address) throws RcmException;


   /**
    ***************************************************************************
    **
    **  Reads from a set of registers.
    **
    **  @param  handle   The handle of the register client object
    **
    **  @param  address  The address of the first register to read
    **
    **  @param  values   An array to receive the register values
    **
    **  @param  offset   The offset to the array element where the first value
    **                   is to be put
    **
    **  @param  count    The number of registers to read
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private native void readRegs(long handle, int address, int[] values,
                                 int offset, int count)
        throws RcmException;


   /**
    ***************************************************************************
    **
    **  Writes to a register.
    **
    **  @param  handle   The handle of the register client object
    **
    **  @param  address  The address of the register to write
    **
    **  @param  value    The value to write to the register
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private native void writeReg(long handle, int address, int value)
        throws RcmException;


   /**
    ***************************************************************************
    **
    **  Writes to a set of registers.
    **
    **  @param  handle   The handle of the register client object
    **
    **  @param  address  The address of the first register to write
    **
    **  @param  values   An array containing the values to write
    **
    **  @param  offset   The offset to the array element containing the first
    **                   value to write
    **
    **  @param  count    The number of registers to write
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private native void writeRegs(long handle, int address, int[] values,
                                  int offset, int count)
        throws RcmException;


   /**
    ***************************************************************************
    **
    **  Updates a register.
    **
    **  @param  handle   The handle of the register client object
    **
    **  @param  address  The address of the register to update
    **
    **  @param  mask     A mask in which set bits indicate which bits of
    **                   value are to be written
    **
    **  @param  value    The value to write to the register
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private native int updateReg(long handle, int address, int mask, int value)
        throws RcmException;

}
