package org.lsst.ccs.drivers.rcm;

/**
 ***************************************************************************
 **
 **  Java interface to the image client routines
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class ImageClient {

   /**
    ***************************************************************************
    **
    **  Defines the data listener.
    **
    ***************************************************************************
    */
    public interface Listener {

       /**
        ***********************************************************************
        **
        **  Receives image data.
        **
        ***********************************************************************
        */
        public void processImage(Image image);

    }


   /**
    ***************************************************************************
    **
    **  Thread that obtains the image data for the listener.
    **
    ***************************************************************************
    */
    public class ReadThread extends Thread {

        Listener listener;   // Attached listener
        Image listenImage;   // Image supplied by listener
        int state = RUNNING;


       /**
        ***********************************************************************
        **
        **  Constructor.
        **
        ***********************************************************************
        */
        public ReadThread(Listener lstnr, Image image)
        {
            listener = lstnr;
            listenImage = image;
        }


       /**
        ***********************************************************************
        **
        **  Receives image data.
        **
        ***********************************************************************
        */
        @Override
        public void run()
        {
            while (state == RUNNING) {
                Image image = waitForImage(listenImage);
                if (state != RUNNING) break;
                getImage(image);
                Listener l = listener;
                if (state != RUNNING) break;
                l.processImage(image);
            }
            if (state == CLOSING) {
                deleteImageClient();
            }
        }


       /**
        ***********************************************************************
        **
        **  Sets the thread state.
        **
        ***********************************************************************
        */
        public void setState(int newState)
        {
            state = newState;
        }

    }


   /**
    ***************************************************************************
    **
    **  Private data.
    **
    ***************************************************************************
    */
    private final static int
        RUNNING = 0,
        CLOSING = 1,
        ENDING  = 2;

    private long client, subscriber;
    private ReadThread reader;
    private Listener imgListener;
    private Image listenImage;


   /**
    ***************************************************************************
    **
    **  Static initializer.
    **
    ***************************************************************************
    */
    static {
        System.loadLibrary("Rcm");
        initSys();
    }


   /**
    ***************************************************************************
    **
    **  Finalizer.
    **
    ***************************************************************************
    */
    @Override
    protected void finalize() throws Throwable
    {
        super.finalize();
        deleteImageClient();
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection.
    **
    **  @param  id  The ID of the RCM to connect to
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void open(int id) throws RcmException
    {
        if (client != 0) {
            throw new RcmException("Connection already open");
        }
        newImageClient(id);
        if (imgListener != null) {
            reader = new ReadThread(imgListener, listenImage);
            reader.setDaemon(true);
            reader.start();
        }
    }


   /**
    ***************************************************************************
    **
    **  Closes a connection.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public void close() throws RcmException
    {
        checkOpen();
        if (reader == null) {
            deleteImageClient();
        }
        else {
            reader.setState(CLOSING);
            reader = null;
        }
    }


   /**
    ***************************************************************************
    **
    **  Awaits an image.
    **
    **  Waits until a new image has been generated.
    **
    **  @param  image  An Image object in which to save the reference and
    **                 metadata for the new image, or null if a new image
    **                 object is to be created.
    **
    **  @return  The Image object containing the new image reference and
    **           metadata.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public Image awaitImage(Image image) throws RcmException
    {
        checkOpen();
        return waitForImage(image);
    }


   /**
    ***************************************************************************
    **
    **  Reads an image.
    **
    **  Gets the pixel data for an image.
    **
    **  @param  image  The Image object containing the valid metadata for an
    **                 image.  If the contained pixel data byte array is not
    **                 null and is large enough, the pixel data us stored
    **                 there.  Otherwise a new array is created.
    **
    **  @return  True if the pixel data was successfully read, false otherwise.
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    public boolean readImage(Image image) throws RcmException
    {
        checkOpen();
        return getImage(image);
    }


   /**
    ***************************************************************************
    **
    **  Deletes an image's metadata reference.
    **
    **  Deletes referenced C++ object.
    **
    **  @param  image  The image whose metadata reference is to be deleted.
    **
    ***************************************************************************
    */
    public native static void deleteImageMetadataRef(Image image);


   /**
    ***************************************************************************
    **
    **  Sets the data listener.
    **
    **  Sets the data listener object, whose class must implement the Listener
    **  interface, and therefore must contain the processImage method.  This
    **  method is called with a complete Image object as its argument whenever
    **  a new image becomes available.
    **
    **  @param  listener  The listener object to be set as the listener.  Any
    **                    existing listener is replaced.
    **
    **  @param  image     An image object to be used to contain received
    **                    images, or null if a new object is to be created
    **                    each time.
    **
    ***************************************************************************
    */
    public void setListener(Listener listener, Image image)
    {
        if (reader != null) {
            reader.setState(ENDING);
            reader = null;
        }
        imgListener = listener;
        listenImage = image;
        if (client != 0) {
            reader = new ReadThread(listener, image);
            reader.setDaemon(true);
            reader.start();
        }
    }


   /**
    ***************************************************************************
    **
    **  Clears the data listener.
    **
    **  Clears the data listener object if it exists.
    **
    ***************************************************************************
    */
    public void clearListener()
    {
        if (reader != null) {
            reader.setState(ENDING);
            reader = null;
        }
        imgListener = null;
        listenImage = null;
    }


   /**
    ***************************************************************************
    **
    **  Performs one-time initialization.
    **
    **  Caches various field IDs for subsequent use.
    **
    ***************************************************************************
    */
    private native static void initSys();


   /**
    ***************************************************************************
    **
    **  Creates a new object.
    **
    **  Creates needed C++ objects.
    **
    **  @param  id  The ID of the RCM to connect to
    **
    ***************************************************************************
    */
    private native void newImageClient(int id);


   /**
    ***************************************************************************
    **
    **  Deletes an object.
    **
    **  Deletes referenced C++ objects.
    **
    ***************************************************************************
    */
    private native void deleteImageClient();


   /**
    ***************************************************************************
    **
    **  Awaits an image.
    **
    **  Waits until a new image has been generated.
    **
    **  @param  image  An Image object in which to save the reference and
    **                 metadata for the new image, or null if a new image
    **                 object is to be created.
    **
    **  @return  The Image object containing the new image reference and
    **           metadata.
    **
    ***************************************************************************
    */
    public native Image waitForImage(Image image);


   /**
    ***************************************************************************
    **
    **  Reads an image.
    **
    **  Gets the pixel data for an image.
    **
    **  @param  image  The Image object containing the valid metadata for an
    **                 image.  If the contained pixel data byte array is not
    **                 null and is large enough, the pixel data us stored
    **                 there.  Otherwise a new array is created.
    **
    **  @return  True if the pixel data was successfully read, false otherwise.
    **
    ***************************************************************************
    */
    public native boolean getImage(Image image);


   /**
    ***************************************************************************
    **
    **  Checks that connection is open
    **
    **  @exception  RcmException 
    **
    ***************************************************************************
    */
    private void checkOpen() throws RcmException
    {
        if (client == 0) {
            throw new RcmException("Connection not open");
        }
    }

}
