package org.lsst.ccs.drivers.pluto;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.modbus.TestModbus;

/**
 ****************************************************
 * 
 *  Program to test the Pluto gateway device driver
 * 
 *  @author Owen Saxton
 * 
 ****************************************************
 */
public class TestPluto extends TestModbus {

    private final Pluto plu;

    //public enum OnOff {
    //    OFF, ON;
    //}


   /**
    *  Constructor
    */
    public TestPluto()
    {
        super(new Pluto());
        plu = (Pluto)mod;
    }


   /**
    *  Opens connection to a device.
    *
    *  @param  node  The network node name
    *
    *  @throws  DriverException
    */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="node", description="Network node name")
                     String node) throws DriverException
    {
        plu.open(node);
    }


   /**
    *  Opens connection to a device.
    *
    *  @param  node  The network node name
    *
    *  @param  port  The port number
    *
    *  @throws  DriverException
    /
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="node", description="Network node name")
                     String node,
                     @Argument(name="port", description="Port number")
                     int port) throws DriverException
    {
        plu.open(node, port);
    }
    */


   /**
    *  Shows the module status.
    * 
    *  @throws  DriverException
    */
    @Command(name="showstatus", description="Show the module status")
    public void showStatus() throws DriverException
    {
        out.format("Node status = 0x%08x\n", plu.readModuleStatus());
    }


   /**
    *  Reads global data from a Pluto module.
    * 
    *  @param  pluto  The Pluto ID
    *
    *  @throws  DriverException
    */
    @Command(name="readglobal", description="Read global data")
    public void readGlobal(@Argument(name="pluto", description="Pluto ID")
                           int pluto) throws DriverException
    {
        out.format("Value = 0x%08x\n", plu.readGlobalData(pluto));
    }


   /**
    *  Reads additional data.
    * 
    *  @param  area  The data area number
    *
    *  @throws  DriverException
    */
    @Command(name="readadditional", description="Read additional data")
    public void readAdditional(@Argument(name="area", description="Area number")
                               int area) throws DriverException
    {
        out.format("Value = 0x%08x\n", plu.readAdditionalData(area));
    }


   /**
    *  Starts a configuration.
    * 
    *  @param  nodeMask   Bit mask of nodes
    *
    *  @param  areaMask   Bit mask of areas
    *
    *  @param  timeout    Timeout (ms)
    *
    *  @param  cycleTime  Cycle time (ms)
    */
    @Command(name="configstart", description="Start a configuration")
    public void readAdditional(@Argument(name="nodemask",
                                         description="Bit mask of nodes")
                               int nodeMask,
                               @Argument(name="areamask",
                                         description="Bit mask of areas")
                               int areaMask,
                               @Argument(name="timeout",
                                         description="To Pluto timeout (ms)")
                               int timeout,
                               @Argument(name="cycletime",
                                         description="To Pluto cycle time")
                               int cycleTime)
    {
        plu.configStart(nodeMask, areaMask, timeout, cycleTime);
    }


   /**
    *  Configures a data area.
    * 
    *  @param  area    Area number
    *
    *  @param  pluto   Pluto number
    *
    *  @param  dataId  Data ID
    */
    @Command(name="configarea", description="Configure a data area")
    public void configArea(@Argument(name="area",
                                     description="Data area number")
                           int area,
                           @Argument(name="pluto",
                                     description="Pluto number")
                           int pluto,
                           @Argument(name="dataid",
                                     description="Data ID")
                           int dataId)
    {
        plu.configDataArea(area, pluto, dataId);
    }


   /**
    *  Writes the configuration.
    * 
    *  @throws  DriverException
    */
    @Command(name="configwrite", description="Write the configutation")
    public void configWrite() throws DriverException
    {
        plu.configWrite();
    }


   /**
    *  Writes a bit to a data area.
    * 
    *  @param  area   The area number
    *
    *  @param  bit    The bit number
    *
    *  @param  value  The bit value
    *
    *  @throws  DriverException
    */
    @Command(name="writebit", description="Write a bit to an area")
    public void writeBit(@Argument(name="area",
                                   description="Data area number")
                         int area,
                         @Argument(name="bit",
                                   description="Bit number")
                         int bit,
                         @Argument(name="value",
                                   description="Bit value")
                         int value) throws DriverException
    {
        plu.writeAreaBit(area, bit, value);
    }


   /**
    *  Writes a register to a data area.
    * 
    *  @param  area   The area number
    *
    *  @param  reg    The register number
    *
    *  @param  value  The register value
    *
    *  @throws  DriverException
    */
    @Command(name="writeregister", description="Write a register to an area")
    public void writeRegister(@Argument(name="area",
                                        description="Data area number")
                              int area,
                              @Argument(name="register",
                                        description="Register number")
                              int reg,
                              @Argument(name="value",
                                        description="Register value")
                              int value) throws DriverException
    {
        plu.writeAreaRegister(area, reg, value);
    }


   /**
    *  Reads local data from a Pluto module.
    * 
    *  @param  pluto  The Pluto ID
    *
    *  @param  type   The type of data
    *
    *  @param  addr   The data address
    *
    *  @throws  DriverException
    */
    @Command(name="readlocal", description="Read local data")
    public void readLocal(@Argument(name="pluto", description="Pluto ID")
                          int pluto,
                          @Argument(name="type", description="Data type")
                          int type,
                          @Argument(name="address", description="Data address")
                          int addr) throws DriverException
    {
        readLocal(pluto, type, addr, 1);
    }


   /**
    *  Reads local data from a Pluto module.
    * 
    *  @param  pluto  The Pluto ID
    *
    *  @param  type   The type of data
    *
    *  @param  addr   The first data address
    *
    *  @param  count  The number of addresses
    *
    *  @throws  DriverException
    */
    @Command(name="readlocal", description="Read local data")
    public void readLocal(@Argument(name="pluto", description="Pluto ID")
                          int pluto,
                          @Argument(name="type", description="Data type")
                          int type,
                          @Argument(name="address", description="Data address")
                          int addr,
                          @Argument(name="count", description="Number of addresses")
                          int count) throws DriverException
    {
        String sep = "";
        for (int j = 0; j < count; j++) {
            try {
                int value = plu.readLocalData(pluto, type, addr + j);
                if ((j & 3) == 0) {
                    out.format("%s%4s:", sep, addr + j);
                    sep = "\n";
                }
                out.format(" %08x", value);
                Thread.sleep(j < count - 1 ? 500 : 0);
            }
            catch (DriverException e) {
                if (j > 0) {
                    out.println();
                }
                throw e;
            }
            catch (InterruptedException e) {
            }
        }
        if (count > 0) {
            out.println();
        }
    }

}
