package org.lsst.ccs.drivers.pluto;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.modbus.Modbus;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *************************************************
 *
 *  Routines for controlling a Pluto PLC system.
 *
 *  @author  Owen Saxton
 *
 *************************************************
 */
public class Pluto extends Modbus {

   /**
    *  Public constants.
    */
    public static final int
        NUM_MODULES               = 32,
        NUM_ADD_AREAS             = 32,
        NUM_DTP_AREAS             = 4,
        DATA_TYPE_GLOBAL          = 0,
        DATA_TYPE_LOCAL_MEMORY    = 1,
        DATA_TYPE_LOCAL_REGISTER  = 2,
        DATA_TYPE_LOCAL_PARAMETER = 3;

   /**
    *  Private data.
    */
    private static final short
        ZERO = 0,
        ONE  = 1,
        TWO  = 2,
        SA_DATA_TO_PLUTO   = 1,
        SA_LOCAL_DATA_RQST = 2,
        SA_PASS_THRU_RQST  = 3,
        SA_GATEWAY_CONFIG  = 4,
        SA_DATA_FROM_PLUTO = 33,
        SA_LOCAL_DATA_RESP = 34,
        SA_PASS_THRU_RESP  = 35,

        DFP_STATUS_ADDR    = 1,
        DFP_DATA_ADDR      = 3,
        DFP_ADDL_DATA_ADDR = 67,

        CFG_LENG_ADDR      = 0,
        CFG_ENAB_DTP_ADDR  = 1,
        CFG_DTP_TMO_ADDR   = 2,
        CFG_EXP_NODES_ADDR = 3,
        CFG_ADDL_DATA_ADDR = 5,
        CFG_DTP_CYCLE_TIME = 37,
        CFG_DATA_LENG      = 38,

        DTP_LENG_ADDR  = 0,
        DTP_VALID_ADDR = 1,
        DTP_BITS_ADDR  = 2,
        DTP_REGS_ADDR  = 3,
        DTP_AREA_LENG  = 3,

        LDRQ_LENG_ADDR  = 0,
        LDRQ_PLUTO_ADDR = 1,
        LDRQ_TYPE_ADDR  = 2,
        LDRQ_ADDR_ADDR  = 3,
        LD_REQUEST_LENG = 4,

        LDRP_LENG_ADDR   = 0,
        LDRP_PLUTO_ADDR  = 1,
        LDRP_TYPE_ADDR   = 2,
        LDRP_ADDR_ADDR   = 3,
        LDRP_ERROR_ADDR  = 4,
        LDRP_DATA_ADDR   = 5,
        LD_RESPONSE_LENG = 7,
        LDRP_ERROR_OK       = 1,
        LDRP_ERROR_TIMEOUT  = 2,
        LDRP_ERROR_BAD_DATA = 4,
        LDRP_ERROR_UNKNOWN  = 8;

    private static final Map<Short, String> ldErrorMap = new HashMap<>();
    static {
        ldErrorMap.put(LDRP_ERROR_TIMEOUT, "timeout");
        ldErrorMap.put(LDRP_ERROR_BAD_DATA, "bad data");
        ldErrorMap.put(LDRP_ERROR_UNKNOWN, "unknown");
    }
    private final short[] configData = new short[CFG_DATA_LENG];


   /**
    *  Constructor.
    */
    public Pluto()
    {
        super(Option.NO_SERIAL);
    }


   /**
    *  Opens a connection to the device.
    *
    *  @param  type   The connection type
    *
    *  @param  ident  The host name or IP address
    *
    *  @param  parm1  The port number
    *
    *  @param  parm2  Unused
    *
    *  @throws  DriverException
    */
    @Override
    public void open(ConnType type, String ident, int parm1, int parm2)
        throws DriverException
    {
        super.open(type, ident, parm1, parm2);
        super.setAddressMode(true);
    }


   /**
    *  Opens a connection to the device.
    *
    *  @param  ident  The host name or IP address
    *
    *  @throws  DriverException
    */
    public void open(String ident) throws DriverException
    {
        open(ConnType.NET, ident);
    }


   /**
    *  Opens a connection to the device.
    *
    *  @param  ident  The host name or IP address
    *
    *  @param  parm   The port number
    *
    *  @throws  DriverException
    */
    public void open(String ident, int parm) throws DriverException
    {
        open(ConnType.NET, ident, parm);
    }


   /**
    *  Sets whether entity numbers are direct addresses.
    *
    *  This routine is a no-op since address mode is always set and should
    *  not be changed.
    *
    *  @param  mode  Whether or not address mode is to be set
    */
    @Override
    public void setAddressMode(boolean mode)
    {
    }


   /**
    *  Reads module status.
    *
    *  @return  The 32-bit status word with a bit set for each module present
    *
    *  @throws  DriverException
    */
    public int readModuleStatus() throws DriverException
    {
        return toInteger(readRegisters(SA_DATA_FROM_PLUTO, DFP_STATUS_ADDR,
                                       TWO));
    }


   /**
    *  Reads global data.
    *
    *  @param  pluto  The pluto number of the module (0 - 31)
    *
    *  @return  The 32-bit data value
    *
    *  @throws  DriverException
    */
    public int readGlobalData(int pluto) throws DriverException
    {
        return toInteger(readRegisters(SA_DATA_FROM_PLUTO,
                                       (short)(DFP_DATA_ADDR + 2 * pluto), TWO));
    }


   /**
    *  Reads additional data.
    *
    *  @param  area  The area number of the data (0 - 31)
    *
    *  @return  The 32-bit additional data value
    *
    *  @throws  DriverException
    */
    public int readAdditionalData(int area) throws DriverException
    {
        return toInteger(readRegisters(SA_DATA_FROM_PLUTO,
                                       (short)(DFP_ADDL_DATA_ADDR + 2 * area),
                                       TWO));
    }


   /**
    *  Starts a gateway configuration.
    *
    *  @param  nodeMask   Bit mask of expected nodes
    *
    *  @param  areaMask   Bit mask of enabled data areas (0 - 15)
    *
    *  @param  timeout    Data to pluto timeout (ms)
    *
    *  @param  cycleTime  Data to pluto cycle time (ms)
    */
    public void configStart(int nodeMask, int areaMask, int timeout,
                            int cycleTime)
    {
        configData[CFG_LENG_ADDR] = CFG_DATA_LENG - 1;
        configData[CFG_EXP_NODES_ADDR] = (short)(nodeMask & 0xffff);
        configData[CFG_EXP_NODES_ADDR + 1] = (short)(nodeMask >> 16);
        configData[CFG_ENAB_DTP_ADDR] = (short)areaMask;
        configData[CFG_DTP_TMO_ADDR] = (short)timeout;
        configData[CFG_DTP_CYCLE_TIME] = (short)cycleTime;
    }


   /**
    *  Configures an additional data area.
    *
    *  @param  area    The data area number (0 - 31)
    *
    *  @param  pluto   The pluto number
    *
    *  @param  dataId  The ID of the data
    */
    public void configDataArea(int area, int pluto, int dataId)
    {
        configData[CFG_ADDL_DATA_ADDR + (area & 0x1f)]
          = (short)((pluto << 8) | (dataId & 0xff));
    }


   /**
    *  Writes the configuration.
    *
    *  @throws  DriverException
    */
    public void configWrite() throws DriverException
    {
        writeRegisters(SA_GATEWAY_CONFIG, ZERO, configData);
    }


   /**
    *  Writes a bit to a data-to-Pluto area.
    *
    *  @param  area   The area number (0 - 3)
    *
    *  @param  bit    The bit number (0 - 15)
    *
    *  @param  value  The value to set (0 or 1)
    *
    *  @throws  DriverException
    */
    public void writeAreaBit(int area, int bit, int value)
        throws DriverException
    {
        area &= 0x03;
        bit &= 0x0f;
        short addr = (short)(DTP_BITS_ADDR + DTP_AREA_LENG * area);
        short bitVal = readRegisters(SA_DATA_TO_PLUTO, addr, ONE)[0];
        if ((value & 1) != 0) {
            bitVal |= 1 << bit;
        }
        else {
            bitVal &= ~(1 << bit);
        }
        writeRegister(SA_DATA_TO_PLUTO, addr, bitVal);
        writeRegisters(SA_DATA_TO_PLUTO, DTP_LENG_ADDR,
                       new short[]{ONE, (short)(1 << area)});
    }


   /**
    *  Writes all bits to a data-to-Pluto area.
    *
    *  @param  area   The area number (0 - 3)
    *
    *  @param  value  The value to set
    *
    *  @throws  DriverException
    */
    public void writeAreaBits(int area, int value) throws DriverException
    {
        area &= 0x03;
        short addr = (short)(DTP_BITS_ADDR + DTP_AREA_LENG * area);
        writeRegister(SA_DATA_TO_PLUTO, addr, (short)value);
        writeRegisters(SA_DATA_TO_PLUTO, DTP_LENG_ADDR,
                       new short[]{ONE, (short)(1 << area)});
    }


   /**
    *  Reads a bit from a data-to-Pluto area.
    *
    *  @param  area   The area number (0 - 3)
    *
    *  @param  bit    The bit number (0 - 15)
    *
    *  @return  The bit value
    *
    *  @throws  DriverException
    */
    public int readAreaBit(int area, int bit) throws DriverException
    {
        area &= 0x03;
        bit &= 0x0f;
        short addr = (short)(DTP_BITS_ADDR + DTP_AREA_LENG * area);
        int value = readRegisters(SA_DATA_TO_PLUTO, addr, ONE)[0];
        return (value >> bit) & 1;
    }


   /**
    *  Reads all bits from a data-to-Pluto area.
    *
    *  @param  area   The area number (0 - 3)
    *
    *  @return  The read value
    *
    *  @throws  DriverException
    */
    public int readAreaBits(int area) throws DriverException
    {
        area &= 0x03;
        short addr = (short)(DTP_BITS_ADDR + DTP_AREA_LENG * area);
        return readRegisters(SA_DATA_TO_PLUTO, addr, ONE)[0];
    }


   /**
    *  Writes a register to a data-to-Pluto area.
    *
    *  @param  area   The area number (0 - 3)
    *
    *  @param  reg    The register number (0 or 1)
    *
    *  @param  value  The value to set (16-bit)
    *
    *  @throws  DriverException
    */
    public void writeAreaRegister(int area, int reg, int value)
        throws DriverException
    {
        area &= 0x03;
        short addr = (short)(DTP_REGS_ADDR + DTP_AREA_LENG * area + (reg & 1));
        writeRegister(SA_DATA_TO_PLUTO, addr, (short)value);
        writeRegisters(SA_DATA_TO_PLUTO, DTP_LENG_ADDR,
                       new short[]{ONE, (short)(1 << area)});
    }


   /**
    *  Reads a register from a data-to-Pluto area.
    *
    *  @param  area   The area number (0 - 3)
    *
    *  @param  reg    The register number (0 or 1)
    *
    *  @return  The register value
    *
    *  @throws  DriverException
    */
    public int readAreaRegister(int area, int reg)
        throws DriverException
    {
        short addr = (short)(DTP_REGS_ADDR + DTP_AREA_LENG * (area & 0x03)
                               + (reg & 0x01));
        return readRegisters(SA_DATA_TO_PLUTO, addr, ONE)[0];
    }


   /**
    *  Reads local data.
    *
    *  @param  pluto  The pluto number of the module (0 - 31)
    *
    *  @param  type   The type of data to read (0 - 3)
    *
    *  @param  addr   The address of the data
    *
    *  @return  The 32-bit data value
    *
    *  @throws  DriverException
    */
    public int readLocalData(int pluto, int type, int addr)
        throws DriverException
    {
        int value = readRegisters(SA_LOCAL_DATA_RQST, ZERO, ONE)[0];
        if (value != 0) {
            throw new DriverException("Gateway busy");
        }
        short[] rqst = new short[LD_REQUEST_LENG];
        rqst[LDRQ_LENG_ADDR] = LD_REQUEST_LENG - 1;
        rqst[LDRQ_PLUTO_ADDR] = (short)pluto;
        rqst[LDRQ_TYPE_ADDR] = (short)type;
        rqst[LDRQ_ADDR_ADDR] = (short)addr;
        writeRegister(SA_LOCAL_DATA_RESP, ZERO, ZERO);
        writeRegisters(SA_LOCAL_DATA_RQST, ZERO, rqst);
        short[] reply = null;
        for (int j = 1; j <= 100; j++) {
            reply = readRegisters(SA_LOCAL_DATA_RESP, ZERO, LD_RESPONSE_LENG);
            if (reply[LDRP_LENG_ADDR] != 0) break;
        }
        if (reply[LDRP_LENG_ADDR] == 0) {
            throw new DriverException("No response received");
        }
        if (reply[LDRP_PLUTO_ADDR] != rqst[LDRQ_PLUTO_ADDR]) {
            throw new DriverException("Incorrect Pluto ID in response");
        }
        if (reply[LDRP_TYPE_ADDR] != rqst[LDRQ_TYPE_ADDR]) {
            throw new DriverException("Incorrect data type in response");
        }
        if (reply[LDRP_ADDR_ADDR] != rqst[LDRQ_ADDR_ADDR]) {
            throw new DriverException("Incorrect data address in response");
        }
        short code = reply[LDRP_ERROR_ADDR];
        if (code != LDRP_ERROR_OK) {
            String desc = ldErrorMap.get(code);
            String text = "Response error: "
                            + (desc != null ? desc
                                            : "unrecognized (" + code + ")");
            throw new DriverException(text);
        }

        return toInteger(reply, LDRP_DATA_ADDR);
    }


   /**
    *  Converts a two-element short array to an integer.
    *
    *  @param  value  The array of two shorts, in big-endian order
    *
    *  @return  The converted value
    */
    private static int toInteger(short[] value)
    {
        return toInteger(value, 0);
    }


   /**
    *  Converts two elements of a short array to an integer.
    *
    *  @param  value   The array of shorts
    *
    *  @param  offset  The offset to the first short of the pair
    *
    *  @return  The converted value
    */
    private static int toInteger(short[] value, int offset)
    {
        return (value[offset] << 16) | (value[offset + 1] & 0xffff);
    }

}
