package org.lsst.ccs.drivers.pfeiffer;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.ascii.TestAscii;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 * Program to test the TPG261 device driver
 *
 * @author Homer
 */
public class TestTPG361 extends TestAscii {

    private final String[]
        readStat = {"OK", "Under range", "Over range", "Sensor error", "Sensor off", "No sensor", "ID error"};
    private final TPG361 mydev;

    public TestTPG361() {
        super(new TPG361());
        mydev = (TPG361)dev;
    }

    @Command(name = "getdeviceinfo", description = "Get info about the device")
    public String getDeviceInfo() throws DriverException {
        String[] info = mydev.getDeviceInfo();
        return String.format("Type:        %s\n" +
                             "Model:       %s\n" +
                             "Serial:      %s\n" +
                             "HW version:  %s\n" +
                             "FW version:  %s",
                             info[TPG361.DEV_INFO_TYPE], info[TPG361.DEV_INFO_MODEL],
                             info[TPG361.DEV_INFO_SERIAL], info[TPG361.DEV_INFO_HW_VER],
                             info[TPG361.DEV_INFO_FW_VER]);
    }

    @Command(name = "getgaugeid", description = "Get ID of the vacuum gauge")
    public String getGaugeID() throws DriverException {
        return mydev.getGaugeID();
    }

    @Command(name = "setunit", description = "Set the pressure unit")
    public void setUnit(@Argument(description="The unit") TPG361.Unit unit) throws DriverException {
         mydev.setUnit(unit);
    }

    @Command(name = "getunit", description = "Get the pressure unit")
    public TPG361.Unit getUnit() throws DriverException {
        return mydev.getUnit();
    }

    @Command(name = "geterror", description = "Get the error status")
    public String getError() throws DriverException {
        return mydev.getError();
    }

    @Command(name = "gaugeon", description = "Turn the vacuum gauge on")
    public void gaugeOn() throws DriverException {
        mydev.gaugeOn();
    }

    @Command(name = "gaugeoff", description = "Turn the vacuum gauge off")
    public void gaugeOff() throws DriverException {
        mydev.gaugeOff();
    }

    @Command(name = "getgaugestatus", description = "Get status of the vacuum gauge")
    public TPG361.GaugeStatus getGaugeStatus() throws DriverException {
        return mydev.getGaugeStatus();
    }

    @Command(name = "readpressure", description = "Read pressure ")
    public String readPressure() throws DriverException {
        double[] press = mydev.readPressure();
        return String.format("%s (%s)", press[0], readStat[(int)press[1]]);
    }

    @Command(name = "readtemperature", description = "Read the temperature")
    public double readTemperature() throws DriverException {
        return mydev.readTemperature();
    }

    @Command(name = "gethours", description = "Get the operating hours")
    public int getHours() throws DriverException {
        return mydev.getOperatingHours();
    }

    @Command(name = "writecommand", description = "Write a command")
    public void writeCommand(@Argument(description="The commmand to send") String command) throws DriverException {
         mydev.writeCommand(command);
    }

    @Command(name = "readresponse", description = "Read a response")
    public String readResponse() throws DriverException {
         return mydev.readResponse();
    }

    @Command(name = "readstring", description = "Write a command and read response")
    public String readString(@Argument(description="The commmand to send") String command) throws DriverException {
         return mydev.readString(command);
    }

}
