package org.lsst.ccs.drivers.pfeiffer;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.command.annotations.Argument;

/**
 *  General access routines for the Pfeiffer ASM380 helium leak detector.
 *
 *  @author Owen Saxton
 */
public class ASM380 extends Ascii {

    /**
     *  Public constants
     */
    public static final int
        ST_FILAMENT_2      = 0x0001,
        ST_FILAMENT_ON     = 0x0002,
        ST_IN_CYCLE        = 0x0004,
        ST_CYCLE_TYPE      = 0x0018,
        ST_SNIFF_TEST      = 0x0020,
        ST_AUTOCAL_OK      = 0x0040,
        ST_PANEL_UNLOCKED  = 0x0080,
        ST_NO_FAULTS       = 0x0100,
        ST_INLET_VENT      = 0x0200,
        ST_CYCLE_AVAIL     = 0x0400,
        ST_HIGH_VAC_SYNC   = 0x0800,
        ST_SNIFF_UNCLOGGED = 0x4000;

    public static final int
        IX_FILAMENT_2      = 0,
        IX_FILAMENT_ON     = 1,
        IX_IN_CYCLE        = 2,
        IX_CYCLE_TYPE      = 3,
        IX_SNIFF_TEST      = 4,
        IX_AUTOCAL_OK      = 5,
        IX_PANEL_UNLOCKED  = 6,
        IX_NO_FAULTS       = 7,
        IX_INLET_VENT      = 8,
        IX_CYCLE_AVAIL     = 9,
        IX_HIGH_VAC_SYNC   = 10,
        IX_SNIFF_UNCLOGGED = 11,
        NUM_STATUS_ITEMS   = 12;
        
    /**
     *  Private constants
     */
    private static final double READ_TIMEOUT = 1.0;
    private static final int BAUD_RATE = 9600;
    private static final byte ACK = 0x06, NAK = 0x15, CR = 0x0d;
    private static String[] CYCLE_TYPES = {"atmos", "grossLeak", "normal", "highSens"};
    private static final Map<Integer, String> STATUS_DESCS = new HashMap<>();
    static {
        STATUS_DESCS.put(IX_FILAMENT_2, "Active filament:");
        STATUS_DESCS.put(IX_FILAMENT_ON, "Filament on:");
        STATUS_DESCS.put(IX_IN_CYCLE, "In cycle:");
        STATUS_DESCS.put(IX_CYCLE_TYPE, "Cycle type:");
        STATUS_DESCS.put(IX_SNIFF_TEST, "Sniff test:");
        STATUS_DESCS.put(IX_AUTOCAL_OK, "Auto calib okay:");
        STATUS_DESCS.put(IX_PANEL_UNLOCKED, "Panel locked:");
        STATUS_DESCS.put(IX_NO_FAULTS, "Errors active:");
        STATUS_DESCS.put(IX_INLET_VENT, "Inlet vent:");
        STATUS_DESCS.put(IX_CYCLE_AVAIL, "Cycle available:");
        STATUS_DESCS.put(IX_HIGH_VAC_SYNC, "High vac sync:");
        STATUS_DESCS.put(IX_SNIFF_UNCLOGGED, "Sniffer clogged:");
    }

    /**
     *  Private fields
     */


    /**
     *  Constructor
     */
    public ASM380() {
        setOptions(Option.NO_NET);
    }


    /**
     *  Opens a connection.
     *
     *  This is the main open method and overrides the Ascii one, adding additional functionality
     *
     *  @param connType The connection type
     *  @param ident The device identifier
     *  @param parm1 The first parameter (ignored)
     *  @param parm2 The second parameter (ignored)
     *  @throws DriverException
     */
    @Override
    public void open(ConnType connType, String ident, int parm1, int parm2) throws DriverException {
        super.open(connType, ident, BAUD_RATE, 0);
        try {
            setTimeout(READ_TIMEOUT);
            setCommandTerm(Terminator.CR);
            String version = getVersion();
            if (!version.startsWith("ASM380")) {
                throw new DriverException("Unrecognized version string: " + version);
            }
        }
        catch (DriverException e) {
            closeSilent();
            throw e;
        }
    }


    /**
     *  Gets the software version.
     * 
     *  @return  The version string
     *  @throws  DriverException 
     */
    public String getVersion() throws DriverException {
        return writeCommand("?MD");
    }


    /**
     *  Gets the instrument status.
     * 
     *  @return  The status word
     *  @throws  DriverException 
     */
    public int getStatus() throws DriverException {
        String resp = writeCommand("?ST");
        if (resp.length() == 5) {
            try {
                return Integer.parseInt(resp);
            }
            catch (NumberFormatException e) {}  // Handled below
        }
        throwResponseException(resp);
        return 0;
    }


    /**
     *  Decodes the instrument status into a string array.
     * 
     *  @return  The array of strings
     *  @throws DriverException 
     */
    public String[] decodeStatus() throws DriverException {
        int status = getStatus();
        String[] items = new String[NUM_STATUS_ITEMS];
        items[IX_FILAMENT_2] = String.format("%-17s %s", "Active filament:", (status & ST_FILAMENT_2) != 0 ? 2 : 1);
        items[IX_FILAMENT_ON] = String.format("%-17s %s", "Filament on:", (status & ST_FILAMENT_ON) != 0);
        items[IX_IN_CYCLE] = String.format("%-17s %s", "In cycle:", (status & ST_IN_CYCLE) != 0);
        items[IX_CYCLE_TYPE] = String.format("%-17s %s", "Cycle type:",
                                             CYCLE_TYPES[(status & ST_CYCLE_TYPE) >> Integer.numberOfTrailingZeros(ST_CYCLE_TYPE)]);
        items[IX_SNIFF_TEST] = String.format("%-17s %s", "Sniff test:", (status & ST_SNIFF_TEST) != 0);
        items[IX_AUTOCAL_OK] = String.format("%-17s %s", "Auto calib okay:", (status & ST_AUTOCAL_OK) != 0);
        items[IX_PANEL_UNLOCKED] = String.format("%-17s %s", "Panel locked:", (status & ST_PANEL_UNLOCKED) == 0);
        items[IX_NO_FAULTS] = String.format("%-17s %s", "Errors active:", (status & ST_NO_FAULTS) == 0);
        items[IX_INLET_VENT] = String.format("%-17s %s", "Inlet vent:", (status & ST_INLET_VENT) != 0 ? "open" : "closed");
        items[IX_CYCLE_AVAIL] = String.format("%-17s %s", "Cycle available:", (status & ST_CYCLE_AVAIL) != 0);
        items[IX_HIGH_VAC_SYNC] = String.format("%-17s %s", "High vac sync:", (status & ST_HIGH_VAC_SYNC) != 0);
        items[IX_SNIFF_UNCLOGGED] = String.format("%-17s %s", "Sniffer clogged:", (status & ST_SNIFF_UNCLOGGED) == 0);
       return items;
    }


    /**
     *  Reads the inlet pressure.
     * 
     *  @return  The pressure
     *  @throws DriverException 
     */
    public double readInletPressure() throws DriverException {
        return decodeCF(writeCommand("?PE"));
    }


    /**
     *  Reads the temperature.
     * 
     *  @return  The temperature
     *  @throws DriverException 
     */
    public int readTemperature() throws DriverException {
        String resp = writeCommand("?TE");
        if (resp.length() == 3) {
            try {
                return Integer.parseInt(resp.substring(0, 2));
            }
            catch (NumberFormatException e) {}  // Handled below
        }
        throwResponseException(resp);
        return 0;
    }


    /**
     *  Gets the primary pump hours.
     * 
     *  @return  The number of hours of operation
     *  @throws  DriverException
     */
    public int getPrimaryHours() throws DriverException {
        String resp = writeCommand("?MC0");
        if (resp.length() == 10) {
            try {
                return Integer.parseInt(resp.substring(0, 5));
            }
            catch (NumberFormatException e) {}  // Handled below
        }
        throwResponseException(resp);
        return 0;
    }


    /**
     *  Gets the high vacuum pump hours.
     * 
     *  @return  The number of hours of operation
     *  @throws  DriverException
     */
    public int getHighVacHours() throws DriverException {
        String resp = writeCommand("?MC1");
        if (resp.length() == 10) {
            try {
                return Integer.valueOf(resp.substring(0, 5));
            }
            catch (NumberFormatException e) {}  // Handled below
        }
        throwResponseException(resp);
        return 0;
    }


    /**
     *  Gets the array of warning codes.
     * 
     *  @return  The array of codes
     *  @throws  DriverException 
     */
    public int[] getWarningCodes() throws DriverException {
        return getCodes(false);
    }


    /**
     *  Gets the array of fault codes.
     * 
     *  @return  The array of codes
     *  @throws  DriverException 
     */
    public int[] getFaultCodes() throws DriverException {
        return getCodes(true);
    }


    /**
     *  Gets an array of warning or fault codes.
     * 
     *  @param  isFault  True for fault codes, false for warning
     *  @return  The array of codes
     *  @throws  DriverException 
     */
    private int[] getCodes(boolean isFault) throws DriverException {
        String resp = writeCommand(isFault ? "?ER" : "?WA");
        if (resp.length() > 0) {
            try {
                int count = Integer.parseInt(resp.substring(0, 1));
                if (resp.length() == 4 * count + 1) {
                    int[] codes = new int[count];
                    for (int j = 0; j < count; j++) {
                        codes[j] = Integer.parseInt(resp.substring(4 * j + 1, 4 * j + 5));
                    }
                    return codes;
                }
            }
            catch (NumberFormatException e) {}  // Handled below
        }
        throwResponseException(resp);
        return null;
    }


    /**
     *  Clears the stored warning codes.
     * 
     *  @throws DriverException 
     */
    public void clearWarnings() throws DriverException {
        writeCommand("!WA");
    }


    /**
     *  Clears the stored fault codes.
     * 
     *  @throws DriverException 
     */
    public void clearFaults() throws DriverException {
        writeCommand("!RE");
    }

    // additional commands for pumping and venting controls


    /**
     *  pumpOn
     * 
     *  @throws DriverException 
     */
    public void pumpOn(boolean on) throws DriverException {
	if (on) {
	    writeCommand("=CYE");
	} else {
	    writeCommand("=CYD");
	}
    }

    /**
     *  check pump status
     * 
     *  @throws DriverException 
     */
    public String getPumpStatus() throws DriverException {
        int status = getStatus();
	return (status & ST_IN_CYCLE) != 0 ? "pumping on" : "pumping off";
    }

    /**
     *  check cycle status
     * 
     *  @throws DriverException 
     */
    public String getCycleStatus() throws DriverException {
	String stat = writeCommand("?CY");
	String status = "";
	switch (stat) {
	    case "ST" :
		status = "start-up phase";
		break;
	    case "CZ" :
		status = "electronic zero calibration";
		break;
	    case "CM" :
		status = "other calibration";
		break;
	    case "HV" :
		status = "hard vacuum cycle";
		break;
	    case "SN" :
		status = "sniffing test mode";
		break;
	    default:
		status = "unexpected response: "+stat;
		break;
	    }
	return status;
    }

    /**
     *  enable alarm
     * 
     *  @throws DriverException 
     */
    public void enableAlarm(boolean enable) throws DriverException {
	if (enable) {
	    writeCommand("=HPE");
	} else {
	    writeCommand("=HPD");
	}
    }

    /**
     *  enable filament of the triode
     * 
     *  @throws DriverException 
     */
    public void enableFilament(boolean enable) throws DriverException {
	if (enable) {
	    writeCommand("=SCE");
	} else {
	    writeCommand("=SCD");
	}
    }

    /**
     *  reset the triode safety
     * 
     *  @throws DriverException 
     */
    public void resetTriodeSafety() throws DriverException {
	writeCommand("=SCR");
    }

    /**
     *  Execute open valve sequence
     * 
     *  @throws DriverException 
     */
    public void openVentValve(boolean open) throws DriverException {
	if (open) {
	    writeCommand("=IVE");
	} else {
	    writeCommand("=IVD");
	}
    }

    /**
     *  Setup open valve sequence
     * 
     *  @throws DriverException 
     */
    public void setVentValveParms(
				  @Argument(description="delay before actuation (0->2 secs)") int delay,
				  @Argument(description="time (seconds) in actuated state") int period) throws DriverException {
	int min = period/60;
	int sec = period - min*60;
	String command = "=IVPM"+delay+"E"+String.format("%02d%02d",min,sec) ; 
	writeCommand(command);

    }

    /**
     *  Setup open valve sequence
     * 
     *  @throws DriverException 
     */
    public void setVentValveNominal() throws DriverException {
	// manual mode, opening timer disabled
	writeCommand("=IVPM0D0000");
    }

    /**
     *  get vent valve setup
     * 
     *  @throws DriverException 
     */
    public String getVentValveParms() throws DriverException {
	String stat = writeCommand("?IVP") ;
	return("parameter settings string: " + stat + "format:xyzmmss x=A/M:auto/man opening|y:delay 0/1/2s|z=E/D:timer enable|mm:minutes open|ss:seconds open");
    }


    /**
     *  check vent valve status
     * 
     *  @throws DriverException 
     */
    public String getVentValveStatus() throws DriverException {
	return writeCommand("?IV").contains("E") ? "open" : "closed" ;
    }


    /**
     *  Writes a command and checks for error.
     * 
     *  Send a command and wait for its acknowledgment
     *  Transmit: command + CR  Receive: (response + CR + ACK) or ACK or NAK
     * 
     *  @param command The command
     *  @return  The command response
     *  @throws  DriverException
     */
    public synchronized String writeCommand(String command) throws DriverException {
        write(command);           // write string (plus terminator)
        byte[] readBuff = new byte[1024];
        int curr = 0, buffIn = 0;
        while (true) {
            try {
                buffIn += readBytes(readBuff, buffIn);
            }
            catch (DriverTimeoutException re) {
                throw re;
            }
            catch (DriverException re) {
                closeSilent();
                throw re;
            }
            while (curr < buffIn) {
                byte buffByte = readBuff[curr++];
                if (buffByte == ACK) {
                    //for (int j = 0; j < curr; j++) {
                    //    System.out.format("%02x ", readBuff[j]);
                    //}
                    //System.out.println();
                    return new String(readBuff, 0, curr > 1 && readBuff[curr - 2] == CR ? curr - 2 : curr - 1);
                }
                if (buffByte == NAK) {
                    throw new DriverException("Invalid command: " + command);
                }
            }
        }
    }


    /**
     *  Decode a compressed number format (CF) string.
     * 
     *  @param  cfData  The CF data
     *  @return  The decoded double precision number
     *  @throws DriverException 
     */
    private static double decodeCF(String cfData) throws DriverException {
        try {
            return Double.parseDouble(cfData.substring(0, 3) + "e" + cfData.substring(3));
        }
        catch(NumberFormatException e) {
            throw new DriverException("Invalid CF data: " + cfData);
        }
    }


    /**
     *  Throw an exception for unexpected command response.
     * 
     *  @param  resp  The response
     *  @throws DriverException 
     */
    private static void throwResponseException(String resp) throws DriverException {
        throw new DriverException("Unexpected command response: " + resp);
    }

}
