package org.lsst.ccs.drivers.pfeiffer;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.ascii.Ascii;

/**
 *  General access routines for the Pfeiffer TPG361/362 Vacuum Gauge Controller
 *
 *  @author Owen Saxton
 */
public class TPG361 extends Ascii {

    /**
     *  Public constants
     */
    public static final int
        DEV_INFO_TYPE   = 0,
        DEV_INFO_MODEL  = 1,
        DEV_INFO_SERIAL = 2,
        DEV_INFO_FW_VER = 3,
        DEV_INFO_HW_VER = 4,
        NUM_DEV_INFO    = 5;

    public enum Unit { MBAR, TORR, PASCAL, MICRON, HPASCAL, VOLT }  // Must be in this order

    public enum GaugeStatus { ABSENT, OFF, ON }  // Must be in this order

    /**
     *  Private constants
     */
    private static final double READ_TIMEOUT = 1.0;
    private static final String ACK = "\u0006", NAK = "\u0015";
    private static final byte ENQ = 0x05, ETX = 0x03;
    private static final Map<String, String> errorMap = new HashMap<>();
    static {
        errorMap.put("1000", "controller error");
        errorMap.put("0100", "no hardware");
        errorMap.put("0010", "invalid parameter");
        errorMap.put("0001", "syntax error");
    }

    /**
     *  Private fields
     */
    private int numGauge = 0;


    /**
     *  Constructor
     */
    public TPG361() {
        setOptions(Option.NO_NET);
    }


    /**
     *  Opens a connection.
     *
     *  This is the main open method and overrides the Ascii one, adding additional functionality
     *
     *  @param connType The connection type
     *  @param ident The device identifier
     *  @param parm1 The first parameter
     *  @param parm2 The second parameter
     *  @throws DriverException
     */
    @Override
    public void open(ConnType connType, String ident, int parm1, int parm2) throws DriverException {
        super.open(connType, ident, parm1, parm2);
        try {
            setTimeout(READ_TIMEOUT);
            setCommandTerm(Terminator.CR);
            setResponseTerm(Terminator.CRLF);
            String type = getDeviceInfo()[DEV_INFO_TYPE];
            if (type.startsWith("TPG36")) {
                numGauge = type.charAt(5) - '0';
            }
            if (numGauge < 1 || numGauge > 2) {
                throw new DriverException("Unrecognized device type: " + type);
            }
        }
        catch (DriverException e) {
            closeSilent();
            numGauge = 0;
            throw e;
        }
    }


    /**
     *  Gets device information.
     *
     *  @return  A 5-element string array of information
     *  @throws  DriverException
     */
    public String[] getDeviceInfo() throws DriverException {
        String rawInfo = readString("AYT");
        String[] devInfo = rawInfo.split(",");
        if (devInfo.length != NUM_DEV_INFO) {
            throw new DriverException("Unrecognized device information: " + rawInfo);
        }
        return devInfo;
    }


    /**
     *  Gets the ID of the first (or only) gauge.
     * 
     *  @return  The ID string
     *  @throws  DriverException
     */
    public String getGaugeID() throws DriverException {
        return getGaugeID(0);
    }


    /**
     *  Gets the ID of a gauge.
     *
     *  @param  gauge  The gauge number (0 or 1)
     *  @return  The ID string:
     *             TPR/PCR (Pirani Gauge or Pirani Capacitive gauge)
     *             IKR (Cold Cathode Gauge 10^-9 & 10^-11)
     *             PKR (FullRange CC Gauge)
     *             PBR (FullRange BA Gauge)
     *             IMR (Pirani / High Pressure Gauge)
     *             CMR/APR (Linear gauge)
     *             noSEn (no SEnsor)
     *             noid (no identifier)
     *  @throws  DriverException
     */
    public String getGaugeID(int gauge) throws DriverException {
        checkGauge(gauge);
        String resp = readString("TID");
        try {
            return resp.split(",")[gauge];
        }
        catch (IndexOutOfBoundsException e) {
            throw new DriverException("Unrecognized gauge ID response: " + resp);
        }
    }


    /**
     *  Sets the pressure unit.
     *
     *  @param  unit  The pressure unit
     *  @throws  DriverException
     */
    public void setUnit(Unit unit) throws DriverException {
        writeCommand("UNI," + unit.ordinal());
    }


    /**
     *  Gets the pressure unit.
     *
     *  @return  The pressure unit
     *  @throws  DriverException
     */
    public Unit getUnit() throws DriverException {
        String resp = readString("UNI");
        try {
            return Unit.values()[Integer.valueOf(resp)];
        }
        catch (NumberFormatException | IndexOutOfBoundsException e) {
            throw new DriverException("Unrecognized unit response: " + resp);
        }
    }


    /**
     *  Gets the error status.
     * 
     *  @return  The error string
     *  @throws  DriverException
     */
    public String getError() throws DriverException {
        return readString("ERR");
    }


    /**
     *  Turns the first (or only) gauge on.
     *
     *  @throws  DriverException
     */
    public void gaugeOn() throws DriverException {
        gaugeOn(0);
    }


    /**
     *  Turns a gauge on.
     *
     *  @param  gauge  The gauge number (0 or 1)
     *  @throws  DriverException
     */
    public void gaugeOn(int gauge) throws DriverException {
        checkGauge(gauge);
        writeCommand("SEN," + (numGauge == 1 ? "2" : gauge == 0 ? "2,0" : "0,2"));
    }


    /**
     *  Turns the first (or only) gauge off.
     *
     *  @throws  DriverException
     */
    public void gaugeOff() throws DriverException {
        gaugeOff(0);
    }


    /**
     *  Turns a gauge off.
     *
     *  @param  gauge  The gauge number (0 or 1)
     *  @throws  DriverException
     */
    public void gaugeOff(int gauge) throws DriverException {
        checkGauge(gauge);
        writeCommand("SEN," + (numGauge == 1 ? "1" : gauge == 0 ? "1,0" : "0,1"));
    }


    /**
     *  Gets the status of the first (or only) gauge.
     *
     *  @return  The status value
     *  @throws  DriverException
     */
    public GaugeStatus getGaugeStatus() throws DriverException {
        return getGaugeStatus(0);
    }


    /**
     *  Gets the status of a gauge.
     *
     *  @param  gauge  The gauge number (0 or 1)
     *  @return  The status enum: ABSENT, OFF or ON
     *  @throws  DriverException
     */
    public GaugeStatus getGaugeStatus(int gauge) throws DriverException {
        checkGauge(gauge);
        String resp = readString("SEN");
        try {
            return GaugeStatus.values()[Integer.valueOf(resp.split(",")[gauge])];
        }
        catch (IndexOutOfBoundsException | NumberFormatException e) {
            throw new DriverException("Unrecognized status information: " + resp);
        }
    }


    /**
     *  Reads the pressure from the first (or only) gauge.
     *
     *  @return  The two-element pressure array
     *  @throws  DriverException
     */
    public double[] readPressure() throws DriverException {
        return readPressure(0);
    }


    /**
     * Reads the pressure from a gauge.
     *
     *  @param  gauge  The gauge number (0 or 1)
     *  @return  A two-element array containing the pressure (in current units) [0]
     *           and the status [1]:
     *             0: Measurement data okay
     *             1: Under range
     *             2: Over range
     *             3: Sensor error
     *             4: Sensor off (IKR, PKR, IMR, PBR)
     *             5: No sensor (output: 5,2.0000E-2 [mbar])
     *             6: Identification error
     *  @throws  DriverException
     */
    public double[] readPressure(int gauge) throws DriverException {
        checkGauge(gauge);
        String resp = readString("PR" + (gauge + 1));
        try {
            String[] statValue = resp.split(",");
            double[] pressure = {0, 0};
            pressure[0] = Double.parseDouble(statValue[1]);
            pressure[1] = Double.parseDouble(statValue[0]);
            return pressure;
        }
        catch (IndexOutOfBoundsException | NumberFormatException e) {
            throw new DriverException("Unrecognized pressure response: " + resp);
        }
    }


    /**
     *  Reads the temperature.
     * 
     *  @return  The temperature (C)
     *  @throws  DriverException
     */
    public double readTemperature() throws DriverException {
        String resp = readString("TMP");
        try {
            return Double.parseDouble(resp);
        }
        catch (NumberFormatException e) {
            throw new DriverException("Unrecognized temperature response: " + resp);
        }
    }


    /**
     *  Gets the operating hours.
     * 
     *  @return  The number of hours of operation
     *  @throws  DriverException
     */
    public int getOperatingHours() throws DriverException {
        String resp = readString("RHR");
        try {
            return Integer.valueOf(resp);
        }
        catch (NumberFormatException e) {
            throw new DriverException("Unrecognized operating hours response: " + resp);
        }
    }


    /**
     *  Writes a command and checks for error.
     * 
     *  Send a command and wait for its acknowledgment
     *  Transmit: command + CR  Receive : (ACK or NAK) + CRLF
     * 
     *  @param command The command
     *  @throws  DriverException
     */
    public synchronized void writeCommand(String command) throws DriverException {
        write(command);           // write string (plus terminator)
        String reply = read();    // get the reply from the device (less terminator)
        if (reply.equals(NAK)) {
            String error = readResponse();
            String text = errorMap.get(error);
            if (text == null) {
                text = "invalid error string (" + error + ")";
            }
            throw new DriverException("Command error: " + text);
        }
        if (!reply.equals(ACK)) {
            throw new DriverException("Unrecognized command response: " + reply);
        }
    }


    /**
     *  Reads a response.
     *
     *  Transmit: ENQ  Receive: reply + CRLF
     *
     *  @return  The response string
     *  @throws  DriverException
     */
    public synchronized String readResponse() throws DriverException {
        writeBytes(new byte[]{ENQ});
        return read();
    }


    /**
     *  Sends a command and gets the response.
     *
     *  @param  command  The command string
     *  @return  The response string
     *  @throws  DriverException
     */
    public synchronized String readString(String command) throws DriverException {
        writeCommand(command);
        return readResponse();
    }


    /**
     *  Clears the input buffer.
     *
     *  Transmit: ETX
     *
     *  @throws  DriverException
     */
    public synchronized void clearInput() throws DriverException {
        writeBytes(new byte[]{ETX});
    }


    /**
     *  Checks whether a gauge number is valid
     *
     *  @param  gauge  The gauge number
     *  @throws  DriverException
     */
    private void checkGauge(int gauge) throws DriverException {
        if (gauge < 0 || gauge >= numGauge) {
            throw new DriverException("Invalid gauge number: " + gauge);
        }
    }

}
