package org.lsst.ccs.drivers.pfeiffer;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.ascii.Ascii;

/**
 * General access routines for the Pfeiffer TPG261 Vacuum Gauge Controller
 *
 * @author Peter Doherty
 */
public class TPG261 extends Ascii {

    /**
     * Public constants
     */
    public enum onOff {  /* not used presently */
        OFF, ON;
    }

    /**
     * Private constants
     */
    private String terminator = "\r\n";
    private int verbose = 1;


    /**
     * Opens a connection.
     *
     * @param serialname
     * @param baud
     * @throws DriverException
     */
    public void open(String serialname, int baud) throws DriverException {
        System.out.println("opening connection to the TPG261 Vacuum Gauge");
        open(Ascii.ConnType.SERIAL, serialname, baud);
    }

    public void init() throws DriverException {
        /* Send a CTRL-C here to clear the input buffer 
         * on the device, just to be sure 
         */
    }

    /**
     * Closes the connection.
     * 
     * @throws DriverException
     */
    public void close() throws DriverException {
//        port.closePort();
//        close();
    }

    /**
     * Get the TPG261 ID Transmit: TID <CR>[<LF>] Receive : <ACK><CR><LF>
     * Transmit: <ENQ>
     * Receive : x,x <CR><LF>
     * | |
     * | - Identification gauge 2 --- Identification gauge 2
     *
     * x = TPR (Pirani Gauge or Pirani Capacitive gauge 1)) IKR9 (Cold Cathode
     * Gauge 10-9) IKR11 (Cold Cathode Gauge 10-11) PKR (FullRange CC Gauge) PBR
     * (FullRange BA Gauge) IMR (Pirani / High Pressure Gauge) CMR (Linear
     * gauge) noSEn (no SEnsor) noid (no identifier)
     */
    public String getID() throws DriverException {
        /* try and get the ID string from the device */
        try {
            String idString = getReply("TID");
            if (this.verbose == 1) {
                System.out.println("TPG261ID: " + idString);
            }
            return idString;
        } catch (DriverException e) {
            System.out.println("TPG261 getID: Caught Exception " + e);
            return "";
        }
    }

    /**
     ** Get the TPG261 error status Transmit: ERR <CR>[<LF>] Receive:
     * <ACK><CR><LF>
     * Transmit: <ENQ>
     * Receive: xxxx <CR><LF>
     * ?? xxxx = 0000 ?> No error 1000 ?> Controller error 0100 ?> NO HWR No
     * hardware 0010 ?> PAR Inadmissible parameter 0001 ?> SYN Syntax error
     */
    public String getError() throws DriverException {
        /* try and get the Error status from the device */
        try {
            String errString = getReply("ERR");

            if (this.verbose == 1) {
                if (errString.charAt(0) == 1) {
                    System.out.println("TPG261 getError: Controller Error");
                }
                if (errString.charAt(1) == 1) {
                    System.out.println("TPG261 getError: Hardware Error");
                }
                if (errString.charAt(2) == 1) {
                    System.out.println("TPG261 getError: Parameter Error");
                }
                if (errString.charAt(3) == 1) {
                    System.out.println("TPG261 getError: Syntax Error");
                }
            }
            return errString;
        } catch (DriverException e) {
            System.out.println("TPG261 getError: Caught Exception" + e);
            return "";
        }
    }

    /**
     * Get the TPG261 status Transmit: SEN <CR>[<LF>] Receive : <ACK><CR><LF>
     * Transmit: <ENQ>
     * Receive : x,x <CR><LF>
     * ? ?
     * ? ?? Status gauge 2 ?? Status gauge 1
     *
     * x = 0 ?> Gauge cannot be turned on/off 1 ?> Gauge turned off 2 ?> Gauge
     * turned on
     */
    public String getStatus() throws DriverException {
        /* try and get the current status from the device */
        try {
            String statString = getReply("SEN");
            String[] subStrs = statString.split(",");
            Integer stat1 = Integer.valueOf(subStrs[0]);
            Integer stat2 = Integer.valueOf(subStrs[1]);
            if (this.verbose == 1) {
                switch (stat1) {
                    case 1:
                        System.out.println("TPG261 getStatus: Gauge 1 off");
                        break;
                    case 2:
                        System.out.println("TPG261 getStatus: Gauge 1 on");
                        break;
                }
                switch (stat2) {
                    case 1:
                        System.out.println("TPG261 getStatus: Gauge 2 off");
                        break;
                    case 2:
                        System.out.println("TPG261 getStatus: Gauge 2 on");
                        break;
                }
            }
            return statString;
        } catch (DriverException e) {
            System.out.println("TPG261 getStatus: Caught Exception");
            return "";
        }
    }

    /**
     * Gets the TPG261 pressure reading from gauge #1 (the only one we have).
     *
     * Transmit: PR1 <CR>[<LF>] Receive : <ACK><CR><LF> (0x6,0xD,0xA) Transmit:
     * <ENQ> (0x5) Receive : x,sx.xxxxEsxx <CR><LF>
     * _ ___________ ? ? ? ?? Measurement value ? [in current pressure unit] ?
     * ?? Status, x = 0 ?> Measurement data okay 1 ?> Underrange 2 ?> Overrange
     * 3 ?> Sensor error 4 ?> Sensor off (IKR, PKR, IMR, PBR) 5 ?> No sensor
     * (output: 5,2.0000E-2 [mbar]) 6 ?> Identification error
     */
    public double getPressure() throws DriverException {
        double pressure = 0.0;
        try {
            String replyString = getReply("PR1");
            String[] subStrs = replyString.split(",");
            Integer status = Integer.valueOf(subStrs[0]);
            pressure = Double.parseDouble(subStrs[1]);
            if (this.verbose == 1) {
                if (status == 0) {
                    System.out.println("TPG261 getPressure: OK");
                } else {
                    switch (status) {
                        case 1:
                            System.out.println("TPG261 getPressure: Under Range");
                            break;
                        case 2:
                            System.out.println("TPG261 getPressure: Over Range");
                            break;
                        case 3:
                            System.out.println("TPG261 getPressure: Sensor Error");
                            break;
                        case 4:
                            System.out.println("TPG261 getPressure: Sensor Off");
                            break;
                        case 5:
                            System.out.println("TPG261 getPressure: No Sensor");
                            break;
                        case 6:
                            System.out.println("TPG261 getPressure: ID Error");
                            break;
                        default:
                            System.out.println("TPG261 getPressure: unknown error");
                            break;
                    }
                }
            }
        } catch (DriverException e) {
            System.out.println("TPG261 getPressure: Caught Exception " + e);
            return 0.0 ;
        }
        return pressure;
    }

    /**
     * Turn a TPG261 gauge on
     *
     * Transmit: SEN [,x,x] <CR>[<LF>] | | ? * | | - Gauge 2 Status | |----
     * Gauge 1 Status
     *
     * x = 0 ?> No status change 1 ?> Turn gauge off 2 ?> Turn gauge on
     */
    public void gaugeOn(Integer gauge) throws DriverException {

        /* check for legitimate gauge number */
        if ((gauge < 1) || (gauge > 2)) {
            System.out.println("TPG261On: bad gauge number");
            return;
        }

        /* build the command string */
        String command = "";
        if (gauge == 1) {
            command = "SEN,2,0";
        } else {
            command = "SEN,0,2";
        }

        try {
            sendCommand(command);
        } catch (DriverException e) {
            System.out.println("TPG261On: Caught Exception");
        }
    }

    /**
     * Turn a TPG261 gauge off
     *
     * Transmit: SEN [,x,x] <CR>[<LF>] | | ? * | | - Gauge 2 Status | |----
     * Gauge 1 Status
     *
     * x = 0 ?> No status change 1 ?> Turn gauge off 2 ?> Turn gauge on
     */
    public void gaugeOff(Integer gauge) throws DriverException {

        /* check for legitimate gauge number */
        if ((gauge < 1) || (gauge > 2)) {
            System.out.println("TPG261On: bad gauge number");
            return;
        }

        /* build the command string */
        String cmd = "";
        if (gauge == 1) {
            cmd = "SEN,1,0";
        } else {
            cmd = "SEN,0,1";
        }

        try {
            sendCommand(cmd);
        } catch (DriverException e) {
            System.out.println("TPG261On: Caught Exception");
            return;
        }
    }

    /**
     * sendCommand(String command)
     *
     * Send a command, but do not wait for a reply other than to check for the
     * acknowledgment of that command
     *
     * Transmit: command <CR>[<LF>] Receive : 0x6 <CR><LF>
     */
    public void sendCommand(String command) throws DriverException {

        /* should check validity of string here */
        /* if being verbose, print out the command */
        if (this.verbose == 1) {
            System.out.println("TPG261Command: " + command);
        }
        /* write the command to the TPG261 */
        writeTPG261(command);         /* write string plus terminator  */

        String ack = readTPG261();    /* get the reply from the device */

        if (ack.charAt(0) != 0x6) {   /* check for a correct reply     */

            System.out.println("TPG261Command: Acknowledge error");
            throw new DriverException();
        }
        /* everything must be OK. Return is void */
    }

    /**
     * getReply(String command)
     *
     * Send a command, and wait for a reply. Return reply String
     *
     * Transmit: command <CR>[<LF>] Receive : 0x6 <CR><LF>
     * Transmit: 0x5 Receive : reply <CR><LF>
     */
    public String getReply(String command) throws DriverException {

        /* should check validity of string here */
        /* if being verbose, print out the command */
        if (this.verbose == 1) {
            System.out.println("TPG261Query: " + command);
        }
        /* write the command to the TPG261 */
        writeTPG261(command);
        String ack = readTPG261();    /* get the acknowledge from the device */

        if (ack.charAt(0) != 0x6) {   /* check for a correct reply           */

//            System.out.println("TPG261Query: Acknowledge error");
            throw new DriverException("PG261Query: Acknowledge error");
        }
        /* if ACK was OK, send query and get the response, return reply string*/
        writeTPG261(String.valueOf((char) 0x05));
        String reply = readTPG261();
        return reply;
    }

    /**
     * Writes a command.
     *
     * @param command The command to write, excluding terminator
     * @throws DriverException
     */
    public synchronized void writeTPG261(String command) throws DriverException {
        write(command + terminator);
    }

    /**
     * Reads a response.
     *
     * @return The command response string
     * @throws DriverException
     */
    public synchronized String readTPG261() throws DriverException {
        return (read());
    }

    /**
     * Reads a response after writing a command.
     *
     * @param command The command to write, excluding terminator
     * @return The command response string
     * @throws DriverException
     */
    public synchronized String readTPG261(String command) throws DriverException {
        writeTPG261(command);
        return readTPG261();
    }

}
