package org.lsst.ccs.drivers.pfeiffer;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.ascii.TestAscii;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 * Program to test the ASM380 device driver
 *
 * @author Saxton
 */
public class TestASM380 extends TestAscii {

    private final ASM380 mydev;

    public TestASM380() {
        super(new ASM380());
        mydev = (ASM380)dev;
    }

    @Command(description = "Show the version string")
    public String showVersion() throws DriverException {
        return mydev.getVersion();
    }

    @Command(description = "Show the instrument status")
    public String showStatus() throws DriverException {
        return String.format("0x%04x", mydev.getStatus());
    }

    @Command(description = "Show the decoded instrument status")
    public String showDecodedStatus() throws DriverException {
        String[] status = mydev.decodeStatus();
        StringBuilder text = new StringBuilder();
        for (int j = 0; j < status.length; j++) {
            if (j != 0) {
                text.append("\n");
            }
            text.append(status[j]);
        }
        return text.toString();
    }

    @Command(description = "Read the inlet pressure")
    public double readInletPressure() throws DriverException {
        return mydev.readInletPressure();
    }

    @Command(description = "Read the temperature")
    public int readTemperature() throws DriverException {
        return mydev.readTemperature();
    }

    @Command(description = "Show the primary pump operating hours")
    public int showPrimaryHours() throws DriverException {
        return mydev.getPrimaryHours();
    }

    @Command(description = "Show the high vacuum pump operating hours")
    public int showHighVacHours() throws DriverException {
        return mydev.getHighVacHours();
    }

    @Command(description = "Show the stored warning codes")
    public String showWarningCodes() throws DriverException {
        return showCodes(mydev.getWarningCodes());
    }

    @Command(description = "Show the stored fault codes")
    public String showFaultCodes() throws DriverException {
        return showCodes(mydev.getFaultCodes());
    }

    private static String showCodes(int[] codes) {
        if (codes.length == 0) {
            return "No codes stored";
        }
        else {
            StringBuilder text = new StringBuilder();
            for (int code : codes) {
                text.append(code).append(" ");
            }
            return text.toString();
        }
    }

    @Command(description = "Clear the stored warning codes")
    public void clearWarnings() throws DriverException {
        mydev.clearWarnings();
    }

    @Command(description = "Clear the stored fault codes")
    public void clearFaults() throws DriverException {
        mydev.clearFaults();
    }

    @Command(description = "set pump on (true) or off (false)")
    public void pumpOn(boolean on) throws DriverException {
        mydev.pumpOn(on);
    }

    @Command(description = "get the pumping status")
    public String getPumpStatus() throws DriverException {
        return mydev.getPumpStatus();
    }

    @Command(description = "enable the alarm (true) or not (false)")
    public void enableAlarm(boolean enable) throws DriverException {
        mydev.enableAlarm(enable);
    }

    @Command(description = "execute open valve sequence")
    public void openVentValve(boolean open) throws DriverException {
        mydev.openVentValve(open);
    }

    @Command(description = "get the vent valve status")
    public String getVentValveStatus() throws DriverException {
        return mydev.getVentValveStatus();
    }

    @Command(description = "get the vent valve actuation parameters")
    public String getVentValveParms() throws DriverException {
        return mydev.getVentValveParms();
    }

    @Command(description = "set valve opening sequence parameters")
    public void setVentValveParms(
		 @Argument(description="delay before actuation (0->2 secs)") int delay,
		 @Argument(description="time (mmss) in actuated state") int period) throws DriverException {
        mydev.setVentValveParms(delay,period);
    }

    @Command(description = "set valve opening sequence nominal")
    public void setVentValveNominal() throws DriverException {
        mydev.setVentValveNominal();
    }


    @Command(description = "Write a command")
    public String writeCommand(@Argument(description="The commmand to send") String command) throws DriverException {
         return mydev.writeCommand(command);
    }

}
