package org.lsst.ccs.drivers.parker;

/**
 * Defines names and numbers for some axis parameters that are of type FP32.
 * @author tether
 */
public enum AxisSingle {
    /** Motor drive feedback gain, proportional term (dimensionless). */
    PROPORTIONAL_GAIN(12304, 256),

    /** Motor drive feedback gain, integral term (dimensionless). */
    INTEGRAL_GAIN(12305, 256),

    /** Motor drive feedback gain, derivative term (dimensionless). */
    DERIVATIVE_GAIN(12308, 256),

    /** Excess error band  (EXC) positive limit. */
    PLUS_EXCESS_ERROR(12320, 256),

    /** Excess error band negative limit. */
    MINUS_EXCESS_ERROR(12321, 256),

    /** Motor drive torque limit for motion in the positive direction. */
    POSITIVE_TORQUE_LIMIT(12328, 256),

    /** Motor drive torque limit for motion in the negative direction. */
    NEGATIVE_TORQUE_LIMIT(12329, 256),

    /** Maximum velocity for the next jog motion (user units/sec). */
    JOG_VELOCITY_SETTING(12348, 256),

    /** Acceleration for the next jog motion (user units/sec/sec).
     * For a trapezoidal velocity profile this is the constant acceleration
     * used at the beginning of the motion. For S-curve motion the acceleration
     * will ramp from zero up to this value then ramp back down to zero.
     */
    JOG_ACCELERATION_SETTING(12349, 256),

    /** Deceleration for the next jog motion (user units/sec/sec). For a trapezoidal
     * velocity profile this is the constant deceleration used at the end of
     * the motion. For S-curve motion the deceleration will ramp from zero up
     * to this value then ramp back down to zero.
     */
    JOG_DECELERATION_SETTING(12350, 256),

    /**
     * The rate of ramp up and ramp down of acceleration and deceleration
     * for the next jog motion (user units/sec^3). Always zero for a trapezoidal velocity profile
     * and a positive value for S-curve motion.
     */
    JOG_JERK_SETTING(12351, 256),

    /** The number of encoder counts per user unit of distance. Doesn't need to be a whole
      * number and must be positive.
      */
    PULSES_PER_UNIT(12375, 256),

    /** The current value of drive current commanded by the controller. Must be updated
     via Drive Talk on ACR9000.*/
    COMMANDED_CURRENT(28738, 256),

    /** The current value of drive torque commanded by the controller (volts). Must be updated
     via Drive Talk on ACR9000. */
    COMMANDED_TORQUE(28739, 256),

    /** The current value of the actual motor torque (volts). Must be updated
     via Drive Talk on ACR9000. */
    ACTUAL_TORQUE(28740, 256),

    /** The current reading of the motor drive temperature. Must be updated
     via Drive Talk on ACR9000. */
    DRIVE_TEMPERATURE(28743, 256),

    /** The motor driver bus voltage. Must be updated
     via Drive Talk on ACR9000. */
    BUS_VOLTAGE(28745, 256);

    private final int base;
    private final int step;

    AxisSingle(final int base, final int step) {
        this.base = base;
        this.step = step;
    }

    /**
     * Gets the controller's index for this parameter for a given axis.
     * @param axis the axis.
     * @return the index.
     */
    public int index(AxisName axis) {return base + axis.index() * step;}

    /**
     * Gets the AcroBasic string reference for this parameter for a given axis.
     * @param axis the axis.
     * @return the reference string.
     */
    public String reference(AxisName axis) {return "P" + index(axis);}
}
