package org.lsst.ccs.drivers.parker;

import static org.lsst.ccs.drivers.parker.MasterUnsigned.*;

/**
 * Names and numbers controller bits associated with particular masters.
 * @author tether
 */
public enum MasterBit {
    
    /** When set stops all the master's motion immediately, ignoring any
     *  programmed end-of-motion deceleration. Motion commands for that master will be ignored
     *  until this bit is cleared. First clear this bit then the {@code KILL_ALL_MOTION} bits in
     *  the slave axes.
     *  <p>The ACR binary comm protocol allows setting and clearing bits directly via
     * their bit index numbers, but for some reason the only way you can get the value
     * of a bit is to read the corresponding flag parameter word and examine the right bit.
     */
    KILL_ALL_MOVES(522, 32, PRIMARY_MASTER_FLAGS, 10);
    
    private final int base;
    private final int step;
    private final MasterUnsigned flagParameter;
    private final long bitMask;
    
    MasterBit(final int base, final int step, final MasterUnsigned flagParameter, final int bitIndex) {
        this.base = base;
        this.step = step;
        this.flagParameter = flagParameter;
        this.bitMask = 1L << bitIndex;
    }
    
    /**
     * Gets the motor controller's index for the bit for the given master. 
     * @param master the master.
     * @return the index.
     */
    public int index(final MasterName master) {return base + master.index() * step;}
    
    /**
     * Gets the string used to refer to the bit for the given master in AcroBasic.
     * @param master the master.
     * @return the reference string.
     */
    public String reference(final MasterName master) {return "BIT" + index(master);}
    
    /**
     * Gets enumerator for the flag parameter containing this bit.
     * @return the enumerator.
     */
    public MasterUnsigned flagParameter() {return flagParameter;}
    
    /**
     * Gets the mask with a single 1 bit in the position assigned to this bit in the
     * associated flag parameter.
     * @return the mask.
     */
    public long flagParameterMask() {return bitMask;}
    
}
