package org.lsst.ccs.drivers.parker;

import static org.lsst.ccs.drivers.parker.AxisUnsigned.*;

/**
 * Names and numbers the some controller bits associated with particular axes. The ACR binary
 * comm protocol allows setting and clearing bits directly via their bit index
 * numbers, but for some reason the only way you can get the value of a bit is to read
 * the corresponding flag parameter word and examine the right bit.
 * @author tether
 */
public enum AxisBit {
    
    /** A capture started with the INTCAP command has completed. */
    CAPTURE_COMPLETE(777, 32, PRIMARY_AXIS_FLAGS, 9),
    
    /** A jog motion is in progress. */
    JOG_ACTIVE(792, 32, PRIMARY_AXIS_FLAGS, 24),
    
    /** Setting this bit also sets {@link MasterBit#KILL_ALL_MOVES} for the master of the axis. 
     *  The bit can be set manually or by any condition that kills motion: 
     *  encountering end-of-travel limits, drive faults, positioning errors or
     *  sending CTRL-X or CTRL-Y to the controller.
     */
    KILL_ALL_MOTION(8467, 32, QUATERNARY_AXIS_FLAGS, 19),
    
    /** Indicates that the final position of a move isn't close enough the target
     *  even though the motion wasn't killed by some other error.
     * This error disables the drive for the axis and stays latched until
     * cleared with a DRIVE ON. Also sets {@link #KILL_ALL_MOTION}, so you have to clear
     * the related flags as well.
    */
    POSITIONING_ERROR(8479, 32, QUATERNARY_AXIS_FLAGS, 31),
    
    /**
     * Hit the hardware limit of travel at the positive (high) end of the encoder scale.
     */
    HIT_POSITIVE_HARD_LIMIT(16132, 32, QUINARY_AXIS_FLAGS, 4),
    
    /**
     * Hit the hardware limit of travel at the negative (low) end of the encoder scale.
     */
    HIT_NEGATIVE_HARD_LIMIT(16133, 32, QUINARY_AXIS_FLAGS, 5),
    
    /**
     * Hit the software limit of travel at the positive (high) end of the encoder scale.
     */
    HIT_POSITIVE_SOFT_LIMIT(16140, 32, QUINARY_AXIS_FLAGS, 8),
    
    /**
     * Hit the software limit of travel at the negative (low) end of the encoder scale.
     */
    HIT_NEGATIVE_SOFT_LIMIT(16141, 32, QUINARY_AXIS_FLAGS, 9),

    /**
     * A JOG HOME command succeeded.
     */
    HOME_FOUND(16134, 32, QUINARY_AXIS_FLAGS, 6),
    
    /**
     * A JOG HOME command failed. Hit both the positive and negative limits of travel
     * while homing.
     */
    HOME_NOT_FOUND(16135, 32, QUINARY_AXIS_FLAGS, 7),
    
    /**
     * Make the positive hard limit signal active low instead of active high.
     */
    INVERT_POSITIVE_HARD_LIMIT_SIGNAL(16144, 32, QUINARY_AXIS_FLAGS, 16),
    
    /**
     * Make the negative hard limit signal active low instead of active high.
     */
    INVERT_NEGATIVE_HARD_LIMIT_SIGNAL(16145, 32, QUINARY_AXIS_FLAGS, 17),
    
    /**
     * Make the home region signal active low instead of active high.
     */
    INVERT_HOME_REGION_SIGNAL(16146, 32, QUINARY_AXIS_FLAGS, 18),
    
    /**
     * Make the controller respect the positive hardware travel limit signal.
     */
    ENABLE_POSITIVE_HARD_LIMIT(16148, 32, QUINARY_AXIS_FLAGS, 20),
    
    /**
     * Make the controller respect the negative hardware travel limit signal.
     */
    ENABLE_NEGATIVE_HARD_LIMIT(16149, 32, QUINARY_AXIS_FLAGS, 21),
    
    /**
     * Make the controller respect the positive software travel limit.
     */
    ENABLE_POSITIVE_SOFT_LIMIT(16150, 32, QUINARY_AXIS_FLAGS, 22),
    
    /**
     * Make the controller respect the negative software travel limit.
     */
    ENABLE_NEGATIVE_SOFT_LIMIT(16151, 32, QUINARY_AXIS_FLAGS, 23),
    
    /**
     * When homing the axis, reverse the direction of motion after encountering
     * a home region edge. When this bit is clear, the controller will bring motion
     * to a stop after encountering the desired edge of the home region. When this
     * bit is set the controller will back up to the edge again after passing it.
     */
    HOMING_USE_BACKUP(16152, 32, QUINARY_AXIS_FLAGS, 24),
    
    /**
     * Seek the negative edge of the home region when homing, rather than the positive
     * edge. "Positive edge" means the one with higher encoder value.
     */
    HOMING_SEEK_NEGATIVE_EDGE(16153, 32, QUINARY_AXIS_FLAGS, 25),
    
    /**
     * Require that the final approach to the selected home region edge be made from
     * the negative side (lower encoder value), rather than from the positive side.
     */
    HOMING_NEGATIVE_FINAL_APPROACH(16154, 32, QUINARY_AXIS_FLAGS, 26);
    
    private final int base;
    private final int step;
    private final AxisUnsigned flagParameter;
    private final long bitMask;
    
    AxisBit(final int base, final int step, final AxisUnsigned flagParameter, final int bitIndex) {
        this.base = base;
        this.step = step;
        this.flagParameter = flagParameter;
        this.bitMask = 1L << bitIndex;
    }
    
    /**
     * Gets the bit index used by the controller.
     * @param axis the axis name.
     * @return the bit index.
     */
    public int index(final AxisName axis) {return base + axis.index() * step;}
    
    /**
     * Gets the string used to refer to the bit in AcroBasic.
     * @param axis
     * @return the bit-reference string.
     */
    public String reference(final AxisName axis) {return "BIT" + index(axis);}
    
    /**
     * Gets the name of the flag parameter containing this bit.
     * @return the flag parameter name.
     */
    public AxisUnsigned flagParameter() {return flagParameter;}
    
    /**
     * Gets the mask with a single 1 bit in the position assigned to this bit in the
     * associated flag parameter.
     * @return the mask.
     */
    public long flagParameterMask() {return bitMask;}
    
}
