package org.lsst.ccs.drivers.parker;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Argument;

import java.io.FileOutputStream;
import java.io.PrintStream;

import java.time.Instant;
import java.time.ZoneId;

import java.time.format.DateTimeFormatter;
import java.time.format.FormatStyle;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import java.util.function.BiConsumer;
import java.util.function.Consumer;

import java.util.stream.Collectors;

/**
 * Provides commands to exercise the AcrComm class from the stand-alone command
 * shell {@link org.lsst.ccs.shell.JLineShell}. See the file {@code RunTestAcrComm.properties}
 * in the main resources directory of this package.
 * <p>
 * Commands that display controller parameters and flags give the name and the AcroBasic
 * reference as well as the value. References to parameters in AcroBasic have the form "P"
 * followed by the parameter's index number, as in "P12345". Bit-flag references
 * are like "BIT12345". Program local variables have a type letter followed by
 * a form letter followed by an index number. "L" denotes long (32-bit signed integer),
 * "S" is for single (32-bit floating point), "D" is for double (64-bit float)
 * and "$" is for string. The form letters are "V" for scalars and "A" for arrays.
 * Examples: LV0, DA10, $V3.
 * <p>
 * Values and bit-flags are divided into a number of categories depending on the controller
 * object, if any, to which they belong.
 * <dl>
 * <dt>Axis</dt><dd>Describes or controls a particular motion axis.</dd>
 * <dt>Connection</dt><dd>Describes or controls a particular comm. connection.</dd>
 * <dt>Encoder</dt><dd>Describes or controls a particular position encoder.</dd>
 * <dt>Master</dt><dd>Describes or controls a particular master object.</dd>
 * <dt>Program</dt><dd>Describes or controls a particular controller program.</dd>
 * <dt>System</dt><dd>General controller description or control.</dd>
 * <dt>User</dt><dd>Set aside for the use of controller programmers.</dd>
 * <dt>Local</dt><dd>A local variable belonging to a controller program.</dd>
 * <dt>Global</dt><dd>Anything not local.</dd>
 * </dl>
 * <p>
 * Note the controller converts all internal floating-point values to and from
 * IEEE single-precision format, even the double floats. The internal format varies between
 * the various controller models.
 * @author tether
 */
public class TestAcrComm {

    // The AcrComm instance currently in use. Making it a single-element array
    // is a hack so that it can be final and thus usable inside lambda expressions.
    private static final AcrComm[] acr = new AcrComm[]{null};

    // For each category of parameter or flag there's
    // a dictionary of setter (getter) objects keyed by parameter/flag name. Each
    // object sets or prints one parameter, possibly taking a master/axis/etc.
    // name as argument to narrow down the particular parameter. Setter objects
    // also take the string form of the new value and convert it to the right type.
    // A similar scheme is used for the local variables of controller programs.


    ////////// Shell command definitions //////////

    /**
     * Opens a network connection to the controller of the given type and IP address.
     * Opening a new connection closes the old one, if any.
     * @param ctrlType the type of controller, for example ACR9000.
     * @param controllerHost the host name or dotted IPv4 address of the controller.
     * @throws Exception if the connection can't be established.
     */
    @Command(name="open", description="Open connections to the controller.")
    public void open_command(
        @Argument(name="controllerType", description="The type of ACR controller.")
        final ControllerType ctrlType,
        @Argument(name="controllerHostname", description="The hostname or dotted IP address of the controller.")
        final String controllerHost
    )
        throws Exception
    {
        if (acr[0] != null) {
            System.out.println("Closing the active connection.");
            close_command();
        }
        acr[0] = AcrComm.newInstance(ctrlType, controllerHost, Optional.of(System.out));
    }

    /**
     * Closes the current connection, if any, to a controller.
     */
    @Command(name="close", description="Close the current connection, if any, to the controller.")
    public void close_command() {
        if (acr[0] != null) {
            acr[0].cleanup();
            acr[0] = null;
        }
    }

    /**
     * Displays all the global parameters and flags known to {@code AcrComm}.
     */
    @Command(name="showGlobals",
        description="Show all the global parameters and flags supported by the Parker driver.")
    public void showGlobals_command() {
        if (acr[0] == null) {System.out.println("Use the 'open' command first!"); return;}
        showGlobals();
    }

    /**
     * Displays all supported system-level parameters and flags.
     */
    @Command(name="showSystem",
        description="Show all system-level parameters supported by the Parker driver.")
    public void showSystem_command() {
        if (acr[0] == null) {System.out.println("Use the 'open' command first!"); return;}
        showSystem();
    }

    /**
     * Displays the supported parameters and flags for a given master.
     * @param masterNum the master number.
     */
    @Command(name="showMaster",
        description="Show the master-level parameters for a given master.")
    public void showMaster_command(
        @Argument(name="masterNumber", description="The master's index number.")
        final int masterNum
    )
    {
        if (acr[0] == null) {System.out.println("Use the 'open' command first!"); return;}
        final MasterName master = convertMasterNumber(masterNum);
        if (master != null) {
            showMaster(master);
        }
    }

    /**
     * Displays the supported parameters and flags for a given encoder.
     * @param encoderNum the encoder number.
     */
    @Command(name="showEncoder",
        description="Show the encoder-level parameters for a given encoder.")
    public void showEncoder_command(
        @Argument(name="encoderNumber", description="The encoder's index number.")
        final int encoderNum
    )
    {
        if (acr[0] == null) {System.out.println("Use the 'open' command first!"); return;}
        final EncoderName encoder = convertEncoderNumber(encoderNum);
        if (encoder != null) {
            showEncoder(encoder);
        }
    }

    /**
     * Displays the supported parameters and flags for a given axis.
     * @param axisNum the axis number.
     */
    @Command(name="showAxis",
        description="Show the axis-level parameters for a given axis.")
    public void showAxis_command(
        @Argument(name="axisNumber", description="The axis' index number.")
        final int axisNum
    )
    {
        if (acr[0] == null) {System.out.println("Use the 'open' command first!"); return;}
        final AxisName axis = convertAxisNumber(axisNum);
        if (axis != null) {
            showAxis(axis);
        }
    }

    /**
     * Displays the supported parameters and flags for a given connection.
     * @param connectionNum the connection number.
     */
    @Command(name="showConnection",
        description="Show the connection-level parameters for a given connection.")
    public void showConnection_command(
        @Argument(name="connectionNumber", description="The connection's index number.")
        final int connectionNum
    )
    {
        if (acr[0] == null) {System.out.println("Use the 'open' command first!"); return;}
        final ConnectionName connection = convertConnectionNumber(connectionNum);
        if (connection != null) {
            showConnection(connection);
        }
    }

    /**
     * Displays the values of the supported user parameters.
     */
    @Command(name="showUser",
        description="Show all user parameters.")
    public void showUser_command()
   {
        if (acr[0] == null) {System.out.println("Use the 'open' command first!"); return;}
        showUser();
    }

    /**
     * Displays the value of a variable local to the given program.
     * @param programNum the program number.
     * @param varName the variable name; see the class description the form of the names.
     */
    @Command(name="showProgram", description="Show a local variable of a program.")
    public void showProgram_command(
        @Argument(name="programNumber", description="The program's index number.")
        final int programNum,
        @Argument(name="varName", description="The name of the variable, for example LV0, SA1, etc.")
        final String varName)
    {
        if (acr[0] == null) {System.out.println("Use the 'open' command first!"); return;}
        final ProgramName program = convertProgramNumber(programNum);
        final Consumer<ProgramName> getter = programGetter(varName);
        if (program != null && getter != null) {
            getter.accept(program);
        }
    }

    /**
     * Sets a particular system parameter or flag.
     * @param paramName the name of the parameter/flag.
     * @param newValue the new value of the right type.
     */
    @Command(name="setSystem", description="Set a system parameter/flag.")
    public void setSystem_command(
        @Argument(name="paramName", description="The name of the system parameter/flag to set.")
        final String paramName,
        @Argument(name="newValue", description="The new value for the parameter/flag.")
        final String newValue)
    {
        if (acr[0] == null) {System.out.println("Use the 'open' command first!"); return;}
        final Consumer<String> setter = systemSetter(paramName);
        if (setter != null) {
            setter.accept(newValue);
        }
    }

    /**
     * Sets a particular master parameter/flag.
     * @param masterNum the master number.
     * @param paramName the parameter/flag name.
     * @param newValue the new value of the right type.
     */
    @Command(name="setMaster", description="Set a master parameter/flag.")
    public void setMaster_command(
        @Argument(name="masterNumber", description="The index number of the target master.")
        final int masterNum,
        @Argument(name="paramName", description="The name of the master parameter/flag to set.")
        final String paramName,
        @Argument(name="newValue", description="The new value for the parameter/flag.")
        final String newValue)
    {
        if (acr[0] == null) {System.out.println("Use the 'open' command first!"); return;}
        final MasterName master = convertMasterNumber(masterNum);
        final BiConsumer<MasterName, String> setter = masterSetter(paramName);
        if (master != null && setter != null) {
            setter.accept(master, newValue);
        }
    }

    /**
     * Sets a particular axis parameter/flag.
     * @param axisNum the axis number.
     * @param paramName the name of the parameter/flag.
     * @param newValue the new value of the right type.
     */
    @Command(name="setAxis", description="Set an axis parameter/flag.")
    public void setAxis_command(
        @Argument(name="axisNumber", description="The index number of the target axis.")
        final int axisNum,
        @Argument(name="paramName", description="The name of the axis parameter/flag to set.")
        final String paramName,
        @Argument(name="newValue", description="The new value for the parameter/flag.")
        final String newValue)
    {
        if (acr[0] == null) {System.out.println("Use the 'open' command first!"); return;}
        final AxisName axis = convertAxisNumber(axisNum);
        final BiConsumer<AxisName, String> setter = axisSetter(paramName);
        if (axis != null && setter != null) {
            setter.accept(axis, newValue);
        }
    }

    /**
     * Sets a particular user parameter.
     * @param userNum the parameter number.
     * @param newValue the new value (floating point).
     */
    @Command(name="setUser", description="Set a user parameter.")
    public void setUser_command(
        @Argument(name="userNumber", description="The index of the user parameter to set.")
        final int userNum,
        @Argument(name="newValue", description="The new value of the parameter.")
        final String newValue)
    {
        if (acr[0] == null) {System.out.println("Use the 'open' command first!"); return;}
        final UserParameter user = convertUserNumber(userNum);
        final double[] decoded = decodeSingles(newValue);
        if (user != null && decoded.length == 1) {
            acr[0].set(user, decoded[0]);
        }
    }

    /**
     * Sets a local variable in a controller program, or part of the variable if
     * it's an array.
     * @param programNum the program number.
     * @param varName the variable name.
     * @param newValue at least one new value, up to as many as the variable can hold.
     */
    @Command(name="setProgram", description="Set a local variable in a program.")
    public void setProgram_command(
        @Argument(name="programNumber", description="The index number of the target program.")
        final int programNum,
        @Argument(name="varName", description="The name of the local variable to set.")
        final String varName,
        @Argument(name="newValues",
                  description="At least one new value, up to as many as the variable can hold.")
        final String... newValue)
    {
        if (acr[0] == null) {System.out.println("Use the 'open' command first!"); return;}
        final ProgramName program = convertProgramNumber(programNum);
        final BiConsumer<ProgramName, String[]> setter = programSetter(varName, newValue);
        if (program != null && setter != null) {
            setter.accept(program, newValue);
        }
    }

    /**
     * Sends AcroBasic commands to the controller and displays the replies. The command
     * is made up of all the whitespace-separated words following "send" up until the end
     * of the line. The words are joined with a space in between each pair of words
     * and sent to the controller.
     * @param word the words to join and send.
     */
    @Command(name="send", description="Send AcroBasic commands to the controller and display the reply.")
    public void send_command(
        @Argument(name="words", description="Words to send to the controller separated by blanks.")
        final String... word
    )
    {
        if (acr[0] == null) {System.out.println("Use the 'open' command first!"); return;}
        final String command = Arrays.stream(word).collect(Collectors.joining(" "));
        acr[0].sendStr(command);
    }


    ////////// Command implementation //////////

    private void showGlobals() {
        System.out.println("\nsection: System parameters and bits");
        showSystem();
        System.out.println("\nsection: Master parameters and bits");
        for (final MasterName m: MasterName.values()) {
            showMaster(m);
        }
        System.out.println("\nsection: Encoder parameters and bits");
        for (final EncoderName e: EncoderName.values()) {
            showEncoder(e);
        }
        System.out.println("\nsection: Axis parameters and bits");
        for (final AxisName a: AxisName.values()) {
            showAxis(a);
        }
        System.out.println("\nsection: Connection parameters");
        for (final ConnectionName c: ConnectionName.values()) {
            showConnection(c);
        }
        System.out.println("\nsection: User parameters");
        showUser();
    }

    private void showSystem() {
        for (final SystemUnsigned x: SystemUnsigned.values()) {
            final String line = String.format("value: %10s %20d %s", x.reference(), acr[0].get(x), x.name());
            System.out.println(line);
        }
        for (final SystemBit x: SystemBit.values()) {
            final String line = String.format("value: %10s %20b %s", x.reference(), acr[0].get(x), x.name());
            System.out.println(line);
        }
    }

    private void showMaster(final MasterName master) {
        System.out.println("master: " + master.index());
        for (final MasterUnsigned x: MasterUnsigned.values()) {
            final String line =
                String.format("value: %10s %20d %s",
                    x.reference(master),
                    acr[0].get(master, x),
                    x.name());
            System.out.println(line);
        }
        for (final MasterBit x: MasterBit.values()) {
            final String line =
                String.format("value: %10s %20b %s",
                    x.reference(master),
                    acr[0].get(master, x),
                    x.name());
            System.out.println(line);
        }
    }

    private void showEncoder(final EncoderName encoder) {
        System.out.println("encoder: " + encoder.index());
        for (final EncoderLong x: EncoderLong.values()) {
            final String line =
                String.format("value: %10s %20d %s",
                    x.reference(encoder),
                    acr[0].get(encoder, x),
                    x.name());
            System.out.println(line);
        }
    }

    private void showAxis(final AxisName axis) {
        System.out.println("axis: " + axis.index());
        for (final AxisUnsigned x: AxisUnsigned.values()) {
            final String line =
                String.format("value: %10s %20d %s",
                    x.reference(axis),
                    acr[0].get(axis, x),
                    x.name());
            System.out.println(line);
        }
        for (final AxisLong x: AxisLong.values()) {
            final String line =
                String.format("value: %10s %20d %s",
                    x.reference(axis),
                    acr[0].get(axis, x),
                    x.name());
            System.out.println(line);
        }
        for (final AxisSingle x: AxisSingle.values()) {
            final String line =
                String.format("value: %10s %20e %s",
                    x.reference(axis),
                    acr[0].get(axis, x),
                    x.name());
            System.out.println(line);
        }
        for (final AxisBit x: AxisBit.values()) {
            final String line =
                String.format("value: %10s %20b %s",
                    x.reference(axis),
                    acr[0].get(axis, x),
                    x.name());
            System.out.println(line);
        }
    }

    private void showConnection(final ConnectionName conn) {
        System.out.println("connection: " + conn.index());
        for (final ConnectionUnsigned x: ConnectionUnsigned.values()) {
            final String line =
                String.format("value: %10s %20d %s",
                    x.reference(conn),
                    acr[0].get(conn, x),
                    x.name());
            System.out.println(line);
        }
    }

    private void showUser() {
        for (final UserParameter user: UserParameter.values()) {
            final String line =
                String.format("value: %10s %20e %s",
                    user.reference(),
                    acr[0].get(user),
                    user.name());
            System.out.println(line);
        }
    }




    private static final MasterName[] MASTER_NAMES = MasterName.values();

    private MasterName convertMasterNumber(final int masterNum) {
        MasterName result = null;
        if (masterNum < 0 | masterNum >= MASTER_NAMES.length) {
            System.out.println("Invalid master number. Use 0-" + (MASTER_NAMES.length-1) + ".");
        }
        else {
            result = MASTER_NAMES[masterNum];
        }
        return result;
    }

    private static final EncoderName[] ENCODER_NAMES = EncoderName.values();

    private EncoderName convertEncoderNumber(final int encoderNum) {
        EncoderName result = null;
        if (encoderNum < 0 | encoderNum >= ENCODER_NAMES.length) {
            System.out.println("Invalid encoder number. Use 0-" + (ENCODER_NAMES.length-1) + ".");
        }
        else {
            result = ENCODER_NAMES[encoderNum];
        }
        return result;
    }

    private static final AxisName[] AXIS_NAMES = AxisName.values();

    private AxisName convertAxisNumber(final int axisNum) {
        AxisName result = null;
        if (axisNum < 0 | axisNum >= AXIS_NAMES.length) {
            System.out.println("Invalid axis number. Use 0-" + (AXIS_NAMES.length-1) + ".");
        }
        else {
            result = AXIS_NAMES[axisNum];
        }
        return result;
    }

    private static final ConnectionName[] CONNECTION_NAMES = ConnectionName.values();

    private ConnectionName convertConnectionNumber(final int connectionNum) {
        ConnectionName result = null;
        if (connectionNum < 0 | connectionNum >= CONNECTION_NAMES.length) {
            System.out.println("Invalid connection number. Use 0-" + (CONNECTION_NAMES.length-1) + ".");
        }
        else {
            result = CONNECTION_NAMES[connectionNum];
        }
        return result;
    }

    private static final ProgramName[] PROGRAM_NAMES = ProgramName.values();

    private ProgramName convertProgramNumber(final int programNum) {
        ProgramName result = null;
        if (programNum < 0 | programNum >= PROGRAM_NAMES.length) {
            System.out.println("Invalid program number. Use 0-" + (PROGRAM_NAMES.length-1) + ".");
        }
        else {
            result = PROGRAM_NAMES[programNum];
        }
        return result;
    }

    private static final UserParameter[] USER_PARAMETERS = UserParameter.values();

    private UserParameter convertUserNumber(final int userNum) {
        UserParameter result = null;
        if (userNum < 0 | userNum >= USER_PARAMETERS.length) {
            System.out.println("Invalid user parameter number. Use 0-" + (USER_PARAMETERS.length-1) + ".");
        }
        else {
            result = USER_PARAMETERS[userNum];
        }
        return result;
    }

    private static double[] decodeSingles(final String... value) {
        final double[] result
            = Arrays.stream(value)
            .map(val -> {
                Double decoded = null;
                try {
                    decoded = Double.valueOf(val);
                } catch (NumberFormatException exc) {
                    System.out.println("Invalid floating-point value: " + val);
                }
                return decoded;
            })
            .filter(val -> val != null)
            .mapToDouble(Double::doubleValue)
            .toArray();
        return result;
    }

    private static long[] decodeLongs(final String... value) {
        final long[] result
            = Arrays.stream(value)
            .map(val -> {
                Long decoded = null;
                try {
                    decoded = Long.valueOf(val);
                } catch (NumberFormatException exc) {
                    System.out.println("Invalid integer: " + val);
                }
                return decoded;
            })
            .filter(val -> val != null)
            .filter(val -> {
                if (val < Integer.MIN_VALUE | val > Integer.MAX_VALUE) {
                    System.out.println("Too big to fit in 31 bits + sign: " + val);
                    return false;
                }
                else {
                    return true;
                }
            })
            .mapToLong(Long::longValue)
            .toArray();
        return result;
    }

    private static long[] decodeUnsigneds(final String... value) {
        final long[] result
            = Arrays.stream(value)
            .map(val -> {
                Long decoded = null;
                try {
                    decoded = Long.valueOf(val);
                } catch (NumberFormatException exc) {
                    System.out.println("Invalid integer: " + val);
                }
                return decoded;
            })
            .filter(val -> val != null)
            .filter(val -> {
                if (val < 0L | val > 0xffffffffL) {
                    System.out.println("Negative or too big to fit in 32 bits: " + val);
                    return false;
                }
                else {
                    return true;
                }
            })
            .mapToLong(Long::longValue)
            .toArray();
        return result;
    }

    private static Boolean decodeBit(final String value) {
        try {
            return Boolean.valueOf(value);
        }
        catch (NumberFormatException exc) {
            System.out.println("Invalid boolean: " + value);
            return null;
        }
    }

    private BiConsumer<AxisName, String> axisSetter(final String paramName) {
        final BiConsumer<AxisName, String> result =
            AXIS_SETTERS.get(paramName.toUpperCase());
        if (result == null) {
            System.out.println("Not a settable axis parameter: " + paramName);
        }
        return result;
    }

    private final static
        Map<String, BiConsumer<AxisName, String>>
        AXIS_SETTERS;

    static {
        final Map<String, BiConsumer<AxisName, String>> setters = new HashMap<>();
        for (final AxisSingle p: AxisSingle.values()) {
            setters.put(p.name(),
                (axis, val) -> {
                    final double[] decoded = decodeSingles(val);
                    if (decoded.length == 1) acr[0].set(axis, p, decoded[0]);
                });
        }
        for (final AxisLong p: AxisLong.values()) {
            setters.put(p.name(),
                (axis, val) -> {
                    final long[] decoded = decodeLongs(val);
                    if (decoded.length == 1) acr[0].set(axis, p, decoded[0]);
            });
        }
        for (final AxisBit b: AxisBit.values()) {
            setters.put(b.name(),
                (axis, val) -> {
                    final Boolean decoded = decodeBit(val);
                    if (decoded != null) acr[0].set(axis, b, decoded);
                });
        }
        AXIS_SETTERS = Collections.unmodifiableMap(setters);
    }

    private BiConsumer<MasterName, String> masterSetter(final String paramName) {
        final BiConsumer<MasterName, String> result =
            MASTER_SETTERS.get(paramName.toUpperCase());
        if (result == null) System.out.println("Not a settable master parameter: " + paramName);
        return result;
    }

    private final static
        Map<String, BiConsumer<MasterName, String>>
        MASTER_SETTERS;

    static {
        final Map<String, BiConsumer<MasterName, String>> setters = new HashMap<>();
        for (final MasterBit b: MasterBit.values()) {
            setters.put(b.name(),
                (master, val) -> acr[0].set(master, b, decodeBit(val)));
        }
        MASTER_SETTERS = Collections.unmodifiableMap(setters);
    }

    private BiConsumer<EncoderName, String> encoderSetter(final String paramName) {
        final BiConsumer<EncoderName, String> result =
            ENCODER_SETTERS.get(paramName.toUpperCase());
        if (result == null) System.out.println("Not a settable encoder parameter: " + paramName);
        return result;
    }

    private final static
        Map<String, BiConsumer<EncoderName, String>>
        ENCODER_SETTERS;

    static {
        final Map<String, BiConsumer<EncoderName, String>> setters = new HashMap<>();
        for (final EncoderLong p: EncoderLong.values()) {
            setters.put(p.name(),
                (encoder, val) -> {
                    final long[] decoded = decodeLongs(val);
                    if (decoded.length == 1) acr[0].set(encoder, p, decoded[0]);
                });
        }
        ENCODER_SETTERS = Collections.unmodifiableMap(setters);
    }

    private Consumer<String> systemSetter(final String paramName) {
        final Consumer<String> result = SYSTEM_SETTERS.get(paramName.toUpperCase());
        if (result == null) System.out.println("Not a settable system parameter: " + paramName);
        return result;
    }

    private final static
        Map<String, Consumer<String>>
        SYSTEM_SETTERS;

    static {
        final Map<String, Consumer<String>> setters = new HashMap<>();
        for (final SystemUnsigned p: SystemUnsigned.values()) {
            setters.put(p.name(),
                val -> {
                    final long[] decoded = decodeUnsigneds(val);
                    if (decoded.length == 1) acr[0].set(p, decoded[0]);
                });
        }
        for (final SystemBit b: SystemBit.values()) {
            setters.put(b.name(),
                val -> {
                    final Boolean decoded = decodeBit(val);
                    if (decoded != null) acr[0].set(b, decoded);
                });
        }
        SYSTEM_SETTERS = Collections.unmodifiableMap(setters);
    }

    private Consumer<ProgramName> programGetter(final String varName) {
        final Consumer<ProgramName> result = PROGRAM_GETTERS.get(varName.toUpperCase());
        if (result == null) {
            System.out.println("Unsupported local variable: " + varName);
        }
        return result;
    }

    private final static
        Map<String, Consumer<ProgramName>>
        PROGRAM_GETTERS;

    static {
        final Map<String, Consumer<ProgramName>> getters = new HashMap<>();
        for (final LocalLong var: LocalLong.values()) {
            getters.put(
                var.name(),
                program -> {
                    System.out.println(acr[0].get(program, var));
                });
        }
        for (final LocalSingle var: LocalSingle.values()) {
            getters.put(
                var.name(),
                program -> {
                    System.out.println(String.format("%.6g", acr[0].get(program, var)));
                });
        }
        for (final LocalDouble var: LocalDouble.values()) {
            getters.put(
                var.name(),
                program -> {
                    System.out.println(String.format("%.12g", acr[0].get(program, var)));
                });
        }
        for (final LocalString var: LocalString.values()) {
            getters.put(
                var.name(),
                program -> {
                    System.out.format("\"%s\"%n", acr[0].get(program, var));
                });
        }
        for (final LocalLongArray var: LocalLongArray.values()) {
            getters.put(
                var.name(),
                program -> {
                    final long[] value = acr[0].get(program, var);
                    for (int i = 0; i < value.length; ++i) {
                        System.out.println(String.format("%d: %d", i, value[i]));
                    }
                });
        }
        for (final LocalSingleArray var: LocalSingleArray.values()) {
            getters.put(
                var.name(),
                program -> {
                    final double[] value = acr[0].get(program, var);
                    for (int i = 0; i < value.length; ++i) {
                        System.out.println(String.format("%d: %.6g", i, value[i]));
                    }
                });
        }
        for (final LocalDoubleArray var: LocalDoubleArray.values()) {
            getters.put(
                var.name(),
                program -> {
                    final double[] value = acr[0].get(program, var);
                    for (int i = 0; i < value.length; ++i) {
                        System.out.println(String.format("%d: %.12g", i, value[i]));
                    }
                });
        }
        for (final LocalStringArray var: LocalStringArray.values()) {
            getters.put(
                var.name(),
                program -> {
                    final String[] value = acr[0].get(program, var);
                    for (int i = 0; i < value.length; ++i) {
                        System.out.format("%d: \"%s\"%n", i, value[i]);
                    }
                });
        }
        PROGRAM_GETTERS = Collections.unmodifiableMap(getters);
    }

    BiConsumer<ProgramName, String[]> programSetter(final String varName, final String[] newValue) {
        BiConsumer<ProgramName, String[]> result = PROGRAM_SETTERS.get(varName.toUpperCase());
        if (result == null) {
            System.out.println("Unsupported local variable: " + varName);
        }
        if (newValue.length == 0) {
            result = null;
            System.out.println("Supply at least one new value.");
        }
        return result;
    }

    private final static
        Map<String, BiConsumer<ProgramName, String[]>>
        PROGRAM_SETTERS;

    static {
        final Map<String, BiConsumer<ProgramName, String[]>> setters = new HashMap<>();
        for (final LocalLong var: LocalLong.values()) {
            setters.put(
                var.name(),
                (program, newValue) -> {
                    final long[] decoded = decodeLongs(newValue);
                    if (decoded.length == newValue.length) acr[0].set(program, var, decoded[0]);
                });
        }
        for (final LocalSingle var: LocalSingle.values()) {
            setters.put(
                var.name(),
                (program, newValue) -> {
                    final double[] decoded = decodeSingles(newValue);
                    if (decoded.length == newValue.length) acr[0].set(program, var, decoded[0]);
                });
        }
        for (final LocalDouble var: LocalDouble.values()) {
            setters.put(
                var.name(),
                (program, newValue) -> {
                    final double[] decoded = decodeSingles(newValue);
                    if (decoded.length == newValue.length) acr[0].set(program, var, decoded[0]);
                });
        }
        for (final LocalString var: LocalString.values()) {
            setters.put(
                var.name(),
                (program, newValue) -> {
                    acr[0].set(program, var, newValue[0]);
                });
        }
        for (final LocalLongArray var: LocalLongArray.values()) {
            setters.put(
                var.name(),
                (program, newValue) -> {
                    final long[] decoded = decodeLongs(newValue);
                    if (decoded.length == newValue.length) acr[0].set(program, var, decoded);
                });
        }
        for (final LocalSingleArray var: LocalSingleArray.values()) {
            setters.put(
                var.name(),
                (program, newValue) -> {
                    final double[] decoded = decodeSingles(newValue);
                    if (decoded.length == newValue.length) acr[0].set(program, var, decoded);
                });
        }
        for (final LocalDoubleArray var: LocalDoubleArray.values()) {
            setters.put(
                var.name(),
                (program, newValue) -> {
                    final double[] decoded = decodeSingles(newValue);
                    if (decoded.length == newValue.length) acr[0].set(program, var, decoded);
                });
        }
        for (final LocalStringArray var: LocalStringArray.values()) {
            setters.put(
                var.name(),
                (program, newValue) -> {
                    acr[0].set(program, var, newValue);
                });
        }
        PROGRAM_SETTERS = Collections.unmodifiableMap(setters);
    }

}
