package org.lsst.ccs.drivers.parker;

import static org.lsst.ccs.drivers.parker.ProgramUnsigned.PROGRAM_FLAGS;

/**
 * Names and numbers controller bits associated with particular programs.
 * @author tether
 */
public enum ProgramBit {
    
    /** When set it indicates that the associated program is running.
     *  <p>The ACR binary comm protocol allows setting and clearing bits directly via
     * their bit index numbers, but for some reason the only way you can get the value
     * of a bit is to read the corresponding flag parameter word and examine the right bit.
     */
    PROGRAM_RUNNING(1024, 32, PROGRAM_FLAGS, 0),
    
    /** When set it indicates that the associated program is dwelling, that is,
     *  sleeping for a specific time.
     */
    PROGRAM_DWELLING(1025, 32, PROGRAM_FLAGS, 1),
    
    /** When set it indicates that the associated program is inhibited, that is,
     *  executing an INH command.
     */
    PROGRAM_INHIBITED(1026, 32, PROGRAM_FLAGS, 2);
    
    private final int base;
    private final int step;
    private final ProgramUnsigned flagParameter;
    private final long bitMask;
    
    ProgramBit(final int base, final int step, final ProgramUnsigned flagParameter, final int bitIndex) {
        this.base = base;
        this.step = step;
        this.flagParameter = flagParameter;
        this.bitMask = 1L << bitIndex;
    }
    
    /**
     * Gets the motor controller's index for the bit for the given program. 
     * @param program the program.
     * @return the index.
     */
    public int index(final ProgramName program) {return base + program.index() * step;}
    
    /**
     * Gets the string used to refer to the bit for the given program in AcroBasic.
     * @param program the program.
     * @return the reference string.
     */
    public String reference(final ProgramName program) {return "BIT" + index(program);}
    
    /**
     * Gets the enumerator for the flag parameter containing this bit.
     * @return the enumerator.
     */
    public ProgramUnsigned flagParameter() {return flagParameter;}
    
    /**
     * Gets the mask with a single 1 bit in the position assigned to this bit in the
     * associated flag parameter.
     * @return the mask.
     */
    public long flagParameterMask() {return bitMask;}
    
}
