package org.lsst.ccs.drivers.parker;

/**
 * Contains all supported Parker ACR-like controller types. All ACR controllers use the
 * same AcroBasic language for motion programs. They differ in where you look for the address
 * of the global variable area and whether memory is addressed per 32-bit word or per byte.
 * @author tether
 */
public enum ControllerType {

    /** Aries AR-xxCE single-axis controller/driver combos. */
    ARIES_CE(0xA0_60_00_20, 4),

    /** ACR 1200 controller.*/
    ACR1200(0x00_40_00_08, 1),

    /** ACR 1500 controller.*/
    ACR1500(0x00_C0_80_08, 1),

    /** ACR 1505 controller.*/
    ACR1505(0x00_40_00_09, 1),

    /** ACR 2000 controller.*/
    ACR2000(0x00_40_00_08, 1),

    /** ACR 8000 controller.*/
    ACR8000(0x00_40_3E_08, 1),

    /** ACR 8010 controller.*/
    ACR8010(0x00_40_3E_08, 1),

    /** ACR 8020 controller.*/
    ACR8020(0x00_40_00_09, 1),

    /** ACR 9xxx series controllers. */
    ACR9000(0x00_40_00_09, 1);

    private final int systemPointerAddress;

    private final int wordBump;

    ControllerType(int sysPtrAddr, int bump) {
        this.systemPointerAddress = sysPtrAddr;
        this.wordBump = bump;
    }

    /**
     * Gets the address at which the pointer to global variables is found.
     * This address varies from one model of controller to the next. At this address
     * is a 32-bit pointer to the global data area. The first 32-bit word
     * in this area, if it exists, is an unsigned integer count of the
     * the number of words of global variables. The variables come immediately after
     * and each takes up two words.
     * @return Either zero (no global variables) or the address of the global area.
     */
    public int getSystemPointerAddress() {return systemPointerAddress;}

    /**
     * Gets the number of addresses occupied by word in controller memory.
     * For a byte-addressed memory this will be four while for a word-addressed
     * memory it will be one. If the global data area starts at address
     * {@code glob} then the address if the i'th global variable, counting from zero, will
     * be at {@code glob + wordBump*(1 + i)}.
     * @return The word size.
     */
    public int getWordBump() {return wordBump;}
}
