package org.lsst.ccs.drivers.parker;

import static org.lsst.ccs.drivers.parker.ConnectionUnsigned.*;

import org.lsst.ccs.utilities.logging.Logger;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.PrintStream;

import java.net.Socket;
import java.net.SocketTimeoutException;

import java.nio.file.Files;
import java.nio.file.Paths;

import java.nio.charset.StandardCharsets;

import java.time.Duration;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import java.util.stream.Stream;


/**
 * Communicates with a Parker ACR-type motor controller via Ethernet. Use the standard
 * CCS {@code logging.properties} file to control the driver's logger
 * {@code org.lsst.ccs.drivers.parker}. AcrComm can also write some output
 * to an open {@code PrintStream} if you give it one, for example {@code System.out}
 * if you're using the driver in an interactive session as does the {@link TestAcrComm}
 * class
 * <p>
 * Note that there's a 200 ms timeout set on all reception of replies from
 * the controller. That's more than 100 times the time it usually takes for
 * the controller to start sending a reply, unless you send a command that
 * takes longer to execute. In particular you should avoid long sleeps using the dwell
 * command DWL, long waits for flags using the inhibit command INH, or long
 * waits for other events using the capture command INTCAP.
 * <p>
 * This version of the driver doesn't implement watchdog requests for the TCP
 * connection to the controller. This may be added when we figure out how to use it
 * and if we can make good use of it.
 * <p>
 * For most LSST-related work you should use only the {@code get()} and {@code set()} methods
 * that take enumerations as arguments, plus the {@code sendStr()} method that sends
 * ASCII commands and {@code cleanup()} that gracefully terminates the connection to the
 * controller. The lower-level methods such as {@code getLongParameter()} that
 * take simple parameter index numbers should be used only if the parameters
 * you want don't have names in any of the public enumeration classes. The low-level
 * methods may one day be split off into another class, as {@code AcrComm} has gotten
 * rather large. It's probably best to have the maintainer of this package add the names
 * you need.
 * @author saxton
 * @author tether
 */
public final class AcrComm {

    private final static Logger log = Logger.getLogger("org.lsst.ccs.drivers.parker");

    private final static int
        // Controller TCP port numbers.
        MANAGEMENT_PORT       = 5004,
        BINARY_COMMAND_PORT   = 5006,
        // Timeouts in milliseconds for reads on the socket
        // connected to the binary command port.
        NO_TIMEOUT       = 0,
        // The controller normally sends a response to a command within a few milliseconds
        // unless the command itself takes longer than that to execute, for example
        // DWL or INH commands that wait for a long time.
        SHORT_TIMEOUT    = 200,
        LONG_TIMEOUT     = 2000;

    ////////// Lifecycle methods //////////


    /**
     * Opens a TCP connection to the controller and makes a new instance of this class.
     * @param ctrlType the controller type.
     * @param controllerHost the host name or dotted IPv4 address of the controller.
     * @param out the optional console output stream. Use a non-empty value
     * when using the driver interactively.
     * @return the new {@code AcrComm} instance.
     * @throws IOException if there's a problem creating or managing the socket.
     */
    public static AcrComm newInstance(
        ControllerType ctrlType,
        String controllerHost,
        Optional<PrintStream> out
        )
            throws IOException
    {
        Socket binSock = null;

        // Connect to the binary command/response port.
        try {
            binSock = new Socket(controllerHost, BINARY_COMMAND_PORT);
            binSock.setSoTimeout(SHORT_TIMEOUT);
        }
        catch (IOException e) {
            throw new AcrComm.Exception(e);
        }

        // Now we can create the instance.
        final AcrComm comm = new AcrComm(
            ctrlType,
            out,
            binSock
        );

        // Get the controller to send us some text.
        comm.kick();

        return comm;
    }

    private final ControllerType ctrlType;
    private final Optional<PrintStream> out;
    private final Socket binSock;
    private final BufferedInputStream commandIn;
    private final BufferedOutputStream commandOut;


    /**
     * Fills in the fields of a new instance.
     * @param ctrlType the type of ACR controller.
     * @param out the optional console output stream.
     * @param binSock the socket connected to the controller's binary command/response port.
     */
     private AcrComm(
        final ControllerType ctrlType,
        final Optional<PrintStream> out,
        final Socket binSock
        )
             throws IOException
    {
        this.ctrlType = ctrlType;
        this.out= out;
        this.binSock = binSock;
        this.commandIn = new BufferedInputStream(binSock.getInputStream());
        this.commandOut = new BufferedOutputStream(binSock.getOutputStream());
    }


     /**
      * Inner exception class for laundering non-runtime exceptions.
      */
     public static class Exception extends RuntimeException {

        public Exception(Throwable e)
        {
            super(e);
        }
    }

    /**
     * Waits for unsolicited output from the controller and if none is
     * forthcoming attempts to generate some by sending a VER command. Use
     * after creating a new instance of this class;
     * {@link #newInstance(org.lsst.ccs.drivers.parker.ControllerType,
     * java.lang.String, java.util.Optional)} does this automatically.
     */
    public void kick() {
            int leng = receiveAscii();
            if (leng == 0) sendStr("VER");

        // Scan the controller's connection info to find out
        // which command-processing stream we're connected to.
        final int binPort = binSock.getLocalPort();
        final byte[] ourIPaddress = binSock.getLocalAddress().getAddress();
        long intIPaddress = 0;
        for (int j = 0; j < ourIPaddress.length; j++) {
            intIPaddress = (intIPaddress << 8) | Byte.toUnsignedInt(ourIPaddress[j]);
        }
        for (final ConnectionName conn: ConnectionName.values()) {
            final long status = get(conn, CONNECTION_STATUS);
            if (status != 1 && status != 2) {continue;}
            // The connection is active, check the IP parameters.
            final long port = get(conn, CLIENT_IP_PORT);
            if (binPort != port) {continue;}
            final long clientip = get(conn, CLIENT_IP_ADDRESS);
            if (intIPaddress != clientip) {continue;}
            log.info("Connected to stream " + get(conn, STREAM_NUMBER));
            out.ifPresent(console ->
                console.println("Connected to stream " + get(conn, STREAM_NUMBER)));
            break;
        }
    }

    /**
     * Drains and closes sockets, closes logs.
     */
    public void cleanup() {
        try {
            drain();
        } catch (RuntimeException e) {
            log.warning("Error draining binary command/response  socket during cleanup.", e);
        }
        try {
            binSock.close();
        } catch (IOException e) {
            log.warning("Error closing binary command/response socket.", e);
        }
    }

    ////////// High-level communication with the controller. //////////

    /**
     * Gets from the controller an axis flag bit's state for a given axis.
     * @param axis the axis.
     * @param axbit the flag bit to get.
     * @return true if the bit is set, else false.
     */
    public boolean get(final AxisName axis, final AxisBit axbit) {
        // To read a bit we have to read the flag parameter for the bit.
        final long value = get(axis, axbit.flagParameter());
        return (value & axbit.flagParameterMask()) != 0;
    }

    /**
     * Gets from the controller an unsigned 32-bit integer axis parameter for a given axis.
     * @param axis the axis.
     * @param axunsigned the axis parameter to get.
     * @return the parameter value, as a Java long.
     */
    public long get(final AxisName axis, final AxisUnsigned axunsigned) {
        return getUnsignedParameter(axunsigned.index(axis));
    }

    /**
     * Gets from the controller a signed 32-bit integer axis parameter for a given axis.
     * @param axis the axis.
     * @param axlong the axis parameter to get.
     * @return the parameter value, as a Java long.
     */
    public long get(final AxisName axis, final AxisLong axlong) {
        return getLongParameter(axlong.index(axis));
    }

    /**
     * Gets from the controller a single-float axis parameter for the given axis.
     * @param axis the axis.
     * @param axsingle the axis parameter to get.
     * @return the parameter value, as a Java double.
     */
    public double get(final AxisName axis, final AxisSingle axsingle) {
        return getSingleParameter(axsingle.index(axis));
    }

    /**
     * Gets from the controller one of the unsigned 32-bit parameters associated with
     * a communications connection.
     * @param conn the connection.
     * @param connunsigned the parameter.
     * @return the value of the parameter.
     */
    public long get(final ConnectionName conn, final ConnectionUnsigned connunsigned) {
        return getUnsignedParameter(connunsigned.index(conn));
    }

    /**
     * Gets from the controller a 32-bit signed encoder parameter for a given encoder.
     * @param encoder the encoder.
     * @param enclong the encoder parameter to get.
     * @return the parameter value, as a Java long.
     */
    public long get(final EncoderName encoder, final EncoderLong enclong) {
        return getLongParameter(enclong.index(encoder));
    }

    /**
     * Gets from the controller the value of a double-float local scalar variable
     * for a given program.
     * @param program the program.
     * @param locdouble the variable to get.
     * @return the variable value, as a Java double.
     */
    public double get(final ProgramName program, final LocalDouble locdouble) {
        final long addr = localAddress(program.index(), DOUBLE_SCALAR, 0, locdouble.index());
        return decodeSingles(peek(addr, 1, FP64_CONVERSION))[0];
    }

    /**
     * Gets from the controller all the elements of a double-float local array variable
     * for a given program.
     * @param program the program.
     * @param ldarray the variable to get.
     * @return the array's values, as Java doubles.
     */
    public double[] get(final ProgramName program, final LocalDoubleArray ldarray) {
        final long addr = localAddress(program.index(), DOUBLE_ARRAY, ldarray.index(), 0);
        final int itemCount = arrayElementCount(addr, false);
        return decodeSingles(peek(addr, itemCount, FP64_CONVERSION));
    }

    /**
     * Gets from the controller the value of a local long scalar variable for a given program.
     * @param program the program.
     * @param loclong the variable to get.
     * @return the variable's value, always signed, as a Java long.
     */
    public long get(final ProgramName program, final LocalLong loclong) {
        final long addr = localAddress(program.index(), LONG_SCALAR, 0, loclong.index());
        return decodeLongs(peek(addr, 1, LONG_CONVERSION))[0];
    }

    /**
     * Gets from the controller all the elements of a local long array for a given
     * program.
     * @param program the program.
     * @param llarray the array to get.
     * @return the array values, always signed, as Java longs.
     */
    public long[] get(final ProgramName program, final LocalLongArray llarray) {
        final long addr = localAddress(program.index(), LONG_ARRAY, llarray.index(), 0);
        final int wordCount = arrayElementCount(addr, false);
        return decodeLongs(peek(addr, wordCount, LONG_CONVERSION));
    }

    /**
     * Gets from the controller the value of a single-float local scalar variable for a given
     * program.
     * @param program the program.
     * @param locsingle the variable to get.
     * @return the variable value, as a Java double.
     */
    public double get(final ProgramName program, final LocalSingle locsingle) {
        final long addr = localAddress(program.index(), SINGLE_SCALAR, 0, locsingle.index());
        return decodeSingles(peek(addr, 1, FP32_CONVERSION))[0];
    }

    /**
     * Gets from the controller the values of a single-float local array for a given program.
     * @param program the program.
     * @param lsarray the array to get.
     * @return the array values as Java doubles.
     */
    public double[] get(final ProgramName program, final LocalSingleArray lsarray) {
        final long addr = localAddress(program.index(), SINGLE_ARRAY, lsarray.index(), 0);
        final int wordCount = arrayElementCount(addr, false);
        return decodeSingles(peek(addr, wordCount, FP32_CONVERSION));
    }

    /**
     * Gets from the controller the value of a local ASCII string scalar variable for a given
     * program.
     * @param program the program.
     * @param locstring the variable to get.
     * @return the value of the string, each byte as a Java char.
     */
    public String get(final ProgramName program, final LocalString locstring) {
        final long addr = localAddress(program.index(), STRING_SCALAR, 0, locstring.index());
        final int wordCount = stringActualWordCount(addr);
        return decodeStrings(wordCount, peek(addr, wordCount, LONG_CONVERSION))[0];
    }

    /**
     * Gets from the controller the values of a local string array for a given program.
     * @param program the program.
     * @param lstrarray the array to get.
     * @return the array's values, each byte as a Java char.
     */
    public String[] get(final ProgramName program, final LocalStringArray lstrarray) {
        final long addr = localAddress(program.index(), STRING_ARRAY, lstrarray.index(), 0);
        // Read the array metadata just below the first array element.
        final long[] metadata = decodeUnsigneds(peek(addr - 2*ctrlType.getWordBump(), 2, LONG_CONVERSION));
        // Convert max byte count to max word count (allocation is always in full words).
        final int wordsPerString = (int)( (metadata[1] + 3) / 4);
        // Get the number of array elements.
        final int stringCount = (int)(metadata[0]);
        return decodeStrings(wordsPerString, peek(addr, stringCount*wordsPerString, LONG_CONVERSION));
    }

    /**
     * Gets from the controller the state of a flag bit for a given master.
     * @param master the master.
     * @param mbit the bit to get.
     * @return true if the bit is set, else false.
     */
    public boolean get(final MasterName master, final MasterBit mbit) {
        final long value = get(master, mbit.flagParameter());
        return (value & mbit.flagParameterMask()) != 0;
    }

    /**
     * Gets from the controller the value of an unsigned 32-bit parameter for a given master.
     * @param master the master.
     * @param munsigned the parameter to get.
     * @return the parameter value as a Java long.
     */
    public long get(final MasterName master, final MasterUnsigned munsigned) {
        return getUnsignedParameter(munsigned.index(master));
    }

    /**
     * Gets from the controller the value of a system flag bit.
     * @param sysbit the bit to get.
     * @return true if the bit is set, else false.
     */
    public boolean get(final SystemBit sysbit) {
        final long value = get(sysbit.flagParameter());
        return (value & sysbit.flagParameterMask()) != 0;
    }

    /**
     * Gets from the controller the value of an unsigned 32-bit system parameter.
     * @param sysul the parameter to get.
     * @return the value as a Java long.
     */
    public long get(final SystemUnsigned sysul) {
        return getUnsignedParameter(sysul.index());
    }

    /**
     * Gets from the controller the value of a user parameter (always a double-float).
     * @param uparam the parameter to get.
     * @return the value as a Java double.
     */
    public double get(final UserParameter uparam) {
        return decodeSingles(peek(globalAddress(uparam), 1, FP64_CONVERSION))[0];
    }

    /**
     * Sets or clears a controller axis flag bit.
     * @param axis the axis.
     * @param axbit the axis flag bit to set.
     * @param value true to set the bit, false to clear it.
     */
    public void set(final AxisName axis, final AxisBit axbit, final boolean value) {
        setOrClearBit(axbit.index(axis), value);
    }

    /**
     * Sets the value of a controller axis parameter that's a signed long.
     * @param axis the axis.
     * @param axlong the axis parameter to set.
     * @param value the new value.
     */
    public void set(final AxisName axis, final AxisLong axlong, final long value) {
        setLongParameter(axlong.index(axis), value);
    }

    /**
     * Sets the value of a controller axis parameter that's a single-float (FP32).
     * @param axis the axis.
     * @param axsingle the axis parameter to set.
     * @param value the new value, which will be rounded to fit.
     */
    public void set(final AxisName axis, final AxisSingle axsingle, final double value) {
        setSingleParameter(axsingle.index(axis), value);
    }

    /**
     * Sets the value of an controller encoder parameter that's a signed 32-bit long.
     * @param encoder the encoder.
     * @param enclong the encoder parameter to set.
     * @param value the new value, which will be truncated on the left to fit.
     */
    public void set(final EncoderName encoder, final EncoderLong enclong, final long value) {
        setLongParameter(enclong.index(encoder), value);
    }

    /**
     * Sets the value of double scalar local to a controller program.
     * @param program the program.
     * @param locdouble the scalar variable to set.
     * @param value the new value.
     */
    public void set(final ProgramName program, final LocalDouble locdouble, final double value) {
        final long addr = localAddress(program.index(), DOUBLE_SCALAR, 0, locdouble.index());
        poke(addr, FP64_CONVERSION, encodeSingles(value));
    }

    /**
     * Sets the value of a double array local to a controller program.
     * @param program the program.
     * @param ldarray the local double array to set.
     * @param value the new values.
     */
    public void set(final ProgramName program, final LocalDoubleArray ldarray, final double[] value) {
        final long addr = localAddress(program.index(), DOUBLE_ARRAY, ldarray.index(), 0);
        checkElementCount(addr, false, value.length);
        poke(addr, FP64_CONVERSION, encodeSingles(value));
    }

    /**
     * Sets the value of a long (signed 32-bit) scalar local to a controller program.
     * @param program the program.
     * @param loclong the local variable to set.
     * @param value the new value, which will be truncated on the left to fit.
     */
    public void set(final ProgramName program, final LocalLong loclong, final long value) {
        final long addr = localAddress(program.index(), LONG_SCALAR, 0, loclong.index());
        poke(addr, LONG_CONVERSION, encodeLongs(value));
    }

    /**
     * Sets the values of a long (signed 32-bit) array local to a controller program.
     * @param program the program.
     * @param llarray the local array to set.
     * @param value the new values, which will be truncated on the left to fit.
     */
    public void set(final ProgramName program, final LocalLongArray llarray, final long[] value) {
      final long addr = localAddress(program.index(), LONG_ARRAY, llarray.index(), 0);
      checkElementCount(addr, false, value.length);
      poke(addr, LONG_CONVERSION, encodeLongs(value));
    }

    /**
     * Sets the value of a single-float scalar local to a controller program.
     * @param program the program.
     * @param locsingle the variable to set.
     * @param value the new value, which will be rounded down to single precision.
     */
    public void set(final ProgramName program, final LocalSingle locsingle, final double value) {
        final long addr = localAddress(program.index(), SINGLE_SCALAR, 0, locsingle.index());
        poke(addr, FP32_CONVERSION, encodeSingles(value));
    }

    /**
     * Sets the values of a single-float array local to a controller program.
     * @param program the program.
     * @param lsarray the array variable to set.
     * @param value the new values, which will be rounded down to single precision.
     */
    public void set(final ProgramName program, final LocalSingleArray lsarray, final double[] value) {
        final long addr = localAddress(program.index(), SINGLE_ARRAY, lsarray.index(), 0);
        checkElementCount(addr, false, value.length);
        poke(addr, FP32_CONVERSION, encodeSingles(value));
    }

    /**
     * Sets the value of a string scalar local to a controller program.
     * @param program the program.
     * @param locstring the string variable to set.
     * @param value the new value, where each Java char will be truncated down to eight bits.
     */
    public void set(final ProgramName program, final LocalString locstring, final String value) {
        // Get the address of the first element of the array of all string scalars.
        final long arrayAddr = localAddress(program.index(), STRING_SCALAR, 0, 0);
        // Fetch the element count and the per-string byte count limit.
        final long[] metadata = decodeUnsigneds(peek(arrayAddr - 2*ctrlType.getWordBump(), 2, LONG_CONVERSION));
        if (value.length() > metadata[1]) {
            throw new IllegalArgumentException("Attempt to overfill an AcroBAsic string.");
        }
        // Get the address of the target string scalar and poke the new value into place, padding
        // the length of the value out to a word boundary. Adding 1 to the word count
        // allows for the byte-count word that's part of every AcroBasic string value.
        final long addr = localAddress(program.index(), STRING_SCALAR, 0, locstring.index());
        poke(addr, LONG_CONVERSION, encodeStrings((value.length()+3)/4+1, value));
    }

    /**
     * Sets the value of a string array local to a controller program.
     * @param program the program.
     * @param lstrarray the string array to set.
     * @param value the new values, where each Java char will be truncated to eight bits.
     */
    public void set(final ProgramName program, final LocalStringArray lstrarray, final String[] value) {
        // Get the address of the first element of the string array.
        final long addr = localAddress(program.index(), STRING_ARRAY, lstrarray.index(), 0);
        // Find the length of the longest string in value and compare that
        // to the byte count limit in the array's metadata.
        final int maxSize = Arrays.stream(value).mapToInt(String::length).max().getAsInt();
        final long[] metadata = decodeUnsigneds(peek(addr - 2*ctrlType.getWordBump(), 2, LONG_CONVERSION));
        if (maxSize > metadata[1]) {
            throw new IllegalArgumentException("Attempt to overfill an AcroBasic string.");
        }
        // Also check that the array has at least as many elements as we want to set.
        if (value.length > metadata[0]) {
            throw new IllegalArgumentException("Attempt to overfill an AcroBasic array.");
        }
        final int wordsPerString = 1 + ((int)metadata[1] + 3) / 4;
        poke(addr, LONG_CONVERSION, encodeStrings(wordsPerString, value));
    }

    /**
     * Sets or clears a controller flag bit belonging to a master.
     * @param master the master.
     * @param mbit the flag bit to set/clear.
     * @param value true to set the bit, false to clear it.
     */
    public void set(final MasterName master, final MasterBit mbit, final boolean value) {
        setOrClearBit(mbit.index(master), value);
    }

    /**
     * Set or clear a system flag bit in the controller.
     * @param sysbit the flag bit to set/clear.
     * @param value true to set the bit, false to clear it.
     */
    public void set(final SystemBit sysbit, final boolean value) {
        setOrClearBit(sysbit.index(), value);
    }

    /**
     * Set in the controller a system parameter that is an unsigned 32-bit long.
     * @param sysul the parameter to set.
     * @param value the new value, which will be truncated on the left to fit.
     */
    public void set(final SystemUnsigned sysul, final long value) {
        setUnsignedParameter(sysul.index(), value);
    }

    /**
     * Sets a sequence of user parameters in the controller (always double-float).
     * @param uparam the first parameter to set.
     * @param value the new values.
     */
    public void set(final UserParameter uparam, final double... value) {
        poke(globalAddress(uparam), FP64_CONVERSION, encodeSingles(value));
    }




    ////////// Low-level communication with the controller. //////////

    // All multi-byte values in binary packets are little-endian.

    // Command codes for binary packets.
    private static final byte BINARY_GET_LONG = (byte)0x88; // Get long parameter.
    private static final byte BINARY_SET_LONG = (byte)0x89; // Set long parameter.
    private static final byte BINARY_GET_IEEE = (byte)0x8a; // Get single-float parameter.
    private static final byte BINARY_SET_IEEE = (byte)0x8b; // Set single-float parameter.
    private static final byte BINARY_PEEK = (byte)0x90; // Peek at controller memory.
    private static final byte BINARY_POKE = (byte)0x91; // Set controller memory.
    private static final byte BINARY_ADDRESS = (byte)0x92; // Get addresses of program-local variables.
    private static final byte BINARY_SET = (byte)0x1c; // Set bit flag.
    private static final byte BINARY_CLR = (byte)0x1d; // Clear bit flag.

    // Conversion codes for peek and poke commands. Note that FP32 and FP64 values
    // on the controller are both converted to single-precision IEEE values. In fact
    // the Parker manuals never do say what FP32 and FP64 look like, leaving them free
    // to change the implementation. On the Aries-xxCE controller it so happened that
    // the FP implementations were IEEE-compliant, making it possible to use peek and
    // poke to transfer FP64 values with full precision using long conversion. It's likely
    // that only obsolete hardware will have non-IEEE compliant floating-point implementations.
    private static final byte LONG_CONVERSION = 0; // 32-bit integer.
    private static final byte FP64_CONVERSION = 1; // Controller FP64 <-> IEEE *single* float.
    private static final byte FP32_CONVERSION = 2; // Controller FP32 <-> IEEE single float.

    // Variable type codes for use with the BINARY_ADDRESS command. For scalar variables
    // the address command returns the address of the contiguous region in which all
    // the scalar variables of the given type live for the given program.
    //
    // For numeric scalars the first word of the region is the number of variables
    // of that type, followed by the values.
    //
    // For string scalars the first word of the region is the number of
    // variables and the second is the max string size in bytes (applies to each string),
    // which are then followed by all the string values. Each string value begins with a
    // current-size-in-bytes word. The max-size word doesn't count the four bytes
    // of the current-size word. Each string is allocated a whole number of words.
    //
    // For array variables the address command returns the address of a region containing
    // 4-byte pointers to the arrays of the given type for the program. The first
    // word in this region counts the number of arrays of the given type. Each array's region
    // is laid out just like the region of all scalars of the same type. In fact we treat
    // the region containing all scalars of a given type as an array.
    private static final byte DOUBLE_SCALAR = 0;
    private static final byte DOUBLE_ARRAY = 1;
    private static final byte SINGLE_SCALAR = 2;
    private static final byte SINGLE_ARRAY = 3;
    private static final byte LONG_SCALAR = 4;
    private static final byte LONG_ARRAY = 5;
    private static final byte STRING_SCALAR = 6;
    private static final byte STRING_ARRAY = 7;

    private static final int WORDS_PER_LONG = 1;
    private static final int WORDS_PER_SINGLE = 1;
    private static final int WORDS_PER_DOUBLE = 2;



    /**
     * Uses the controller's System Pointer to calculate the address in controller memory
     * of a given user parameter. User parameters are stored at the beginning of the
     * global data area.
     * @param uparam the parameter.
     * @return the address of the parameter, or 0 if it doesn't exist.
     */
    public long globalAddress(final UserParameter uparam) {
        final long globalBase = decodeUnsigneds(peek(ctrlType.getSystemPointerAddress(), 1, LONG_CONVERSION))[0];
        final long wbump = ctrlType.getWordBump();
        // Global data area: one word counting the number of following global data words
        // followed by the data words themselves. Each item (user parameter) is
        // an FP64 value taking up two words.
        return globalBase + wbump * (1 + WORDS_PER_DOUBLE * uparam.index());
    }

    /**
     * Gets the address in controller memory of an element in a local array.
     * @param programNumber the index of the program whose data we want the address of.
     * @param typeCode the type code to use in the address command.
     * @param arrayNumber for array types, the index of the particular array we want. For scalar
     * types this parameter is ignored and we use the implicit array of all scalar variables
     * of the given type.
     * @param arrayIndex the index number of the array element we want.
     * @return the address of the first real element of the array, past any metadata.
     */
    public long localAddress(int programNumber, byte typeCode, int arrayNumber, int arrayIndex) {
        final long wordBump = ctrlType.getWordBump();
        long elementBump = 0;  // Array element size in addressable units (bytes or words).
        int elementCount = 0;  // No. of array elements.
        long[] metadata = null; // Array metadata (comes before first array element).

        // Get the address of the array of scalars (or pointers).
        final byte[] command = new byte[] {
            0,
            BINARY_ADDRESS,
            (byte)programNumber,
            typeCode
        };
        final byte[] response;
        sendBinary(command);
        response = receiveBinary(command.length + 4);
        // The address follows the 4-byte packet header.
        long addr = decodeUnsigneds(response)[1];

        // If we specified an scalar type then the address we have now is the address
        // of the implicit array of scalar variables of that type. If we specified
        // an array type then we have the address of an array of pointers to the
        // actual declared arrays of that type.
        if ((typeCode & 1) != 0) {
            // Array type. Get the pointer we want (index by arrayNumber).
            metadata = decodeUnsigneds(peek(addr, 1, LONG_CONVERSION));
            elementCount = (int)metadata[0]; // No. of pointers.
            elementBump = wordBump;
            if (arrayNumber < 0 || arrayNumber >= elementCount) {
                throw new IllegalArgumentException("No such array was declared.");
            }
            addr += wordBump + elementBump * arrayNumber; // Address of pointer.
            addr = decodeUnsigneds(peek(addr, 1, LONG_CONVERSION))[0]; // Fetch pointer.
        }

        // Now we have the address of the array we want, either the implicit array of all
        // scalars of a given type or an actual declared array. Use the metadata to get
        // the number and size of the array elements.
        switch (typeCode) {
            case LONG_SCALAR:
            case LONG_ARRAY:
            case SINGLE_SCALAR:
            case SINGLE_ARRAY:
                metadata = decodeUnsigneds(peek(addr, 1, LONG_CONVERSION));
                elementBump = wordBump;
                break;
            case DOUBLE_SCALAR:
            case DOUBLE_ARRAY:
                metadata = decodeUnsigneds(peek(addr, 1, LONG_CONVERSION));
                elementBump = 2 * wordBump;
                break;
            case STRING_ARRAY:
            case STRING_SCALAR:
                metadata = decodeUnsigneds(peek(addr, 2, LONG_CONVERSION));
                // The second word of metadata is the maximum number of bytes in
                // the string, which we round up to a whole number of words
                // since that's how much space the string is actually allocated.
                elementBump = wordBump * (metadata[1] + 3) / 4;
        }
        elementCount = (int)metadata[0];
        addr += wordBump * metadata.length;

        // Now we have the address of the first data element. If the array index is valid
        // then return the address of the desired element.
        if (arrayIndex < 0 || arrayIndex >= elementCount) {
            throw new ArrayIndexOutOfBoundsException("No such scalar or array element.");
        }
        return addr + arrayIndex * elementBump;
    }

    /**
     * Gets a controller LONG parameter as an unsigned value.
     * @param parameterIndex the index number of the parameter.
     * @return the parameter value.
     */
    public long getUnsignedParameter(final int parameterIndex) {
        return decodeUnsigneds(getParameterBytes(parameterIndex, BINARY_GET_LONG))[0];
    }

    /**
     * Sets a controller LONG parameter as an unsigned value.
     * @param parameterIndex the index number of the parameter.
     * @param value the parameter value.
     */
    public void setUnsignedParameter(final int parameterIndex, final long value) {
        setParameterBytes(parameterIndex, BINARY_SET_LONG, encodeUnsigneds(value));
    }

    /**
     * Gets a controller LONG parameter as a signed value.
     * @param parameterIndex the index number of the parameter.
     * @return the parameter value.
     */
    public long getLongParameter(final int parameterIndex) {
        return decodeLongs(getParameterBytes(parameterIndex, BINARY_GET_LONG))[0];
    }

    /**
     * Sets a controller LONG parameter as a signed value.
     * @param parameterIndex the index number of the parameter.
     * @param value the parameter value.
     */
    public void setLongParameter(final int parameterIndex, final long value) {
        setParameterBytes(parameterIndex, BINARY_SET_LONG, encodeLongs(value));
    }

    /**
     * Gets a controller FP32 (single-float) parameter.
     * @param parameterIndex the index number of the parameter.
     * @return the parameter value.
     */
    public double getSingleParameter(final int parameterIndex) {
        return decodeSingles(getParameterBytes(parameterIndex, BINARY_GET_IEEE))[0];
    }

    /**
     * Sets a controller FP32 (single-float) parameter.
     * @param parameterIndex the index number of the parameter.
     * @param value the parameter value.
     */
    public void setSingleParameter(final int parameterIndex, final double value) {
        setParameterBytes(parameterIndex, BINARY_SET_IEEE, encodeSingles(value));
    }

    /**
     * Gets the raw bytes of a controller non-user parameter value using a binary command.
     * Just the parameter value bytes are returned; the echoed command
     * bytes are stripped away.
     * @param parameterIndex the index number of the parameter.
     * @param commandCode the binary command code, either BINARY_GET_LONG
     * or BINARY_GET_IEEE.
     * @return the parameter value bytes from the controller response.
     */
    private byte[] getParameterBytes(final int parameterIndex, final byte commandCode) {
        final byte[] command = new byte[]{
            0,
            commandCode,
            (byte)parameterIndex,
            (byte)(parameterIndex >> 8)
        };
        sendBinary(command);
        final byte[] response = receiveBinary(command.length + 4);
        return Arrays.copyOfRange(response, command.length, response.length);
    }

    /**
     * Sets a controller non-user parameter to a value already encoded as bytes.
     * @param parameterIndex the index number of the parameter.
     * @param commandCode the binary command code, either BINARY_SET_IEEE or BINARY_SET_LONG.
     * @param encodedValue the new value, encoded as bytes.
     */
    private void setParameterBytes(
        final int parameterIndex,
        final byte commandCode,
        final byte[] encodedValue)
    {
        final byte[] command = new byte[]{
            0,
            commandCode,
            (byte)parameterIndex,
            (byte)(parameterIndex >> 8),
            encodedValue[0],
            encodedValue[1],
            encodedValue[2],
            encodedValue[3]
        };
        sendBinary(command);
    }

    /**
     * Sets or clears a controller bit according to a boolean value.
     * @param bitIndex the bit index number.
     * @param value true to set the bit, false to clear it.
     */
    public void setOrClearBit(final int bitIndex, final boolean value) {
        // Unlike the other binary commands, this one doesn't have zero in
        // the first byte. It's still non-printing ASCII control characters, though.
        final byte[] command = new byte[] {
            value ? BINARY_SET : BINARY_CLR,
            (byte)bitIndex,
            (byte)(bitIndex >> 8),
            0
        };
        sendBinary(command);
    }

    /**
     * Checks whether the number of new elements to be sent to a controller program's
     * local array is <= the dimension of the array.
     * @param firstElementAddress the address of the array's first element.
     * @param isString true if this is an array of strings, else false.
     * @param newElementCount the number of new values to be sent.
     * @throws IllegalArgumentException if the check fails.
     */
    private void checkElementCount(
        final long firstElementAddress,
        final boolean isString,
        final int newElementCount
    )
    {
        final int maxElements = arrayElementCount(firstElementAddress, isString);
        if (newElementCount > maxElements) {
            throw new IllegalArgumentException("Attempt to overfill an AcroBasic array.");
        }
    }

    /**
     * Gets the dimension of a controller program's local array from the array's metadata.
     * @param firstElementAddress the address of the array's first element.
     * @param isString true if it's a string array, else false.
     * @return the number of elements dimensioned for the array.
     */
    private int arrayElementCount(final long firstElementAddress, final boolean isString) {
        // For string arrays the element count is two words before the first
        // element, for all other types it's just one word before.
        final long countAddress =
            firstElementAddress
            - ctrlType.getWordBump() * (isString ? 2 : 1);
        return (int)decodeUnsigneds(peek(countAddress, 1, LONG_CONVERSION))[0];
    }

    /**
     * Gets the number of words taken up by an AcroBasic string. The first word
     * of the string value is a count of the actual number of characters
     * currently in use followed by the characters themselves. We round
     * this up to a whole number of 4-byte words and add one for
     * the byte count itself.
     * @param address the address of the string in controller memory.
     * @return the word count.
     */
    private int stringActualWordCount(final long address) {
        final int byteCount = (int)decodeUnsigneds(peek(address, 1, LONG_CONVERSION))[0];
        return 1 + (byteCount + 3) / 4;
    }

    /**
     * Encodes as bytes a series of 32-bit unsigned integer values. Each value is encoded
     * as four bytes in little-endian order.
     * @param value a series of values between 0 and 2**32 - 1 inclusive.
     * @return the encoded values, end to end, in the order given.
     */
    public byte[] encodeUnsigneds(long... value) {
        final byte[] result = new byte[4 * value.length];
        int i = 0;
        for (long v: value) {
            result[i++] = (byte)v;
            result[i++] = (byte)(v >> 8);
            result[i++] = (byte)(v >> 16);
            result[i++] = (byte)(v >> 24);
        }
        return result;
    }

    /**
     * Decode a byte sequence sent from the controller as a series of unsigned 32-bit
     * integer values. The byte order is assumed to be little-endian.
     * @param value the series of bytes.
     * @return the decoded values, where value i is derived from bytes 4*i through 4*i+3
     * inclusive.
     */
    public long[] decodeUnsigneds(byte... value) {
        final long[] result = new long[(value.length + 3) / 4];
        int i = 0;
        int shift = 0;
        long r = 0;
        for (byte v: value) {
            r |= Byte.toUnsignedLong(v) << shift;
            shift += 8;
            if (shift >= 32) {
                // Save the old long and begin a new one.
                shift = 0;
                result[i++] = r;
                r = 0;
            }
        }
        if (shift != 0) result[i++] = r; // Save any leftover accumulated result.
        return result;
    }

    /**
     * Encodes as bytes a series of 32-bit signed integer values. Each value is encoded
     * as four bytes in little-endian order.
     * @param value a series of values between -2**31 and 2**31 - 1 inclusive.
     * @return the encoded values, end to end, in the order given.
     */
    public byte[] encodeLongs(long... value) {
        return encodeUnsigneds(value);
    }


    /**
     * Decode a byte sequence sent from the controller as a series of signed 32-bit
     * integer values. The byte order is assumed to be little-endian.
     * @param value the series of bytes.
     * @return the decoded values, where value i is derived from bytes 4*i through 4*i+3
     * inclusive.
     */
    public long[] decodeLongs(byte... value) {
        final long[] result = new long[(value.length + 3) / 4];
        int i = 0;
        int shift = 0;
        long r = 0;
        for (byte v: value) {
            // Preserve the extended sign of the high-order byte.
            r |= ((shift == 24) ? (long)v : ((long)v & 0xffL)) << shift;
            shift += 8;
            if (shift >= 32) {
                shift = 0;
                result[i++] = r;
                r = 0;
            }
        }
        if (shift != 0) result[i++] = r;
        return result;
    }

    /**
     * Encodes as bytes a series of double values after conversion
     * to floats. Each value is rounded to single precision
     * and then converted to an integer using {@code Float.floatToIntBits(}}.
     * Each integer is encoded using {@link #encodeUnsigneds(long...)}.
     * @param value a series of double values.
     * @return the encoded values, end to end, in the order given.
     */
    public byte[] encodeSingles(double... value) {
        final long[] asLong = new long[value.length];
        int i = 0;
        for (double v: value) {
            asLong[i++] = Float.floatToIntBits((float)v);
        }
        return encodeUnsigneds(asLong);
    }

    /**
     * Decodes a series of float values sent from the controller
     * as bytes. First the bytes are converted to integers
     * using {@link #decodeUnsigneds(byte...)}. The integers are then
     * converted to floats using {@code Float.intBitsToFloat(}} and
     * the resulting floats are widened to doubles.
     * @param value a series of bytes.
     * @return the decoded and widened values.
     */
    public double[] decodeSingles(byte... value) {
        if (value.length % 4 != 0) {
            throw new IllegalArgumentException("decodeSingles() needs a multiple of four bytes.");
        }
        final long[] asLong = decodeUnsigneds(value);
        final double[] result = new double[asLong.length];
        int i = 0;
        for (long al: asLong) {
            result[i++] = (double)(Float.intBitsToFloat((int)al));
        }
        return result;
    }


    /**
     * Encodes as bytes a series of double values.
     * Each value is converted to a long using {@code Double.doubleToLongBits(}}.
     * Each long is broken into two 32-bit halves and encoded using
     * {@link #encodeUnsigneds(long...)}, the low -order half being encoded first.
     * @param value a series of double values.
     * @return the encoded values, end to end, in the order given.
     */
    public byte[] encodeDoubles(double... value) {
        final long[] asLong = new long[WORDS_PER_DOUBLE * value.length];
        int i = 0;
        for (double v: value) {
            final long l = Double.doubleToLongBits(v);
            asLong[i++] = l & 0xffffffffL;
            asLong[i++] = l >>> 32;
        }
        return encodeUnsigneds(asLong);
    }

    /**
     * Decodes as a double each 8-byte section of a byte string. First the bytes
     * are assembled into 32-bit integers using {@link #decodeUnsigneds(byte...)}.
     * Each pair of integers is combined into a single long, the first member
     * of each pair becoming the low order part. Then the longs are converted to
     * doubles using {@code Double.longBitsToDouble(}}.
     * @param value the series of bytes to decode.
     * @return the decoded doubles.
     */
    public double[] decodeDoubles(byte... value) {
        if (value.length % 8 != 0) {
            throw new IllegalArgumentException("decodeDoubles() needs a multiple of 8 bytes.");
        }
        final long[] asLong = decodeUnsigneds(value);
        final double[] result = new double[asLong.length / 2];
        for (int i = 0; i < result.length; ++i) {
            result[i] = Double.longBitsToDouble((asLong[2*i+1]<<32) | asLong[2*i]);
        }
        return result;
    }

    /**
     * Encodes a series of Strings as a byte sequence. Each string's encoding is
     * zero-padded to a given multiple of four bytes; this is needed when
     * poking a series of strings into an array since each array element must
     * have the same size, and space allocation by AcroBasic is always in units
     * of words. Encoding of strings to bytes is done using ISO-8859-1 so
     * any character value not in the range 0-255 is converted to a question mark.
     * We follow the AcroBasic convention of representing each string value
     * as a 1-word byte count followed by the encoded string bytes.
     * @param elementWordCount the number of words to use for each string
     * (including the byte count word), padding with zero-valued bytes if needed.
     * @param value the series of strings.
     * @return the encoded strings in the given order, one after another.
     */
    public byte[] encodeStrings(final int elementWordCount, final String... value) {
        final int elementByteCount = 4 * elementWordCount;
        final byte[] result = new byte[elementByteCount * value.length];
        Arrays.fill(result, (byte)0);
        int resultNext = 0;
        for (String v: value) {
            final byte[] byteCount = encodeUnsigneds(v.length());
            final byte[] stringBytes = v.getBytes(StandardCharsets.ISO_8859_1);
            System.arraycopy(byteCount, 0, result, resultNext, byteCount.length);
            System.arraycopy(
                stringBytes, 0,
                result, resultNext + byteCount.length,
                stringBytes.length);
            resultNext += elementByteCount;
        }
        return result;
    }

    /**
     * Reverses the encoding performed by {@link #encodeStrings(int, java.lang.String...)},
     * turning AcroBasic strings represented a byte sequences into Java Strings.
     * @param elementWordCount the number of 4-byte words occupied by each
     * AcroBasic string, which includes the byte-count word.
     * @param value the byte array containing encoded string values.
     * @return the array of decoded strings.
     */
    public String[] decodeStrings(final int elementWordCount, byte... value) {
        final List<String> result = new ArrayList<>();
        final int elementByteCount = 4 * elementWordCount;
        int i = 0;
        while (i < value.length) {
            final int byteCount = (int)decodeUnsigneds(value[i], value[i+1], value[i+2], value[i+3])[0];
            result.add(new String(
                Arrays.copyOfRange(value, i+4, i+4+byteCount),
                StandardCharsets.ISO_8859_1));
            i += elementByteCount + 4;
        }
        return result.toArray(new String[]{});
    }

    /**
     * Uses the controller's binary-peek operation to get a number of 32-bit words
     * from memory.
     * @param address the memory address of the first, lowest-addressed word.
     * @param wordCount the number of words to get.
     * @param conversion the conversion code to use, for instance LONG_CONVERSION.
     * @return the words, as a Java byte array. Each word is in little-endian order.
     */
    public byte[] peek(long address, int wordCount, final byte conversion) {
        // No matter the conversion there are always four bytes per item. That's
        // because FP64_CONVERSION returns single-precision IEEE.
        final byte[] result = new byte[4 * wordCount];
        int resultNext = 0;
        while (wordCount > 0) {
            byte[] command;
            byte[] response;
            final int n = Math.min(wordCount, 255);
            final long header =
                    (Byte.toUnsignedLong(BINARY_PEEK)<<8)
                    |(Byte.toUnsignedLong(conversion)<<16)
                    |(Integer.toUnsignedLong(n)<<24);
            command = encodeUnsigneds(header, address);
            sendBinary(command);
            response = receiveBinary(command.length + 4 * n);
            System.arraycopy(
                response,
                command.length, // Skip echoed command at start of response.
                result,
                resultNext,
                response.length - command.length);
            resultNext += response.length - command.length;
            wordCount -= n;
            address += n * ctrlType.getWordBump();
        }
        return result;
    }


    /**
     * Uses the controller's binary-poke operation to set a number of 32-bit words
     * from memory.
     * @param address the memory address of the first, lowest-addressed word.
     * @param conversion one of the conversion codes, for instance FP64_CONVERSION.
     */
    public void poke(long address, final byte conversion, final byte... value) {
        if (value.length % 4 != 0) {
            throw new IllegalArgumentException("The byte count for poke() is not a multiple of four.");
        }
        int valueNext = 0;
        int wordCount = value.length / 4;
        while (wordCount > 0) {
            final int n = Math.min(wordCount, 255);
            final byte[] command = new byte[8 + 4 * n];
            final long header =
                    (Byte.toUnsignedLong(BINARY_POKE)<<8)
                    |(Byte.toUnsignedLong(conversion)<<16)
                    |(Integer.toUnsignedLong(n)<<24);
            System.arraycopy(
                encodeUnsigneds(header, address),
                0,
                command,
                0, 8);
            System.arraycopy(value, valueNext, command, 8, 4 * n);
            sendBinary(command);
            valueNext += 4 * n;
            wordCount -= n;
            address += n * ctrlType.getWordBump();
        }
    }

   /**
    * Reads a file line by line and sends each line to the controller.
    * @param fName the pathname of the file.
    * @param echo true to have the controller echo what we send it, else false.
    */
    public void sendFile(final String fName, final boolean echo)
    {
        try (Stream<String> stream = Files.lines(Paths.get(fName))) {
            // Change the controller's echo setting.
            if (echo) sendStr("echo 0"); // Prompts, error msgs but no echo.
            stream.forEachOrdered(line -> {
                sendStr(line);
            });
        }
        catch (IOException exc) {
            throw new AcrComm.Exception(exc);
        }
        finally {
            // Restore the controller's echo setting to the default.
            sendStr("echo 1"); // Prompts, error msgs and character echo.
        }
    }

    /**
     * Sends an ASCII line to the controller and receives any reply, which is sent to the console.
     * If logging is in effect then both sends and receives are logged. String codepoints
     * are translated to and from bytes using the seven-bit US-ASCII encoding.
     * @param cmnd the command line, without the line termination which will be added.
     * @return the number of characters received in reply.
     */
    public int sendStr(final String cmnd)
    {
        final byte[] data = cmnd.getBytes(StandardCharsets.US_ASCII);
        try {
            // First put in the bytes and add the line termination.
            commandOut.write(data);
            commandOut.write("\r".getBytes(StandardCharsets.US_ASCII));
            // We're using the binary command port which requires that ASCII
            // commands start with a printable character and be padded with NUL
            // to a multiple of four bytes.
            int nsent = data.length + 1;
            while (nsent % 4 != 0) {
                commandOut.write(0);
                ++nsent;
            }
            commandOut.flush();
            logStr(cmnd);
        } catch (IOException e) {
            throw new AcrComm.Exception(e);
        }
        return receiveAscii();
    }

    /**
     * Collects ASCII data coming from the controller, relaying it to the console
     * and, if logging is enabled, logging it.
     * @return the number of characters collected, or -1 if EOF was seen.
     */
    public int receiveAscii() {
        IOException savedExc = null;
        final StringBuilder builder = new StringBuilder();
        int ch = 0;
        int count = 0;
        try {
           while (true) {
                ch = commandIn.read();
                if (ch < 0) {
                    break; // EOF
                }
                builder.append((char) ch);
                ++count;
            }
        } catch (SocketTimeoutException exc) {
            // Stop collecting input.
        } catch (IOException exc) {
            // Save the exception for later.
            savedExc = exc;
        }
        final String collected = builder.toString();
        out.ifPresent(console -> console.println(collected));
        logAscii(collected);

        // Throw an unchecked exception wrapping any saved exception.
        if (savedExc != null) {
            throw new AcrComm.Exception(savedExc);
        }

        return (ch < 0) ? -1 : count;
    }


   /**
    * Sends a binary command to the controller.
    * @param cmnd the byte array containing the encoded command, which is assumed
    * to take up the entire array.
    */
    public void sendBinary(final byte[] cmnd)
    {
        try {
            commandOut.write(cmnd, 0, cmnd.length);
            commandOut.flush();
            logBin(true, cmnd, cmnd.length);
        }
        catch (IOException e) {
            throw new AcrComm.Exception(e);
        }
    }


   /** Receives a binary reply from the controller
    *  @param leng the expected length of the reply.
    */
    public byte[] receiveBinary(final int leng)
    {
        int nRead, offs = 0;
        final byte[] buff = new byte[leng];
        while (offs < leng) {
            try {
                nRead = commandIn.read(buff, offs, leng - offs);
                if (nRead < 0) {
                    throw new IOException("Premature EOF on binary receive.");
                }
            }
            catch (SocketTimeoutException e) {
                throw new AcrComm.Exception(e);
            }
            catch (IOException e) {
                throw new AcrComm.Exception(e);
            }
            offs += nRead;
        }
        logBin(false, buff, offs);
        return buff;
    }


   /** Discards any unread input from the controller socket. */
    public void drain() {
        if (binSock == null) {
            return;
        }

        try {
            final byte[] buff = new byte[64];
            int nRead = 1;
            while (nRead > 0) {
                nRead = commandIn.read(buff, 0, buff.length);
                if (nRead > 0) {
                    logBin(false, buff, 0);
                }
            }
        } catch (SocketTimeoutException e) {
            // No real problem.
        } catch (IOException e) {
            throw new AcrComm.Exception(e);
        }
    }


   /**
    * Logs at the DEBUG level the binary data sent to or received from the controller.
    * @param sent true if we sent this data, false if we received it.
    * @param buff the buffer of bytes sent or received.
    * @param leng the number of bytes in the buffer.
    */
    private void logBin(boolean sent, byte[] buff, int leng) {
        if (log.isDebugEnabled()) {
            final int bytesPerGroup = 4;
            final int bytesPerLine = bytesPerGroup * 8;
            final StringBuilder display = new StringBuilder(sent ? "cmnd\n" : "resp\n");
            for (int j = 0; j < leng; j++) {
                display.append(String.format("%02x", buff[j]));
                if (j % bytesPerLine == bytesPerLine - 1) {
                    display.append("\n");
                }
                else if (j % bytesPerGroup == bytesPerGroup -1) {
                    display.append(" ");
                }
            }
            if (leng % bytesPerLine != 0) {
                display.append("\n");
            }
            log.debug(display.toString());
        }
    }


    /**
     * Logs at the DEBUG level the ASCII text received from the controller.
     * @param text the collected text.
     */
    private void logAscii(final String text) {
        log.debug(text);
        if (!text.endsWith("\n")) log.debug("\n");
    }


    /**
     * Logs at the DEBUG level an ASCII command sent to the controller.
     * @param cmnd the command string.
     */
    private void logStr(final String cmnd) {
        log.debug("Asc cmnd: " + cmnd);
    }
}
