package org.lsst.ccs.drivers.parker;

import org.lsst.ccs.utilities.conv.Convert;
import org.lsst.ccs.utilities.sa.Output;
import org.lsst.ccs.utilities.sa.ConsOut;
import java.net.Socket;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.io.*;
import java.util.GregorianCalendar;

/**
 ***************************************************************************
 **
 **  Routines for communicating with a Parker (ACR) motor controller
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public final class AcrComm {


    public final static int
        ACO_GET_RESPONSE = 0x01,
        ACO_ECHO_COMMAND = 0x02,
        ACO_SHOW_PROMPT  = 0x04,
        ACT_CONV_INT     = 0,
        ACT_CONV_FP64    = 1,
        ACT_CONV_FP32    = 2,
        ACV_DV           = 0,
        ACV_DA           = 1,
        ACV_SV           = 2,
        ACV_SA           = 3,
        ACV_LV           = 4,
        ACV_LA           = 5,
        ACV_$V           = 6,
        ACV_$A           = 7;

    public final static int
        INP_LINE_PRM    = 4096,
        PRIM_AXIS_PRM   = 4120,
        ENC_POSN_PRM    = 6144,
        SAMP_PERD_PRM   = 6915,
        SYS_CLOCK_PRM   = 6916,
        CAP_POSN_PRM    = 12292,
        PGAIN_PRM       = 12304,
        IGAIN_PRM       = 12305,
        DGAIN_PRM       = 12308,
        POS_TLM_PRM     = 12328,
        NEG_TLM_PRM     = 12329,
        JOG_VEL_PRM     = 12348,
        JOG_ACC_PRM     = 12349,
        JOG_DEC_PRM     = 12350,
        JOG_JRK_PRM     = 12351,
        PPU_PRM         = 12375,
        CMDD_CURR_PRM   = 28738,
        CMDD_TORQ_PRM   = 28739,
        ACTL_TORQ_PRM   = 28740,
        DRIVE_TEMP_PRM  = 28743,
        BUS_VOLT_PRM    = 28745,
        TEMP_INT_PRM    = 39167,
        TEMP_FLT_PRM    = 39423;

    public final static int
        INPUT4_BIT     = 4,
        OUTPUT33_BIT   = 33,
        SAMP_ARM_BIT   = 104,
        SAMP_ACTV_BIT  = 105,
        SAMP_MODE_BIT  = 106,
        MOVE_SYNC_BIT  = 128,
        CAL_SYNC_BIT   = 129,
        KILL_MOVE_BIT  = 522,
        CAP_CMPL_BIT   = 777,
        JOG_ACTV_BIT   = 792,
        LINE_NUM_BIT   = 5651,
        KILL_MOTN_BIT  = 8467,
        POSN_ERR_BIT   = 8479,
        POS_EOT_BIT    = 16132,
        NEG_EOT_BIT    = 16133,
        POS_SLM_BIT    = 16140,
        NEG_SLM_BIT    = 16141,
        HOME_FND_BIT   = 16134,
        HOME_FAIL_BIT  = 16135,
        INVT_PEOT_BIT  = 16144,
        INVT_NEOT_BIT  = 16145,
        INVT_HOME_BIT  = 16146,
        ENAB_PEOT_BIT  = 16148,
        ENAB_NEOT_BIT  = 16149,
        ENAB_PSLM_BIT  = 16150,
        ENAB_NSLM_BIT  = 16151,
        HOME_BKUP_BIT  = 16152,
        HOME_NEDG_BIT  = 16153,
        HOME_NFNL_BIT  = 16154;

    public final static int
        ICM_RISE_1ST   = 2,
        ICM_FALL_1ST   = 6,
        ICM_RISE_2ND   = 3,
        ICM_FALL_2ND   = 7,
        ICM_RISE_3RD   = 10,
        ICM_FALL_3RD   = 14,
        ICM_RISE_4TH   = 11,
        ICM_FALL_4TH   = 15;

    private final static int
        SOCK_MGMT    = 5004,
        SOCK_BIN     = 5006,
        TMO_NONE     = 0,
        TMO_SHORT    = 100,
        TMO_LONG     = 2000,
        L_PROMPT     = 4,
        CON_STAT_PRM = 7680,
        CON_ADDR_PRM = 7681,
        CON_PORT_PRM = 7682,
        CON_STRM_PRM = 7684;

    private final static byte
        LF  = '\n',
        CR  = '\r',
        GT  = '>',
        PEE = 'P';

    private final ControllerType ctrlType;
    private Output out;
    private Socket mgmtSock = null, binSock = null;
    private InputStream mgmtInp, binInp;
    private OutputStream mgmtOut, binOut;
    private String prompt = "ERR>";
    private Cleanup clean = new Cleanup();
    private PrintStream log = null;
    private Thread watchdog = new Thread(new Watchdog());
    private boolean logLF = true, exiting = false;
    private int binPort, ipAddr = 0, acrStream = 0;


   /**
    ***************************************************************************
    **
    **  Inner exception class for laundering non-runtime exceptions
    **
    ***************************************************************************
    */
    public class Exception extends RuntimeException {

        public Exception(Throwable e)
        {
            super(e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Inner class to implement a cleanup thread
    **
    ***************************************************************************
    */
    private class Cleanup extends Thread {

        @Override
        public void run()
        {
            exiting = true;

            try {
                drain();
            }
            catch (RuntimeException e) {
                out.println("Error draining sockets");
            }

            try {
                if (mgmtSock != null) mgmtSock.close();
            }
            catch (IOException e) {
                out.println("Error closing watchdog socket");
            }

            try {
                if (binSock != null) binSock.close();
            }
            catch (IOException e) {
                out.println("Error closing binary socket");
            }

            if (log != null) log.close();
        }
    }


   /**
    ***************************************************************************
    **
    **  Inner class to implement motor connection watchdog
    **
    ***************************************************************************
    */
    private class Watchdog implements Runnable {

        @Override
        public void run()
        {
            int period = 5000;
            byte[] cmnd = new byte[24], resp = new byte[24];
            Convert.intToBytesBE(1, cmnd, 0);
            Convert.intToBytesBE(period, cmnd, 4);
            Convert.intToBytesBE(3, cmnd, 8);
            Convert.intToBytesBE(ipAddr, cmnd, 12);
            Convert.intToBytesBE(binPort, cmnd, 16);
            Convert.intToBytesBE(0, cmnd, 20);

            /*
            **  Loop until program is exiting
            */
            while (!exiting) {

                /*
                **  Send the watchdog packet
                */
                try {
                    mgmtOut.write(cmnd, 0, cmnd.length);
                }
                catch (IOException e) {
                    throw new AcrComm.Exception(e);
                }

                /*
                **  Wait for the response
                */
                int nRead, offs = 0;

                while (true) {
                    try {
                        nRead = mgmtInp.read(resp, offs, resp.length - offs);
                    }
                    catch (SocketTimeoutException e) {
                        nRead = 0;
                    }
                    catch (IOException e) {
                        throw new AcrComm.Exception(e);
                    }
                    if (nRead > 0) offs += nRead;
                    if (offs >= resp.length || nRead <= 0) break;
                }

                /*
                **  Sleep for a while
                */
                try {
                    Thread.sleep(period);
                }
                catch (InterruptedException e) {
                    break;
                }
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Main constructor
    **
    ***************************************************************************
    */
    public AcrComm(ControllerType ctrlType, String node, boolean logging, Output iOut)
    {
        this.ctrlType = ctrlType;

        /*
        **  Set up shutdown hook to clean up network connection
        */
        Runtime.getRuntime().addShutdownHook(clean);

        /*
        **  Open log file if requested to
        */
        if (logging) logOpen(null);

        /*
        **  Set up console output
        */
        out = (iOut != null) ? iOut : new ConsOut();

        /*
        **  Connect to connection management socket and get streams
        */
        try {
            mgmtSock = new Socket(node, SOCK_MGMT);
            mgmtInp = mgmtSock.getInputStream();
            mgmtOut = mgmtSock.getOutputStream();
        }
        catch (UnknownHostException e) {
            throw new AcrComm.Exception(e);
        }
        catch (SocketException e) {
            throw new AcrComm.Exception(e);
        }
        catch (IOException e) {
            throw new AcrComm.Exception(e);
        }

        /*
        **  Connect to binary stream socket and get streams
        */
        try {
            binSock = new Socket(node, SOCK_BIN);
            binInp = binSock.getInputStream();
            binOut = binSock.getOutputStream();
        }
        catch (UnknownHostException e) {
            throw new AcrComm.Exception(e);
        }
        catch (SocketException e) {
            throw new AcrComm.Exception(e);
        }
        catch (IOException e) {
            throw new AcrComm.Exception(e);
        }

        /*
        **  Wait for unsolicited input; if none, provoke some
        */
        try {
            binSock.setSoTimeout(TMO_LONG);
            int leng = rcveAsc(ACO_ECHO_COMMAND);
            if (leng == 0)
                leng = sendStr("ver", 0);
            binSock.setSoTimeout(TMO_NONE);
        }
        catch (SocketException e) {
            throw new AcrComm.Exception(e);
        }

        /*
        **  Get stream number and display message
        */
        binPort = binSock.getLocalPort();
        byte[] bAddr = binSock.getLocalAddress().getAddress();
        for (int j = 0; j < bAddr.length; j++)
            ipAddr = (ipAddr << 8) | (bAddr[j] & 0xff);
        for (int j = 0; j < 24; j += 8) {
            int status = getIntParm(CON_STAT_PRM + j);
            if (status != 1 && status != 2) continue;
            if (binPort != getIntParm(CON_PORT_PRM + j)) continue;
            if (ipAddr != getIntParm(CON_ADDR_PRM + j)) continue;
            acrStream = getIntParm(CON_STRM_PRM + j);
            break;
        }
        out.println("Connected to stream " + acrStream);

        /*
        **  Start the connection watchdog
        */
        watchdog.start();
    }

    public AcrComm(ControllerType ctrlType, String node)
    {
        this(ctrlType, node, false, null);
    }

    public AcrComm(ControllerType ctrlType, String node, boolean logging)
    {
        this(ctrlType, node, logging, null);
    }

    public AcrComm(ControllerType ctrlType, String node, Output iOut)
    {
        this(ctrlType, node, false, iOut);
    }


   /**
    ***************************************************************************
    **
    **  Get the stored prompt
    **
    ***************************************************************************
    */
    public String getPrompt()
    {
        return prompt;
    }


   /**
    ***************************************************************************
    **
    **  Get the stored connection stream number
    **
    ***************************************************************************
    */
    public int getStream()
    {
        return acrStream;
    }


   /**
    ***************************************************************************
    **
    **  Read a file and send it to the controller
    **
    ***************************************************************************
    */
    public void sendFile(String fName, int optns)
    {
        int off = 0;
        byte[] buff = new byte[1000], save = new byte[3];
        FileInputStream file;

        try {
            file = new FileInputStream(fName);
        }
        catch (FileNotFoundException e) {
            out.println(e);
            return;
        }

        if ((optns & ACO_ECHO_COMMAND) == 0)
            sendStr("echo 0", 0);

        while (true) {
            int leng, txt = 0, end, pad, j;

            try {
                leng = file.read(buff, off, buff.length - 3 - off);
            }
            catch (IOException e) {
                out.println(e);
                break;
            }
            if (leng < 0) break;
            leng += off;
            while (true) {
                for (end = txt; end < leng && buff[end] != LF; end++);
                if (end < leng) {
                    int lEnd = end;
                    buff[end++] = CR;
                    if ((optns & ACO_SHOW_PROMPT) != 0)
                        out.print(prompt);
                    if ((pad = (end - txt)  & 3) != 0) {
                        pad = 4 - pad;
                        for (j = 0; j < pad; j++) {
                            save[j] = buff[end];
                            buff[end++] = 0;
                        }
                    }
                    try {
                        binOut.write(buff, txt, end - txt);
                        if (log != null) logAscii(true, true, buff, txt,
                                                  lEnd - txt);
                    }
                    catch (IOException e) {
                        throw new AcrComm.Exception(e);
                    }
                    for (j = pad - 1; j >= 0; j--)
                        buff[--end] = save[j];
                    rcveAsc(ACO_ECHO_COMMAND);
                    txt = end;
                }
                else {
                    off = leng - txt;
                    if (off > 0)
                        System.arraycopy(buff, txt, buff, 0, off);
                    break;
                }
            }
        }

        if ((optns & ACO_ECHO_COMMAND) == 0)
            sendStr("echo 1", ACO_ECHO_COMMAND);

        try {
            file.close();
        }
        catch (IOException e) {
        }
    }


   /**
    ***************************************************************************
    **
    **  Send a string to the controller and receive reply
    **
    ***************************************************************************
    */
    public int sendStr(String cmnd, int optn)
    {
        if (cmnd != null) {
            int pad = 4 - (cmnd.length() & 3);
            try {
                binOut.write((cmnd + "\r\0\0\0".substring(0, pad)).getBytes());
                if (log != null) logStr(cmnd);
            }
            catch (IOException e) {
                throw new AcrComm.Exception(e);
            }
        }
        return rcveAsc(optn);
    }


   /**
    ***************************************************************************
    **
    **  Receive and process ASCII data from the controller
    **
    ***************************************************************************
    */
    public int rcveAsc(int optn)
    {
        int leng, offs = 0, txt, prm;
        byte[] buff = new byte[1000];
        boolean skip = (optn & ACO_ECHO_COMMAND) == 0, first = true;

        while (true) {
            try {
                leng = binInp.read(buff, offs, buff.length - offs);
                if (log != null) {
                    logAscii(false, first, buff, offs, leng);
                    first = false;
                }
            }
            catch (SocketTimeoutException e) {
                out.write(buff, 0, offs);
                return 0;
            }
            catch (IOException e) {
                out.write(buff, 0, offs);
                throw new AcrComm.Exception(e);
            }
            if (leng < 0) {
                out.write(buff, 0, offs);
                return -1;
            }
            leng += offs;
            txt = 0;
            if (skip) {
                for (; txt < leng && buff[txt] != LF; txt++);
                if (txt < leng) {
                    txt++;
                    skip = false;
                }
            }
            for (prm = leng; prm > 0 && buff[prm - 1] != LF; prm--);
            if (txt < prm)
                out.write(buff, txt, prm - txt);
            offs = leng - prm;
            if (prm > 0)
                System.arraycopy(buff, prm, buff, 0, offs);
            leng = offs - L_PROMPT;
            if (leng >= 0 && buff[offs - 1] == GT) {
                String tPrompt = new String(buff, leng, L_PROMPT);
                if (tPrompt.equals("SYS>")
                      || (buff[leng] == PEE && Character.isDigit(buff[leng + 1])
                            && Character.isDigit(buff[leng + 2]))) {
                    if (leng > 0) out.write(buff, 0, leng);
                    prompt = tPrompt;
                    return 1;
                }
            }
            if (offs >= buff.length) {
                out.write(buff, 0, buff.length);
                offs = 0;
                skip = false;
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Send binary command to the controller
    **
    ***************************************************************************
    */
    public void sendBin(byte[] cmnd)
    {
        try {
            binOut.write(cmnd, 0, cmnd.length);
            if (log != null) logBin(true, cmnd, cmnd.length);
        }
        catch (IOException e) {
            throw new AcrComm.Exception(e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Receive binary data from the controller
    **
    ***************************************************************************
    */
    public int rcveBin(byte[] buff, int leng)
    {
        int nRead, offs = 0;

        while (true) {
            try {
                nRead = binInp.read(buff, offs, leng - offs);
            }
            catch (SocketTimeoutException e) {
                nRead = 0;
            }
            catch (IOException e) {
                throw new AcrComm.Exception(e);
            }
            if (nRead < 0) return -1;
            offs += nRead;
            if (offs >= leng || nRead == 0) {
                if (log != null) logBin(false, buff, offs);
                return offs;
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Get a parameter (int or float) from the controller as int
    **
    ***************************************************************************
    */
    public int getParmAsInt(int pnum)
    {
        byte[] cmnd = new byte[4], resp = new byte[8];

        encodePnum(pnum, cmnd);
        sendBin(cmnd);
        rcveBin(resp, resp.length);
        return Convert.bytesToInt(resp, 4);
    }


   /**
    ***************************************************************************
    **
    **  Get an integer parameter from the controller
    **
    ***************************************************************************
    */
    public int getIntParm(int pnum)
    {
        byte[] cmnd = new byte[4], resp = new byte[8];

        cmnd[0] = 0;
        cmnd[1] = (byte)0x88;
        Convert.shortToBytes((short)pnum, cmnd, 2);
        sendBin(cmnd);
        rcveBin(resp, resp.length);
        return Convert.bytesToInt(resp, 4);
    }


   /**
    ***************************************************************************
    **
    **  Set an integer parameter into the controller
    **
    ***************************************************************************
    */
    public void setIntParm(int pnum, int value)
    {
        byte[] cmnd = new byte[8];

        cmnd[0] = 0;
        cmnd[1] = (byte)0x89;
        Convert.shortToBytes((short)pnum, cmnd, 2);
        Convert.intToBytes(value, cmnd, 4);
        sendBin(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Get a float parameter from the controller as integer
    **
    ***************************************************************************
    */
    public int getFloatParmAsInt(int pnum)
    {
        byte[] cmnd = new byte[4], resp = new byte[8];

        cmnd[0] = 0;
        cmnd[1] = (byte)0x8a;
        Convert.shortToBytes((short)pnum, cmnd, 2);
        sendBin(cmnd);
        rcveBin(resp, resp.length);
        return Convert.bytesToInt(resp, 4);
    }


   /**
    ***************************************************************************
    **
    **  Get a float parameter from the controller
    **
    ***************************************************************************
    */
    public float getFloatParm(int pnum)
    {
        byte[] cmnd = new byte[4], resp = new byte[8];

        cmnd[0] = 0;
        cmnd[1] = (byte)0x8a;
        Convert.shortToBytes((short)pnum, cmnd, 2);
        sendBin(cmnd);
        rcveBin(resp, resp.length);
        return Convert.bytesToFloat(resp, 4);
    }


   /**
    ***************************************************************************
    **
    **  Set a float parameter into the controller
    **
    ***************************************************************************
    */
    public void setFloatParm(int pnum, float value)
    {
        byte[] cmnd = new byte[8];

        cmnd[0] = 0;
        cmnd[1] = (byte)0x8b;
        Convert.shortToBytes((short)pnum, cmnd, 2);
        Convert.floatToBytes(value, cmnd, 4);
        sendBin(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Get user parameters (doubles) from the controller. User parameters
    *   are stored in AcroBasic global variables, each of which is an
    **  IEEE double occupying two words of controller memory.
    **  @param pnum The parameter number, counting from zero, of the first parameter to fetch.
    **  @param value The array to fill with parameter values.
    **  @param count The number of parameters to fetch.
    **
    ***************************************************************************
    */
    public int getUserParms(int pnum, double[] value, int count)
    {
        final int[] iValue = new int[2 * count];

        peek(ctrlType.getSystemPointerAddress(), ACT_CONV_INT, iValue, 1);
        final int globalBase = iValue[0];
        if (globalBase == 0) return 0;
        peek(globalBase, ACT_CONV_INT, iValue, 1); // One word (int) at globalBase.
        final int numGlobals = iValue[0] / 2;
        if (numGlobals < (pnum + 1)) return 0;
        if (numGlobals < (pnum + count))
            count = numGlobals - pnum;

        // If we asked the controller to do a floating point fetch it would give
        // us a float instead of a double, despite storing the values internally
        // as doubles. So we fetch two ints per parameter and assemble them
        // into doubles ourselves.
        final int wb = ctrlType.getWordBump();
        peek(globalBase + wb*(1 + 2 * pnum), ACT_CONV_INT, iValue, 2 * count);
        for (int j = 0; j < count; j++) {
            long lValue = ((long)iValue[2 * j] & 0xffffffffL)
                            | ((long)iValue[2 * j + 1] << 32);
            value[j] = Double.longBitsToDouble(lValue);
        }

        return count;
    }


   /**
    ***************************************************************************
    **
    **  Set user parameters (doubles) in the controller. User parameters
    *   are stored in AcroBasic global variables, each of which is an
    **  IEEE double occupying two words of controller memory.
    **  @param pnum The parameter number, counting from zero, of the first parameter to store.
    **  @param value The array of parameter values.
    **  @param count The number of parameters to set.
    **
    ***************************************************************************
    */
    public int setUserParms(int pnum, double[] value, int count)
    {
        int[] iValue = new int[2 * count];

        peek(ctrlType.getSystemPointerAddress(), ACT_CONV_INT, iValue, 1);
        final int globalBase = iValue[0];
        if (globalBase == 0) return 0;
        peek(globalBase, ACT_CONV_INT, iValue, 1);
        final int numGlobals = iValue[0] / 2;
        if (numGlobals < (pnum + 1)) return 0;
        if (numGlobals < (pnum + count))
            count = numGlobals - pnum;

        // As for getting, we have to manage the two halves of each double ourselves.
        for (int j = 0; j < count; j++) {
            long lValue = Double.doubleToLongBits(value[j]);
            iValue[2 * j] = (int)lValue;
            iValue[2 * j + 1] = (int)(lValue >> 32);
        }
        final int wb = ctrlType.getWordBump();
        poke(globalBase + wb*(1 + 2 * pnum), ACT_CONV_INT, iValue, 2 * count);

        return count;
    }


   /**
    ***************************************************************************
    **
    **  Gets the address of a system parameter from the controller.
    **  @param pnum The parameter number.
    **
    ***************************************************************************
    */
    public int getParmAddr(int pnum)
    {
        byte[] cmnd = new byte[4], resp = new byte[8];

        cmnd[0] = 0;
        cmnd[1] = (byte)0x93;
        Convert.shortToBytes((short)pnum, cmnd, 2);
        sendBin(cmnd);
        rcveBin(resp, resp.length);
        return Convert.bytesToInt(resp, 4);
    }


   /**
    ***************************************************************************
    **
    **  Gets the base address of a program's storage area for a given type of variable.
    **  The different types of variables are allocated in separate areas. Each
    **  area starts with a count of the number of variables in it.
    **  @param pnum The program number.
    **  @param vcode The variable type-code.
    **
    ***************************************************************************
    */
    public int getVarAddr(int pnum, int vcode)
    {
        byte[] cmnd = new byte[4], resp = new byte[8];

        cmnd[0] = 0;
        cmnd[1] = (byte)0x92; // "Get address" command.
        cmnd[2] = (byte)pnum;
        cmnd[3] = (byte)vcode;
        sendBin(cmnd);
        rcveBin(resp, resp.length);
        return Convert.bytesToInt(resp, 4);
    }


   /**
    ***************************************************************************
    **
    **  For a given program, gets either the number of variables of the given type
    **  or, for array variables only, the number of values in the array.
    **  @param pnum The program number.
    **  @param type The variable type code.
    **  @param array Applicable to array types only. If less than zero then the number of
    **  arrays of the given type is returned. A value i >= 0 will cause the number of elements
    **  of the i'th array of the given type to be returned (0 for SA0/LA0/etc)..
    **  @return The count requested, or zero if the variable doesn't exist.
    **
    ***************************************************************************
    */
    public int getVarCount(int pnum, int type, int array)
    {
        /*
        **  Get variable base address; return 0 if zero
        */
        int vAddr = getVarAddr(pnum, type);
        if (vAddr == 0) return 0;

        /*
        **  Get word at base, the number of entries.
        **  If the given variable type is scalar or if the given array index no. is negative
        **  then this is the count we want.
        */
        int[] nVar = new int[1];
        peek(vAddr, ACT_CONV_INT, nVar, 1);
        if ((type & 1) == 0 || array < 0) return nVar[0];

        /*
        **  For array variables the count word is followed not by the array values
        **  but by the addresses of the arrays. Each array has an element-count word
        **  at the beginning which is what we want.
        */
        if (array >= nVar[0]) return 0;
        int[] addr = new int[1];
        final int wb = ctrlType.getWordBump();
        peek(vAddr + wb*(1 + array), ACT_CONV_INT, addr, 1);
        if (addr[0] == 0) return 0;
        peek(addr[0], ACT_CONV_INT, nVar, 1);
        return nVar[0];
    }


   /**
    ***************************************************************************
    **
    **  Dumps numeric scalar program variable values from the controller as integers,
    **  no matter the actual type of the variable(s).
    **  @param pnum The program number.
    **  @param type The variable type code. No arrays or strings.
    **  @param index The index of the first variable (0 for LV0/SV0/DV0/etc.).
    **  @param count The number of variables to dump.
    **
    ***************************************************************************
    */
    public int peekVars(int pnum, int type, int index, int count, int[] value)
    {
        if ((type & 1) != 0 || type == ACV_$V) return 0;
        int vAddr = getVarAddr(pnum, type);
        if (vAddr == 0) return 0;

        int[] nVar = new int[1];
        peek(vAddr, ACT_CONV_INT, nVar, 1);
        if (index < 0) index = 0;
        if (index >= nVar[0]) return 0;
        if (count <= 0) return 0;
        if (count > nVar[0] - index) count = nVar[0] - index;
        final int nw = (type == ACV_DV) ? 2 : 1;
        final int wb = ctrlType.getWordBump();
        peek(vAddr + wb*(1 + nw * index), ACT_CONV_INT, value, nw * count);

        return count;
    }

            
   /**
    ***************************************************************************
    **
    **  Gets numeric program array variable values from the controller as
    **  integers.
    **  @param pnum The program number.
    **  @param type The variable type code (no scalars or string arrays).
    **  @param array The numerical index of the target array (0 for LA0/SA0/DV0/etc.).
    **  @param count The number of array elements to fetch.
    **  @param value The Java array in which to place the fetched data.
    **
    ***************************************************************************
    */
    public int peekVars(int pnum, int type, int array, int index, int count,
                        int[] value)
    {
        if ((type & 1) == 0 || type == ACV_$A) return 0; // No scalars or string arrays.
        int vAddr = getVarAddr(pnum, type);
        if (vAddr == 0) return 0;

        int[] nArr = new int[1];
        peek(vAddr, ACT_CONV_INT, nArr, 1);
        if (array < 0) array = 0;
        if (array >= nArr[0]) return 0;
        int[] addr = new int[1];
        final int wb = ctrlType.getWordBump();
        peek(vAddr + wb*(1 + array), ACT_CONV_INT, addr, 1);
        if (addr[0] == 0) return 0;

        int[] nVar = new int[1];
        peek(addr[0], ACT_CONV_INT, nVar, 1);
        if (index < 0) index = 0;
        if (index >= nVar[0]) return 0;
        if (count <= 0) return 0;
        if (count > nVar[0] - index) count = nVar[0] - index;
        final int nw = (type == ACV_DA) ? 2 : 1;
        peek(addr[0] + wb*(1 + nw * index), ACT_CONV_INT, value, nw * count);

        return count;
    }

            
   /**
    ***************************************************************************
    **
    **  Gets integer (long) program variable values from the controller.
    **  @param pnum The program number.
    **  @param index The index of the first variable (0 LV0, 1 for LV1, etc.).
    **  @param count The number of variable values to retrieve.
    **  @param value The Java array in which to store the values.
    * 
    **
    ***************************************************************************
    */
    public int getVars(int pnum, int index, int count, int[] value)
    {
        int vAddr = getVarAddr(pnum, ACV_LV);
        if (vAddr == 0) return 0;

        int[] nVar = new int[1];
        peek(vAddr, ACT_CONV_INT, nVar, 1);
        if (index < 0) index = 0;
        if (index >= nVar[0]) return 0;
        if (count <= 0) return 0;
        if (count > nVar[0] - index) count = nVar[0] - index;
        final int wb = ctrlType.getWordBump();
        peek(vAddr + wb*(1 + index), ACT_CONV_INT, value, count);

        return count;
    }

            
   /**
    ***************************************************************************
    **
    **  Gets integer (long) program array variable values from the controller.
    **  @param pnum The program number.
    **  @param array The index of the array (0 for LA0, 1 for LA1, etc.).
    **  @param index The index of the first element to fetch (0 for first element).
    **  @param count The number of element values to fetch.
    **  @param value The Java array in which the values will be stored.
    **
    ***************************************************************************
    */
    public int getVars(int pnum, int array, int index, int count, int[] value)
    {
        int vAddr = getVarAddr(pnum, ACV_LA);
        if (vAddr == 0) return 0;

        int[] nArr = new int[1];
        peek(vAddr, ACT_CONV_INT, nArr, 1);
        if (array < 0) array = 0;
        if (array >= nArr[0]) return 0;
        int[] addr = new int[1];
        final int wb = ctrlType.getWordBump();
        peek(vAddr + wb*(1 + array), ACT_CONV_INT, addr, 1); // Get pointer to array.
        if (addr[0] == 0) return 0;

        int[] nVar = new int[1];
        peek(addr[0], ACT_CONV_INT, nVar, 1);
        if (index < 0) index = 0;
        if (index >= nVar[0]) return 0;
        if (count <= 0) return 0;
        if (count > nVar[0] - index) count = nVar[0] - index;
        peek(addr[0] + wb*(1 + index), ACT_CONV_INT, value, count); // Get elements.

        return count;
    }

            
   /**
    ***************************************************************************
    **
    **  Gets float (single) program variable values from the controller.
    **  @param pnum The program number.
    **  @param index The index of the first variable (0 to get SV0, etc).
    **  @param count The number of variable values to fetch.
    **  @param value The Java array in which to store the values.
    ***************************************************************************
    */
    public int getVars(int pnum, int index, int count, float[] value)
    {
        int vAddr = getVarAddr(pnum, ACV_SV);
        if (vAddr == 0) return 0;

        int[] nVar = new int[1];
        peek(vAddr, ACT_CONV_INT, nVar, 1);
        if (index < 0) index = 0;
        if (index >= nVar[0]) return 0;
        if (count <= 0) return 0;
        if (count > nVar[0] - index) count = nVar[0] - index;
        int[] iValue = new int[count];
        final int wb = ctrlType.getWordBump();
        peek(vAddr + wb*(1 + index), ACT_CONV_INT, iValue, count);
        for (int j = 0; j < count; j++)
            value[j] = Float.intBitsToFloat(iValue[j]);

        return count;
    }

            
   /**
    ***************************************************************************
    **
    **  Gets float (single) program array variable values from the controller.
    **  @param pnum The program number.
    **  @param array The index of the array (0 for SA0, 1 for SA1, etc.).
    **  @param index The index of the first element to fetch (0 for first element).
    **  @param count The number of element values to fetch.
    **  @param value The Java array in which the values will be stored.
    **
    ***************************************************************************
    */
    public int getVars(int pnum, int array, int index, int count, float[] value)
    {
        int vAddr = getVarAddr(pnum, ACV_SA);
        if (vAddr == 0) return 0;

        int[] nArr = new int[1];
        peek(vAddr, ACT_CONV_INT, nArr, 1);
        if (array < 0) array = 0;
        if (array >= nArr[0]) return 0;
        int[] addr = new int[1];
        final int wb = ctrlType.getWordBump();
        peek(vAddr + wb*(1 + array), ACT_CONV_INT, addr, 1);
        if (addr[0] == 0) return 0;

        int[] nVar = new int[1];
        peek(addr[0], ACT_CONV_INT, nVar, 1);
        if (index < 0) index = 0;
        if (index >= nVar[0]) return 0;
        if (count <= 0) return 0;
        if (count > nVar[0] - index) count = nVar[0] - index;
        int[] iValue = new int[count];
        peek(addr[0] + wb*(1 + index), ACT_CONV_INT, iValue, count);
        for (int j = 0; j < count; j++)
            value[j] = Float.intBitsToFloat(iValue[j]);

        return count;
    }

            
   /**
    ***************************************************************************
    **
    **  Gets double program variable values from the controller.
    **  @param pnum The program number.
    **  @param index The index of the first variable (0 DV0, 1 for DV1, etc.).
    **  @param count The number of variable values to retrieve.
    **  @param value The Java array in which to store the values.
    **
    ***************************************************************************
    */
    public int getVars(int pnum, int index, int count, double[] value)
    {
        int vAddr = getVarAddr(pnum, ACV_DV);
        if (vAddr == 0) return 0;

        int[] nVar = new int[1];
        peek(vAddr, ACT_CONV_INT, nVar, 1);
        if (index < 0) index = 0;
        if (index >= nVar[0]) return 0;
        if (count <= 0) return 0;
        if (count > nVar[0] - index) count = nVar[0] - index;
        int[] iValue = new int[2 * count];
        final int wb = ctrlType.getWordBump();
        peek(vAddr + wb*(1 + 2 * index), ACT_CONV_INT, iValue, 2 * count);
        for (int j = 0; j < count; j++) {
            long lValue = ((long)iValue[2 * j] & 0xffffffffL)
                            | ((long)iValue[2 * j + 1] << 32);
            value[j] = Double.longBitsToDouble(lValue);
        }

        return count;
    }

            
   /**
    ***************************************************************************
    **
    **  Gets double program array variable values from the controller.
    **  @param pnum The program number.
    **  @param array The index of the array (0 for DA0, 1 for DA1, etc.).
    **  @param index The index of the first element to fetch (0 for first element).
    **  @param count The number of element values to fetch.
    **  @param value The Java array in which the values will be stored.
    **
    ***************************************************************************
    */
    public int getVars(int pnum, int array, int index, int count,
                       double[] value)
    {
        int vAddr = getVarAddr(pnum, ACV_DA);
        if (vAddr == 0) return 0;

        int[] nArr = new int[1];
        peek(vAddr, ACT_CONV_INT, nArr, 1);
        if (array < 0) array = 0;
        if (array >= nArr[0]) return 0;
        int[] addr = new int[1];
        final int wb = ctrlType.getWordBump();
        peek(vAddr + wb*(1 + array), ACT_CONV_INT, addr, 1); // Get address of array.
        if (addr[0] == 0) return 0;

        int[] nVar = new int[1];
        peek(addr[0], ACT_CONV_INT, nVar, 1);
        if (index < 0) index = 0;
        if (index >= nVar[0]) return 0;
        if (count <= 0) return 0;
        if (count > nVar[0] - index) count = nVar[0] - index;
        int[] iValue = new int[2 * count];
        peek(addr[0] + wb*(1 + 2 * index), ACT_CONV_INT, iValue, 2 * count);
        for (int j = 0; j < count; j++) {
            long lValue = ((long)iValue[2 * j] & 0xffffffffL)
                            | ((long)iValue[2 * j + 1] << 32);
        }

        return count;
    }

            
   /**
    ***************************************************************************
    **
    **  Gets string program variable values from the controller.
    **  @param pnum The program number.
    **  @param index The index of the first variable (0 $V0, 1 for $V1, etc.).
    **  @param count The number of variable values to retrieve.
    **  @param value The Java array in which to store the values.
    **
    ***************************************************************************
    */
    public int getVars(int pnum, int index, int count, String[] value)
    {
        int vAddr = getVarAddr(pnum, ACV_$V);
        if (vAddr == 0) return 0;

        // The first two words in the string area contain the number of string variables
        // and the maximum size of each in bytes.
        int[] dims = new int[2];
        peek(vAddr, ACT_CONV_INT, dims, 2);
        if (index < 0) index = 0;
        if (index >= dims[0]) return 0;
        if (count <= 0) return 0;
        if (count > dims[0] - index) count = dims[0] - index;
        // Each string is allocated a whole number of words so round up the max size.
        // The first word of the string contains the actual size in bytes so add one for that.
        int eSize = 1 + (dims[1] + 3) / 4;
        int[] iValue = new int[eSize * count];
        final int wb = ctrlType.getWordBump();
        peek(vAddr + wb*(2 + eSize * index), ACT_CONV_INT, iValue,
             eSize * count);
        byte[] strng = new byte[4 * (eSize - 1)];
        for (int j = 0; j < count; j++) {
            int sSize = iValue[eSize * j];
            for (int k = 0; k < sSize; k += 4)
                Convert.intToBytes(iValue[eSize * j + 1 + k / 4], strng, k); 
            value[j] = new String(strng, 0, sSize);
        }

        return count;
    }

            
   /**
    ***************************************************************************
    **
    **  Gets string program array variable values from the controller.
    **  @param pnum The program number.
    **  @param array The index of the array (0 for $A0, 1 for $A1, etc.).
    **  @param index The index of the first element to fetch (0 for first element).
    **  @param count The number of element values to fetch.
    **  @param value The Java array in which the values will be stored.
    **
    ***************************************************************************
    */
    public int getVars(int pnum, int array, int index, int count,
                       String[] value)
    {
        int vAddr = getVarAddr(pnum, ACV_$A);
        if (vAddr == 0) return 0;

        int[] nArr = new int[1];
        peek(vAddr, ACT_CONV_INT, nArr, 1);
        if (array < 0) array = 0;
        if (array >= nArr[0]) return 0;
        // Fetch the address of the array.
        final int wb = ctrlType.getWordBump();
        int[] addr = new int[1];
        peek(vAddr + wb*(1 + array), ACT_CONV_INT, addr, 1);
        if (addr[0] == 0) return 0;

        int[] dims = new int[2];
        peek(addr[0], ACT_CONV_INT, dims, 2); // No. of elements, max element size in bytes.
        if (index < 0) index = 0;
        if (index >= dims[0]) return 0;
        if (count <= 0) return 0;
        if (count > dims[0] - index) count = dims[0] - index;
        int eSize = 1 + (dims[1] + 3) / 4;
        int[] iValue = new int[eSize * count];
        peek(addr[0] + wb*(2 + eSize * index), ACT_CONV_INT, iValue,
             eSize * count);
        byte[] strng = new byte[4 * (eSize - 1)];
        for (int j = 0; j < count; j++) {
            int sSize = iValue[eSize * j];
            for (int k = 0; k < sSize; k += 4)
                Convert.intToBytes(iValue[eSize * j + 1 + k / 4], strng, k); 
            value[j] = new String(strng, 0, sSize);
        }

        return count;
    }

            
   /**
    ***************************************************************************
    **
    **  Get a bit from the controller
    **
    ***************************************************************************
    */
    public int getBit(int bnum)
    {
        return (getIntParm((bnum >> 5) + 4096) >> (bnum & 0x1f)) & 1;
    }


   /**
    ***************************************************************************
    **
    **  Set a bit in the controller
    **
    ***************************************************************************
    */
    public void setBit(int bnum)
    {
        byte[] cmnd = new byte[4];

        cmnd[0] = 0x1c;
        Convert.shortToBytes((short)bnum, cmnd, 1);
        cmnd[3] = 0;
        sendBin(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Clear a bit in the controller
    **
    ***************************************************************************
    */
    public void clearBit(int bnum)
    {
        byte[] cmnd = new byte[4];

        cmnd[0] = 0x1d;
        Convert.shortToBytes((short)bnum, cmnd, 1);
        cmnd[3] = 0;
        sendBin(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Get values from controller memory
    **
    ***************************************************************************
    */
    public void peek(int addr, int type, int[] value, int count)
    {
        int size = (count > 255) ? 255 : count, index;
        byte[] cmnd = new byte[8], resp = new byte[4 * size + 8];

        final int wb = ctrlType.getWordBump();
        cmnd[0] = 0;
        cmnd[1] = (byte)0x90;
        cmnd[2] = (byte)type;
        for (index = 0; index < count;) {
            if (size > count - index) size = count - index;
            cmnd[3] = (byte)size;
            Convert.intToBytes(addr, cmnd, 4);
            sendBin(cmnd);
            rcveBin(resp, 4 * size + 8);
            for (int j = 0; j < size; j++)
                value[index++] = Convert.bytesToInt(resp, 4 * j + 8);
            addr += wb * size;
        }
    }


   /**
    ***************************************************************************
    **
    **  Set values into controller memory
    **
    ***************************************************************************
    */
    public void poke(int addr, int type, int[] value, int count)
    {
        int size = (count > 255) ? 255 : count, index;
        byte[] cmnd = new byte[4 * size + 8];

        cmnd[0] = 0;
        cmnd[1] = (byte)0x91;
        cmnd[2] = (byte)type;
        for (index = 0; index < count;) {
            if (size > count - index) size = count - index;
            cmnd[3] = (byte)size;
            Convert.intToBytes(addr, cmnd, 4);
            for (int j = 0; j < size; j++)
                Convert.intToBytes(value[index++], cmnd, 4 * j + 8);
            sendBin(cmnd);
            addr += 4 * size;
        }
    }


   /**
    ***************************************************************************
    **
    **  Drain the input from the controller sockets
    **
    ***************************************************************************
    */
    public void drain()
    {
        if (binSock == null) return;

        try {
            byte[] buff = new byte[64];
            binSock.setSoTimeout(TMO_SHORT);
            while (rcveBin(buff, buff.length) > 0);
            binSock.setSoTimeout(TMO_NONE);
        }            
        catch (SocketException e) {
            out.println(e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Start logging commands
    **
    ***************************************************************************
    */
    public void logOpen(String name)
    {
        if (log != null) {
            out.println("Already logging");
            return;
        }

        String file = (name != null ? name : "AcrComm") + "_"
                        + (System.currentTimeMillis() / 1000) + ".log";
        try {
            log = new PrintStream(file);
        }
        catch (IOException e) {
            out.println("Cannot open log file " + file);
            out.println(e);
            return;
        }

    }


   /**
    ***************************************************************************
    **
    **  Stop logging commands
    **
    ***************************************************************************
    */
    public void logClose()
    {
        if (log == null)
            out.println("Not logging");
        else {
            log.close();
            log = null;
        }
    }


   /**
    ***************************************************************************
    **
    **  Encode a parameter number as code, index, and mask bit
    **
    ***************************************************************************
    */
    private final static int[][] encParms = {{0xf8, 3, 0xff, 0},
                                             {0xff, 0, 0x8f, 4},
                                             {0xff, 3, 0x1f, 0},
                                             {0xf8, 0, 0xff, 8},
                                             {0x40, 0, 0x8f, 4},
                                             {0x40, 0, 0x1f, 5},
                                             {0x40, 3, 0x1f, 0}};

    private void encodePnum(int pnum, byte[] cmnd)
    {
        int index;
        pnum &= 0xffff;
        if (pnum < 6144) index = 0;
        else if (pnum < 6912) index = 1;
        else if (pnum < 7168) index = 2;
        else if (pnum < 7680) index = 1;
        else if (pnum < 8192) index = 2;
        else if (pnum < 24576) index = 3;
        else if (pnum < 24832) index = 1;
        else if (pnum < 32768) index = 3;
        else if (pnum < 33280) index = 6;
        else if (pnum < 36864) index = 4;
        else if (pnum < 37376) index = 5;
        else if (pnum < 38144) index = 6;
        else if (pnum < 38392) index = 4;
        else index = 6;

        int codeMask = encParms[index][0];
        int indexShift = encParms[index][1];
        int indexMask = encParms[index][2];
        int maskShift = encParms[index][3];

        cmnd[0] = 0;
        if (codeMask == 0x40)
            cmnd[1] = (byte)((pnum >> 8) | codeMask);
        else
            cmnd[1] = (byte)((pnum >> 8) & codeMask);
        cmnd[2] = (byte)((pnum >> indexShift) & indexMask);
        cmnd[3] = (byte)(1 << ((pnum >> maskShift) & 7));
    }

            
   /**
    ***************************************************************************
    **
    **  Log binary data sent to or received from the controller
    **
    ***************************************************************************
    */
    private void logBin(boolean sent, byte[] buff, int leng)
    {
        if (!logLF) log.println();
        GregorianCalendar cal = new GregorianCalendar();
        cal.setTimeInMillis(System.currentTimeMillis());
        for (int j = 0; j < leng; j++) {
            if ((j & 0x1f) == 0) {
                if (j == 0)
                    log.format("%tF %<tT.%<tL: Bin %s:", cal,
                               sent ? "cmnd" : "resp");
                else
                    log.format("\n%34s", "");
            }
            if ((j & 3) == 0) log.print(" ");
            log.format("%02x", buff[j ^ 3]);
        }
        log.println();
        logLF = true;
    }


   /**
    ***************************************************************************
    **
    **  Log Ascii data sent to or received from the controller
    **
    ***************************************************************************
    */
    private void logAscii(boolean sent, boolean first, byte[] buff, int offs,
                          int leng)
    {
        if (first) {
            if (!logLF) log.println();
            GregorianCalendar cal = new GregorianCalendar();
            cal.setTimeInMillis(System.currentTimeMillis());
            log.format("%tF %<tT.%<tL: Asc %s: ", cal, sent ? "cmnd" : "resp");
            logLF = false;
        }
        int end = offs + leng, pos = offs, line = 0;
        for (pos = offs; pos < end; pos++) {
            if (buff[pos] == LF) {
                if (logLF) log.format("%35s", "");
                log.write(buff, offs, pos + 1 - offs);
                offs = pos + 1;
                logLF = true;
            }
        }
        if (pos > offs) {
            if (logLF) log.format("%35s", "");
            log.write(buff, offs, pos - offs);
            logLF = false;
        }
    }


   /**
    ***************************************************************************
    **
    **  Log string data sent to the controller
    **
    ***************************************************************************
    */
    private void logStr(String cmnd)
    {
        if (!logLF) log.println();
        GregorianCalendar cal = new GregorianCalendar();
        cal.setTimeInMillis(System.currentTimeMillis());
        log.format("%tF %<tT.%<tL: Asc cmnd: %s\n", cal, cmnd);
        logLF = true;
    }

}
