package org.lsst.ccs.drivers.parker;

import java.util.Scanner;
import java.util.GregorianCalendar;
import jline.ConsoleReader;
import org.lsst.ccs.utilities.sa.CmndProc;
import org.lsst.ccs.utilities.sa.ConsOut;
import org.lsst.ccs.utilities.sa.Output;

/**
 ***************************************************************************
 **
 **  Program to communicate with a Parker (ACR or Aries) motor controller
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestAcrComm implements CmndProc.Dispatch {

    private final static int
        CMD_SEND        = 0,
        CMD_DRAIN       = 1,
        CMD_PDUMP       = 2,
        CMD_PIDUMP      = 3,
        CMD_PFDUMP      = 4,
        CMD_PADUMP      = 5,
        CMD_PISHOW      = 6,
        CMD_PFSHOW      = 7,
        CMD_IPEEK       = 8,
        CMD_FPEEK       = 9,
        CMD_BSHOW       = 10,
        CMD_BSET        = 11,
        CMD_BCLR        = 12,
        CMD_CTRLX       = 13,
        CMD_CTRLY       = 14,
        CMD_CTRLZ       = 15,
        CMD_VADUMP      = 16,
        CMD_VSHOW       = 17,
        CMD_VDUMP       = 18,
        CMD_VCSHOW      = 19,
        CMD_PMAP        = 20,
        CMD_PGSHOW      = 21,
        CMD_PISET       = 22,
        CMD_PFSET       = 23,
        CMD_PUSHOW      = 24,
        CMD_PUSET       = 25,
        CMD_TIME        = 26,
        CMD_LOPEN       = 27,
        CMD_LCLOSE      = 28,
        CMD_CLOCK       = 29,
        CMD_PGDUMP      = 30,
        NUM_CMDS        = 31;

    private final static String[] helpSend = {
        "Send a file to the controller",
        "send <filename>",
        "filename  The name of the file to send"
    };

    private final static String[] helpDrain = {
        "Drain the input from the controller",
        "drain",
    };

    private final static String[] helpPmap = {
        "Show map of valid parameters",
        "pmap <first> [<count>]",
        "first   The number of the first parameter to map",
        "count   The optional number of parameters to map (default 64)"
    };

    private final static String[] helpPdump = {
        "Dump parameter values",
        "pdump <first> [<count>]",
        "first   The number of the first parameter to dump",
        "count   The optional number of parameters to dump (default 1)"
    };

    private final static String[] helpPidump = {
        "Dump parameter values as integers",
        "pidump <first> [<count>]",
        "first   The number of the first parameter to dump",
        "count   The optional number of parameters to dump (default 1)"
    };

    private final static String[] helpPfdump = {
        "Dump parameter values as floats",
        "pfdump <first> [<count>]",
        "first   The number of the first parameter to dump",
        "count   The optional number of parameters to dump (default 1)"
    };

    private final static String[] helpPadump = {
        "Dump parameter addresses",
        "padump <first> [<count>]",
        "first   The number of the first parameter address to dump",
        "count   The optional number of addresses to dump (default 1)"
    };

    private final static String[] helpPgdump = {
        "Dump (in hexadecimal) a named group of parameters",
        "pgdump <group>",
        "group   The name of the parameter group to dump",
    };

    private final static String[] helpPishow = {
        "Show parameter values as integers",
        "pishow <first> [<count>]",
        "first   The number of the first parameter to show",
        "count   The optional number of parameters to show (default 1)"
    };

    private final static String[] helpPfshow = {
        "Show parameter values as floats",
        "pfshow <first> [<count>]",
        "first   The number of the first parameter to show",
        "count   The optional number of parameters to show (default 1)"
    };

    private final static String[] helpPushow = {
        "Show user parameter values (doubles)",
        "pushow <first> [<count>]",
        "first   The number of the first user parameter to show",
        "count   The optional number of parameters to show (default 1)"
    };

    private final static String[] helpPgshow = {
        "Show a named group of parameters",
        "pgshow <group>",
        "group   The name of the parameter group to display",
    };

    private final static String[] helpPiset = {
        "Set the value of an integer parameter",
        "piset   <parm> <value>",
        "parm    The number of the parameter to set",
        "value   The value to set"
    };

    private final static String[] helpPfset = {
        "Set the value of a float parameter",
        "pfset   <parm> <value>",
        "parm    The number of the parameter to set",
        "value   The value to set"
    };

    private final static String[] helpPuset = {
        "Set the values of user parameters (doubles)",
        "puset   <parm> <value1>... [<value8>] ",
        "parm    The number of the first parameter to set",
        "valuen  The values to set"
    };

    private final static String[] helpVadump = {
        "Dump variable base address",
        "vadump <prog> [<type1>]... [<type8>]",
        "prog   The number of the program containing the variable(s)",
        "typen  The type of variable (dv, da, sv, sa, lv, la, $v, $a) dflt all",
    };

    private final static String[] helpVcshow = {
        "Show variable count",
        "vcshow <prog> [<type>] [<array>]",
        "prog   The number of the program containing the variable(s)",
        "type   The type of variable (dv, da, sv, sa, lv, la, $v, $a) dflt all",
        "array  The array index for array types, or -1 for all (default -1)",
    };

    private final static String[] helpVshow = {
        "Show variable values",
        "vshow <prog> <type> [<array>] [<index>] [<count>]",
        "prog   The number of the program containing the variable(s)",
        "type   The type of variable (dv, da, sv, sa, lv, la, $v or $a)",
        "array  The array index (default 0)",
        "index  The index of the first variable to show (default 0)",
        "count  The number of variables to show or -1 for all (default 1)",
    };

    private final static String[] helpVdump = {
        "Dump variable values",
        "vdump <prog> <type> [<array>] [<index>] [<count>]",
        "prog   The number of the program containing the variable(s)",
        "type   The type of variable (dv, da, sv, sa, lv or la)",
        "array  The array index (default 0)",
        "index  The index of the first variable to show (default 0)",
        "count  The number of variables to show or -1 for all (default 1)",
    };

    private final static String[] helpIpeek = {
        "Dump memory contents as integers",
        "ipeek <first> [<count>]",
        "first   The address of the first word to dump",
        "count   The optional number of words to dump (default 1)"
    };

    private final static String[] helpFpeek = {
        "Dump memory contents as floats",
        "fpeek <first> [<count>]",
        "first   The address of the first word to dump",
        "count   The optional number of words to dump (default 1)"
    };

    private final static String[] helpBshow = {
        "Show flag bit values",
        "bshow <first> [<count>]",
        "first   The number of the first flag bit to show",
        "count   The optional number of flag bits to show (default 1)"
    };

    private final static String[] helpBset = {
        "Set a flag bit",
        "bset <bitnum>",
        "bitnum  The number (0 - 65535) of the flag bit to set"
    };

    private final static String[] helpBclr = {
        "Clear a flag bit",
        "bclr <bitnum>",
        "bitnum  The number (0 - 65535) of the flag bit to clear"
    };

    private final static String[] helpTime = {
        "Time various operations",
        "time <type> [<count>]",
        "type   The type of operation to time (binary, ascii, null, program)",
        "count  The number of operations to do (default 10)",
    };

    private final static String[] helpClock = {
        "Display controller clock periodically",
        "clock <period> [<type>]",
        "period  The period (in seconds) between clock displays; 0 = stop",
        "type    The clock to display: 0 = uptime; 1 = realtime",
    };

    private final static String[] helpLopen = {
        "Open the log file",
        "lopen [<name>]",
        "name  The name of the root portion of the log file"
    };

    private final static String[] helpLclose = {
        "Close the log file",
        "lclose",
    };

    private final static String[] helpX = {
        "Send ctrl-X",
        "x",
    };

    private final static String[] helpY = {
        "Send ctrl-Y",
        "y",
    };

    private final static String[] helpZ = {
        "Send ctrl-Z",
        "z",
    };

    private final static int   TIM_TYP_BINARY = 0,
                               TIM_TYP_ASCII  = 1,
                               TIM_TYP_NULL   = 2,
                               TIM_TYP_PROG   = 3;

    private final static int   PGRP_MSTATUS  = 0,
                               PGRP_MCONFIG  = 1,
                               PGRP_JOG      = 2,
                               PGRP_POSITION = 3,
                               PGRP_SERVO    = 4,
                               PGRP_LIMITS   = 5,
                               PGRP_MISC     = 6,
                               NUM_PGRPS     = 7;

    private int clockPeriod, clockType = 0;
    private final Output out;
    private final CmndProc.Lookup varTypes, groupNames, timeTypes;
    private ConsoleReader reader;
    private CmndProc proc;
    private AcrComm acr;
    private boolean acrMode = false;
    private ParmGroup[] groups = new ParmGroup[NUM_PGRPS];
    private Thread clockThread;


   /**
    ***************************************************************************
    **
    **  Private inner class for storing parameter group information
    **
    ***************************************************************************
    */
    private class ParmGroup {

        private int nItem = 0;
        private String gName;
        private int mxlDesc = 0;
        private String[] desc;
        private int[] pnum;
        private int[] type;

        ParmGroup(String name, int count)
        {
            gName = name;
            desc  = new String[count];
            pnum  = new int[count];
            type  = new int[count];
        }

        private void add(String iDesc, int iPnum, int iType)
        {
            desc[nItem]  = iDesc;
            pnum[nItem]  = iPnum;
            type[nItem]  = iType;
            nItem++;
            int lDesc = iDesc.length();
            if (lDesc > mxlDesc) mxlDesc = lDesc;
        }
    }


   /**
    ***************************************************************************
    **
    **  Private inner class for implementing clock test thread
    **
    ***************************************************************************
    */
    private class ClockTest implements Runnable {

        @Override
        public void run()
        {
            GregorianCalendar timeC = new GregorianCalendar(),
                              sTimeC = new GregorianCalendar();
            while (true) {
                long time = System.currentTimeMillis();
                if (clockType == 0) {
                    acr.sendStr("p" + AcrComm.TEMP_INT_PRM + "=s33", 0);
                    int hour = acr.getIntParm(AcrComm.TEMP_INT_PRM);
                    acr.sendStr("p" + AcrComm.TEMP_INT_PRM + "=s34", 0);
                    int minute = acr.getIntParm(AcrComm.TEMP_INT_PRM);
                    long sTime = time - 60000L * (minute + 60 * hour);
                    timeC.setTimeInMillis(time);
                    sTimeC.setTimeInMillis(sTime);
                    out.format("Cur Time: %te-%<tb-%<tY %<tT, "
                                  + "Up Time: %02d:%02d, "
                                  + "Sta Time: %te-%<tb-%<tY %<tT\n",
                               timeC, hour, minute, sTimeC);
                }
                else {
                    long sTime = acr.getIntParm(AcrComm.SYS_CLOCK_PRM);
                    sTime = time - (0xffffffffl & sTime);
                    timeC.setTimeInMillis(time);
                    sTimeC.setTimeInMillis(sTime);
                    out.format("Curr Time: %te-%<tb-%<tY %<tT.%<tL, "
                                  + "Base Time: %te-%<tb-%<tY %<tT.%<tL\n",
                                timeC, sTimeC);
                }
                try {
                    Thread.sleep(1000 * clockPeriod);
                }
                catch(InterruptedException e) {
                    break;
                }
                if (clockPeriod <= 0) break;
            }
            clockThread = null;
        }
    }


   /**
    ***************************************************************************
    **
    **  Main constructor
    **
    ***************************************************************************
    */
    public TestAcrComm(AcrComm iAcr, String node, CmndProc cProc, boolean log,
                       Output iOut) throws java.io.IOException
    {
        out = (iOut != null) ? iOut : new ConsOut();

        CmndProc.Command cmnd = new CmndProc.Command(NUM_CMDS);
        cmnd.add("send",        CMD_SEND,        helpSend);
        cmnd.add("drain",       CMD_DRAIN,       helpDrain);
        cmnd.add("pmap",        CMD_PMAP,        helpPmap);
        cmnd.add("pdump",       CMD_PDUMP,       helpPdump);
        cmnd.add("pidump",      CMD_PIDUMP,      helpPidump);
        cmnd.add("pfdump",      CMD_PFDUMP,      helpPfdump);
        cmnd.add("padump",      CMD_PADUMP,      helpPadump);
        cmnd.add("pgdump",      CMD_PGDUMP,      helpPgdump);
        cmnd.add("pishow",      CMD_PISHOW,      helpPishow);
        cmnd.add("pfshow",      CMD_PFSHOW,      helpPfshow);
        cmnd.add("pushow",      CMD_PUSHOW,      helpPushow);
        cmnd.add("pgshow",      CMD_PGSHOW,      helpPgshow);
        cmnd.add("piset",       CMD_PISET,       helpPiset);
        cmnd.add("pfset",       CMD_PFSET,       helpPfset);
        cmnd.add("puset",       CMD_PUSET,       helpPuset);
        cmnd.add("vadump",      CMD_VADUMP,      helpVadump);
        cmnd.add("vshow",       CMD_VSHOW,       helpVshow);
        cmnd.add("vcshow",      CMD_VCSHOW,      helpVcshow);
        cmnd.add("vdump",       CMD_VDUMP,       helpVdump);
        cmnd.add("ipeek",       CMD_IPEEK,       helpIpeek);
        cmnd.add("fpeek",       CMD_FPEEK,       helpFpeek);
        cmnd.add("bshow",       CMD_BSHOW,       helpBshow);
        cmnd.add("bset",        CMD_BSET,        helpBset);
        cmnd.add("bclr",        CMD_BCLR,        helpBclr);
        cmnd.add("time",        CMD_TIME,        helpTime);
        cmnd.add("clock",       CMD_CLOCK,       helpClock);
        cmnd.add("lopen",       CMD_LOPEN,       helpLopen);
        cmnd.add("lclose",      CMD_LCLOSE,      helpLclose);
        cmnd.add("x",           CMD_CTRLX,       helpX);
        cmnd.add("y",           CMD_CTRLY,       helpY);
        cmnd.add("z",           CMD_CTRLZ,       helpZ);

        timeTypes = new CmndProc.Lookup(4);
        timeTypes.add("binary",  TIM_TYP_BINARY);
        timeTypes.add("ascii",   TIM_TYP_ASCII);
        timeTypes.add("program", TIM_TYP_PROG);
        timeTypes.add("null",    TIM_TYP_NULL);

        varTypes = new CmndProc.Lookup(8);
        varTypes.add("dv", AcrComm.ACV_DV);
        varTypes.add("da", AcrComm.ACV_DA);
        varTypes.add("sv", AcrComm.ACV_SV);
        varTypes.add("sa", AcrComm.ACV_SA);
        varTypes.add("lv", AcrComm.ACV_LV);
        varTypes.add("la", AcrComm.ACV_LA);
        varTypes.add("$v", AcrComm.ACV_$V);
        varTypes.add("$a", AcrComm.ACV_$A);

        groupNames = new CmndProc.Lookup(NUM_PGRPS);
        groupNames.add("mstatus",  PGRP_MSTATUS);
        groupNames.add("mconfig",  PGRP_MCONFIG);
        groupNames.add("jog",      PGRP_JOG);
        groupNames.add("position", PGRP_POSITION);
        groupNames.add("servo",    PGRP_SERVO);
        groupNames.add("limits",   PGRP_LIMITS);
        groupNames.add("misc",     PGRP_MISC);

        ParmGroup pGroup;
        pGroup = groups[PGRP_MSTATUS] = new ParmGroup("Motor status", 16);
        pGroup.add("S6  - Continuous Current Rating",  28736, 1);
        pGroup.add("S7  - Maximum Current Rating",     28737, 1);
        pGroup.add("S9  - Commanded Current",          28738, 1);
        pGroup.add("S10 - Commanded Torque / Force",   28739, 1);
        pGroup.add("S11 - Actual Torque / Force",      28740, 1);
        pGroup.add("S16 - Position of Encoder",        28693, 0);
        pGroup.add("S20 - Drive Temperature",          28743, 1);
        pGroup.add("S21 - Modelled Motor Temperature", 28744, 1);
        pGroup.add("S22 - Bus Voltage",                28745, 1);
        pGroup.add("S24 - Operating Hours",            28695, 0);
        pGroup.add("S25 - Operating Minutes",          28696, 0);
        pGroup.add("S26 - Operating Milliseconds",     28697, 0);
        pGroup.add("S27 - Hall State",                 28694, 0);
        pGroup.add("S33 - Power-Up Hours",             -1,    0);
        pGroup.add("S34 - Power-Up Minutes",           -1,    0);
        pGroup.add("S35 - Feedback Source",            -1,    0);

        pGroup
          = groups[PGRP_MCONFIG] = new ParmGroup("Motor configuration", 41);
        pGroup.add("C0   - Encoder resolution",                    28674, 0);
        pGroup.add("C1   - Continuous current",                    28704, 1);
        pGroup.add("C2   - Continuous Current Derating",           28705, 1);
        pGroup.add("C3   - Peak Current",                          28706, 1);
        pGroup.add("C4   - Motor Inductance",                      28707, 1);
        pGroup.add("C5   - Motor Inductance Factor",               28708, 1);
        pGroup.add("C6   - Maximum Motor Winding Temperature",     28709, 1);
        pGroup.add("C7   - Motor Winding Resistance",              28710, 1);
        pGroup.add("C8   - Motor Rated Speed",                     28711, 1);
        pGroup.add("C9   - Number of Motor Pole Pairs",            28675, 0);
        pGroup.add("C10  - Motor Damping",                         28712, 1);
        pGroup.add("C11  - Motor Rotor Inertia / Forcer Mass",     28713, 1);
        pGroup.add("C12  - Motor Electrical Pitch",                28715, 1);
        pGroup.add("C14  - Torque / Force Limit",                  28717, 1);
        pGroup.add("C15  - Motor Ke",                              28714, 1);
        pGroup.add("C16  - Current Loop Bandwidth",                -1,    0);
        pGroup.add("C18  - Current Foldback Enable",               28684, 0);
        pGroup.add("C20  - Encoder Polarity",                      -1,    0);
        pGroup.add("C21  - Max. Pre-Quadrature Encoder Frequency", -1,    0);
        pGroup.add("C22  - Hall Sensor Configuration",             -1,    0);
        pGroup.add("C23  - Hall-Only Commutation",                 -1,    0);
        pGroup.add("C25  - Motor Temperature Switch Type",         28682, 0);
        pGroup.add("C26  - Motor Ambient Temperature",             28719, 1);
        pGroup.add("C27  - Motor Winding Thermal Resistance",      28720, 1);
        pGroup.add("C28  - Motor Thermal Time Constant",           28721, 1);
        pGroup.add("C29  - Motor Winding Time Constant",           28722, 1);
        pGroup.add("C38  - Drive PWM Frequency",                   -1,    0);
        pGroup.add("C39  - Thermal Switch Checking",               -1,    0);
        pGroup.add("C41  - Output Brake Delay",                    -1,    0);
        pGroup.add("C44  - Error Log Selection",                   -1,    0);
        pGroup.add("C46  - Fault on Drive Disable",                28685, 0);
        pGroup.add("C47  - Custom Winding Number",                 -1,    0);
        pGroup.add("C67  - Drive Control Mode",                    28680, 0);
        pGroup.add("C81  - Set Alignment Angle",                   -1,    0);
        pGroup.add("C82  - Encoder Offset",                        -1,    1);
        pGroup.add("C103 - Drive Disable Delay",                   -1,    0);
        pGroup.add("C117 - I/O Debounce Time",                     -1,    0);
        pGroup.add("C119 - Hall-Less Commutation Gain",            -1,    0);
        pGroup.add("C120 - Hall-Less Commutation Peak Current",    -1,    1);
        pGroup.add("C121 - Hall-Less Commutation Damping",         -1,    0);
        pGroup.add("C137 - Encoder Loss Time",                     -1,    0);

        pGroup = groups[PGRP_JOG] = new ParmGroup("JOG Parameters", 6);
        pGroup.add("JOG VEL Setting", 12348, 1);
        pGroup.add("JOG ACC Setting", 12349, 1);
        pGroup.add("JOG DEC Setting", 12350, 1);
        pGroup.add("JOG JRK Setting", 12351, 1);
        pGroup.add("Current JOG VEL", 12346, 1);
        pGroup.add("Current JOG ACC", 12347, 1);

        pGroup
          = groups[PGRP_POSITION] = new ParmGroup("Position Parameters", 8);
        pGroup.add("Current Position",   12288, 0);
        pGroup.add("Target Position",    12289, 0);
        pGroup.add("Actual Position",    12290, 0);
        pGroup.add("Following Error",    12291, 0);
        pGroup.add("Hardware Capture",   12292, 0);
        pGroup.add("Software Capture",   12293, 0);
        pGroup.add("Primary Setpoint",   12294, 0);
        pGroup.add("Secondary Setpoint", 12295, 0);

        pGroup = groups[PGRP_SERVO] = new ParmGroup("Servo Parameters", 8);
        pGroup.add("Proportional Gain (PGAIN)",        12304, 1);
        pGroup.add("Integral Gain (IGAIN)",            12305, 1);
        pGroup.add("Integral Limit (ILIMIT)",          12306, 1);
        pGroup.add("Integral Delay (IDELAY)",          12307, 1);
        pGroup.add("Derivative Gain (DGAIN)",          12308, 1);
        pGroup.add("Derivative Width (DWIDTH)",        12309, 1);
        pGroup.add("Feedforward Velocity (FFVEL)",     12310, 1);
        pGroup.add("Feedforward Acceleration (FFACC)", 12311, 1);

        pGroup = groups[PGRP_LIMITS] = new ParmGroup("Limit Parameters", 15);
        pGroup.add("Plus Excess Error (EXC)",  12320, 1);
        pGroup.add("Minus Excess Error (EXC)", 12321, 1);
        pGroup.add("Plus In Position (IPB)",   12322, 1);
        pGroup.add("Minus In Position (IPB)",  12323, 1);
        pGroup.add("Plus A Limit (ALM)",       12324, 1);
        pGroup.add("Minus A Limit (ALM)",      12325, 1);
        pGroup.add("Plus B Limit (BLM)",       12326, 1);
        pGroup.add("Minus B Limit (BLM)",      12327, 1);
        pGroup.add("Plus Torque Limit (TLM)",  12328, 1);
        pGroup.add("Minus Torque Limit (TLM)", 12329, 1);
        pGroup.add("Plus Torque Band",         12330, 1);
        pGroup.add("Minus Torque Band",        12331, 1);
        pGroup.add("Backlash Setting (BKL)",   12332, 1);
        pGroup.add("Plus Jog Limit (JLM)",     12334, 1);
        pGroup.add("Minus Jog Limit (JLM)",    12335, 1);

        pGroup
          = groups[PGRP_MISC] = new ParmGroup("Miscellaneous Parameters", 3);
        pGroup.add("System Clock",      6916, 0);
        pGroup.add("Encoder Position",  6144, 0);
        pGroup.add("PPU",              12375, 1);

        reader = new ConsoleReader();
        acr = (iAcr == null) ? new AcrComm(node, log, out) : iAcr;
        proc = (cProc == null) ? new CmndProc() : cProc;
        proc.add(this, cmnd);
    }

    public TestAcrComm(AcrComm acr, CmndProc cProc, boolean log, Output iOut)
             throws java.io.IOException
    {
        this(acr, "", cProc, log, iOut);
    }
    
    public TestAcrComm(String node, CmndProc cProc, boolean log, Output iOut)
             throws java.io.IOException
    {
        this(null, node, cProc, log, iOut);
    }



   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        TestAcrComm tacr = null;

        if (args.length == 0) {
            System.out.println("No controller node specified");
            System.exit(0);
        }

        try {
            tacr = new TestAcrComm(args[0], null, args.length > 1, null);
            tacr.run();
        }
        catch (org.lsst.ccs.drivers.parker.AcrComm.Exception e) {
            System.out.println(e);
        }
        catch (java.io.IOException e) {
            System.out.println(e);
        }

        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Run the test
    **
    **  Loops reading and processing each new typed command line.
    **
    ***************************************************************************
    */
    public void run() throws java.io.IOException
    {
        while (true) {
            String line = reader.readLine(getPrompt());
            if (line == null) break;
            if (!process(line)) break;
        }
    }


   /**
    ***************************************************************************
    **
    **  Process an input line
    **
    ***************************************************************************
    */
    public boolean process(String line)
    {
        /*
        **  Check for switch character (/) at the start
        */
        boolean other = false;
        String cLine = line;

        if (line.length() > 0 && line.substring(0, 1).equals("/")) {
            other = true;
            cLine = line.substring(1);
            if (cLine.isEmpty()) {
                acrMode ^= true;
                return true;
            }
        }

        /*
        **  Send command to controller if requested to
        */
        if (other ^ acrMode) {
            int leng = acr.sendStr(cLine, 0);
            boolean cont = true;
            if (leng <= 0) {
                out.println(leng == 0 ? "Controller not responding"
                                      : "Controller disconnected");
                cont = false;
            }
            return cont;
        }

        /*
        **  Process local command
        */
        return proc.process(cLine);
    }


   /**
    ***************************************************************************
    **
    **  Get the stored prompt
    **
    ***************************************************************************
    */
    public String getPrompt()
    {
        return acrMode ? acr.getPrompt() : ">> ";
    }


   /**
    ***************************************************************************
    **
    **  Get the controller communication object
    **
    ***************************************************************************
    */
    public AcrComm getComm()
    {
        return acr;
    }


   /**
    ***************************************************************************
    **
    **  Dispatch local command for processing
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int code, Scanner scan)
    {
        int found;
        Object[] args = new Object[16];

        switch (code) {

        case CMD_SEND:
            if ((found = CmndProc.scanArgs(scan, "Sb", args)) >= 0)
                acr.sendFile((String)args[0],
                             (found & 0x02) == 0 ? 0 : (Byte)args[1]);
            break;
        
        case CMD_DRAIN:
            acr.drain();
            break;

        case CMD_PMAP:
            if ((found = CmndProc.scanArgs(scan, "Ii", args)) >= 0)
                mapParms((Integer)args[0],
                         (found & 0x02) == 0 ? 64 : (Integer)args[1]);
            break;
        
        case CMD_PDUMP:
            if ((found = CmndProc.scanArgs(scan, "Ii", args)) >= 0)
                dumpParms((Integer)args[0], 
                          (found & 0x02) == 0 ? 1 : (Integer)args[1], 0);
            break;
        
        case CMD_PIDUMP:
            if ((found = CmndProc.scanArgs(scan, "Ii", args)) >= 0)
                dumpParms((Integer)args[0],
                          (found & 0x02) == 0 ? 1 : (Integer)args[1], 1);
            break;
        
        case CMD_PFDUMP:
            if ((found = CmndProc.scanArgs(scan, "Ii", args)) >= 0)
                dumpParms((Integer)args[0],
                          (found & 0x02) == 0 ? 1 : (Integer)args[1], 2);
            break;
        
        case CMD_PADUMP:
            if ((found = CmndProc.scanArgs(scan, "Ii", args)) >= 0)
                dumpParms((Integer)args[0],
                          (found & 0x02) == 0 ? 1 : (Integer)args[1], 3);
            break;
        
        case CMD_PISHOW:
            if ((found = CmndProc.scanArgs(scan, "Ii", args)) >= 0)
                showParms((Integer)args[0],
                          (found & 0x02) == 0 ? 1 : (Integer)args[1], 1);
            break;
        
        case CMD_PFSHOW:
            if ((found = CmndProc.scanArgs(scan, "Ii", args)) >= 0)
                showParms((Integer)args[0],
                          (found & 0x02) == 0 ? 1 : (Integer)args[1], 2);
            break;
        
        case CMD_PUSHOW:
            if ((found = CmndProc.scanArgs(scan, "Ii", args)) >= 0)
                showUserParms((Integer)args[0],
                              (found & 0x02) == 0 ? 1 : (Integer)args[1]);
            break;
        
        case CMD_PGSHOW:
        case CMD_PGDUMP:
            if ((found = CmndProc.scanArgs(scan, "s", args)) >= 0) {
                int pnum;
                if ((found & 0x01) == 0) pnum = -1;
                else pnum = groupNames.encode((String)args[0], true);
                if (pnum < 0) showPgroups();
                else showPgroup(pnum, code == CMD_PGDUMP);
            }
            break;
        
        case CMD_PISET:
            if ((found = CmndProc.scanArgs(scan, "II", args)) >= 0)
                acr.setIntParm((Integer)args[0], (Integer)args[1]);
            break;
        
        case CMD_PFSET:
            if ((found = CmndProc.scanArgs(scan, "IF", args)) >= 0)
                acr.setFloatParm((Integer)args[0], (Float)args[1]);
            break;
        
        case CMD_PUSET:
            if ((found = CmndProc.scanArgs(scan, "IDddddddd", args)) >= 0) {
                double[] dVal = new double[8];
                int count = 0;
                for (int j = 1; j <= 8; j++)
                    if ((found & (1 << j)) != 0)
                        dVal[count++] = (Double)args[j];
                acr.setUserParms((Integer)args[0], dVal, count);
            }
            break;
        
        case CMD_VADUMP:
            if ((found = CmndProc.scanArgs(scan, "Issssssss", args)) >= 0) {
                int mask = 0;
                for (int j = 1; j < 9; j++) {
                    if ((found & (1 << j)) == 0) continue;
                    int type = varTypes.encode((String)args[j], true);
                    mask |= (type < 0) ? type : (1 << type);
                    if (mask < 0) break;
                }
                if (mask >= 0) {
                    if (mask == 0) mask = 0xff;
                    dumpVaddr((Integer)args[0], mask);
                }
            }
            break;

        case CMD_VSHOW:
            if ((found = CmndProc.scanArgs(scan, "ISiii", args)) >= 0) {
                int type = varTypes.encode((String)args[1], true);
                if (type >= 0)
                    showVars((Integer)args[0], type,
                             (found & 0x04) == 0 ? 0 : (Integer)args[2],
                             (found & 0x08) == 0 ? 0 : (Integer)args[3],
                             (found & 0x10) == 0 ? 1 : (Integer)args[4]);
            }
            break;
    
        case CMD_VCSHOW:
            if ((found = CmndProc.scanArgs(scan, "ISi", args)) >= 0) {
                int type = varTypes.encode((String)args[1], true);
                if (type >= 0)
                    showVcount((Integer)args[0], type,
                               (found & 0x04) == 0 ? -1 : (Integer)args[2]);
            }
            break;
    
        case CMD_VDUMP:
            if ((found = CmndProc.scanArgs(scan, "ISiii", args)) >= 0) {
                int type = varTypes.encode((String)args[1], true);
                if (type >= 0)
                    dumpVars((Integer)args[0], type,
                             (found & 0x04) == 0 ? 0 : (Integer)args[2],
                             (found & 0x08) == 0 ? 0 : (Integer)args[3],
                             (found & 0x10) == 0 ? 1 : (Integer)args[4]);
            }
            break;
    
        case CMD_IPEEK:
            if ((found = CmndProc.scanArgs(scan, "Ii", args)) >= 0)
                dumpAddr((Integer)args[0],
                         (found & 0x02) == 0 ? 1 : (Integer)args[1],
                         AcrComm.ACT_CONV_INT);
            break;

        case CMD_FPEEK:
            if ((found = CmndProc.scanArgs(scan, "Ii", args)) >= 0)
                dumpAddr((Integer)args[0],
                         (found & 0x02) == 0 ? 1 : (Integer)args[1],
                         AcrComm.ACT_CONV_FP32);
            break;

        case CMD_BSHOW:
            if ((found = CmndProc.scanArgs(scan, "Ii", args)) >= 0)
                showBits((Integer)args[0],
                         (found & 0x02) == 0 ? 1 : (Integer)args[1]);
            break;

        case CMD_BSET:
            if ((found = CmndProc.scanArgs(scan, "I", args)) >= 0)
                acr.setBit((Integer)args[0]);
            break;

        case CMD_BCLR:
            if ((found = CmndProc.scanArgs(scan, "I", args)) >= 0)
                acr.clearBit((Integer)args[0]);
            break;

        case CMD_TIME:
            if ((found = CmndProc.scanArgs(scan, "Si", args)) >= 0) {
                int type = timeTypes.encode((String)args[0], true);
                if (type >= 0)
                    doTime(type, (found & 0x02) == 0 ? 10 : (Integer)args[1]);
            }
            break;

        case CMD_CLOCK:
            if ((found = CmndProc.scanArgs(scan, "Ii", args)) >= 0) {
                clockPeriod = (Integer)args[0];
                if ((found & 0x02) != 0) clockType = (Integer)args[1];
                if (clockPeriod > 0) {
                    if (clockThread == null) {
                        clockThread = new Thread(new ClockTest());
                        clockThread.start();
                    }
                }
                else {
                    if (clockThread != null) clockThread.interrupt();
                }
            }
            break;

        case CMD_LOPEN:
            if ((found = CmndProc.scanArgs(scan, "s", args)) >= 0)
                acr.logOpen((found & 0x01) == 0 ? null : (String)args[0]);
            break;

        case CMD_LCLOSE:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0)
                acr.logClose();
            break;

        case CMD_CTRLX:
            byte[] ctrlx = {0x18, 0, 0, 0};
            acr.sendBin(ctrlx);
            break;
        
        case CMD_CTRLY:
            byte[] ctrly = {0x19, 0, 0, 0};
            acr.sendBin(ctrly);
            break;
        
        case CMD_CTRLZ:
            byte[] ctrlz = {0x1a, 0, 0, 0};
            acr.sendBin(ctrlz);
            break;
        
        default:
            out.println("Command not fully implemented");

        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Display map of valid parameters
    **
    ***************************************************************************
    */
    private void mapParms(int pnum, int count)
    {
        char[] line = new char[72];
        for (int j = 0; j < line.length; j += 9) line[j] = ' ';

        int nValid = 0, posn = 1, sPnum = pnum;
        for (int j = 0; j < count; j++, pnum++) {
            if ((j & 63) == 0) {
                if (nValid != 0) out.format("%6s:%s\n", "P" + sPnum,
                                            String.copyValueOf(line));
                nValid = 0;
                posn = 1;
                sPnum = pnum;
            }
            else if ((j & 7) == 0) posn++;
            if (acr.getParmAddr(pnum) == 0)
                line[posn++] = '.';
            else {
                line[posn++] = 'X';
                nValid++;
            }
        }
        if (nValid != 0) out.format("%6s:%s\n", "P" + sPnum,
                                    String.copyValueOf(line, 0, posn));
    }


   /**
    ***************************************************************************
    **
    **  Display parameter values or addresses in hexadecimal format
    **
    ***************************************************************************
    */
    private void dumpParms(int pnum, int count, int type)
    {
        for (int j = 0; j < count; j++, pnum++) {
            if ((j & 7) == 0) {
                if (j != 0) out.println();
                out.format("%6s:", "P" + pnum);
            }
            int value = (type == 0) ? acr.getParmAsInt(pnum) :
                        (type == 1) ? acr.getIntParm(pnum) :
                        (type == 2) ? acr.getFloatParmAsInt(pnum) :
                                      acr.getParmAddr(pnum);
            out.format(" %08x", value);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Display parameter values
    **
    ***************************************************************************
    */
    private void showParms(int pnum, int count, int type)
    {
        int lLine = 81;

        for (int j = 0; j < count; j++, pnum++) {
            String word = (type == 1) ? "  " + acr.getIntParm(pnum) :
                                        "  " + acr.getFloatParm(pnum);
            if (lLine + word.length() > 80) {
                if (j != 0) out.println();
                out.format("%6s:", "P" + pnum);
                lLine = 7;
            }
            out.print(word);
            lLine += word.length();
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Display user parameter values
    **
    ***************************************************************************
    */
    private void showUserParms(int pnum, int count)
    {
        double[] value = new double[32];
        int nLine = 0, lLine = 81, nGet = value.length;

        for (int j = 0; j < count; j += nGet) {
            if (nGet > count - j) nGet = count - j;
            acr.getUserParms(pnum, value, nGet);
            for (int k = 0; k < nGet; k++, pnum++) {
                String word = "  " + value[k];
                if (lLine + word.length() > 80) {
                    if (nLine++ != 0) out.println();
                    out.format("%6s:", "P" + pnum);
                    lLine = 7;
                }
                out.print(word);
                lLine += word.length();
            }
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Display the names of all parameter group
    **
    ***************************************************************************
    */
    private void showPgroups()
    {
        out.println("Defined parameter groups:");
        for (int j = 0; j < groupNames.count(); j++)
            out.format("  %-10s - %s\n", groupNames.name(j),
                       groups[groupNames.code(j)].gName);
    }


   /**
    ***************************************************************************
    **
    **  Display or dump parameter group values
    **
    ***************************************************************************
    */
    private void showPgroup(int gnum, boolean dump)
    {
        int tempInt = acr.getIntParm(AcrComm.TEMP_INT_PRM);
        float tempFloat = acr.getFloatParm(AcrComm.TEMP_FLT_PRM);
        ParmGroup group = groups[gnum];

        byte[] blank = new byte[group.mxlDesc];
        for (int j = 0; j < blank.length; j++) blank[j] = 0x20;
        String sBlank = new String(blank);

        out.println(group.gName + ":");
        for (int j = 0; j < group.nItem; j++) {
            String sPnum, sValue, desc = group.desc[j];
            int pnum = group.pnum[j];
            if (pnum >= 0) {
                if (group.type[j] == 0) {
                    if (!dump)
                        sValue = " " + acr.getIntParm(pnum);
                    else
                        sValue = String.format(" %08x", acr.getIntParm(pnum));
                }
                else {
                    if (!dump)
                        sValue = " " + acr.getFloatParm(pnum);
                    else
                        sValue = String.format(" %08x",
                                               acr.getFloatParmAsInt(pnum));
                }
                sPnum = String.format("[P%-5s]", pnum);
            }
            else {
                String src = desc.substring(0, 4);
                if (group.type[j] == 0) {
                    acr.sendStr("p" + AcrComm.TEMP_INT_PRM + " = " + src, 0);
                    int value = acr.getIntParm(AcrComm.TEMP_INT_PRM);
                    if (!dump)
                        sValue = " " + value;
                    else
                        sValue = String.format(" %08x", value);
                }
                else {
                    acr.sendStr("p" + AcrComm.TEMP_FLT_PRM + " = " + src, 0);
                    if (!dump)
                        sValue = " " + acr.getFloatParm(AcrComm.TEMP_FLT_PRM);
                    else {
                        int value = acr.getFloatParmAsInt(AcrComm.TEMP_FLT_PRM);
                        sValue = String.format(" %08x", value);
                    }
                }
                sPnum = "        ";
            }
            out.format("%s  %s:%s  %s\n", sPnum, desc,
                       sBlank.substring(0, group.mxlDesc - desc.length()),
                       sValue);
        }

        acr.setIntParm(AcrComm.TEMP_INT_PRM, tempInt);
        acr.setFloatParm(AcrComm.TEMP_FLT_PRM, tempFloat);
    }


   /**
    ***************************************************************************
    **
    **  Display variable addresses in hexadecimal format
    **
    ***************************************************************************
    */
    private void dumpVaddr(int pnum, int tMask)
    {
        int nDump = 0;
        for (int type = 0; tMask != 0; type++, tMask >>>= 1) {
            if ((tMask & 1) != 0) {
                if (nDump++ == 5) out.println();
                int value = acr.getVarAddr(pnum, type);
                out.format("%s: %08x   ", varTypes.decode(type), value);
            }
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Display variable counts
    **
    ***************************************************************************
    */
    private void showVcount(int pnum, int type, int array)
    {
        int count = acr.getVarCount(pnum, type, array);
        if (count == 0) return;

        String cType = varTypes.decode(type);
        if ((type & 1) == 0)
            out.println(cType + ":  " + count);

        else if (array >= 0)
            out.println(cType + array + ":  " + count);

        else {
            for (int j = 0; j < count; j++) {
                int aCount = acr.getVarCount(pnum, type, j);
                if (aCount > 0)
                    out.println(cType + j + ":  " + aCount);
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Display variable values
    **
    ***************************************************************************
    */
    private void showVars(int pnum, int type, int array, int index, int count)
    {
        int lLine = 81;
        String label, word, prefix;

        if (count <= 0) {
            count = acr.getVarCount(pnum, type, array);
            if (count == 0) return;
        }
        if (index < 0) index = 0;
        if (array < 0) array = 0;

        switch (type) {

        case AcrComm.ACV_LV:
        case AcrComm.ACV_LA:
            int[] iVal = new int[count];
            if (type == AcrComm.ACV_LV) {
                label = "LV(";
                count = acr.getVars(pnum, index, count, iVal);
            }
            else {
                label = "LA" + array + "(";
                count = acr.getVars(pnum, array, index, count, iVal);
            }
            for (int j = 0; j < count; j++, index++) {
                word = "  " + iVal[j];
                if (lLine + word.length() > 80) {
                    if (j != 0) out.println();
                    prefix = label + index + "):";
                    out.print(prefix);
                    lLine = prefix.length();
                }
                out.print(word);
                lLine += word.length();
            }
            break;

        case AcrComm.ACV_SV:
        case AcrComm.ACV_SA:
            float[] fVal = new float[count];
            if (type == AcrComm.ACV_SV) {
                label = "SV(";
                count = acr.getVars(pnum, index, count, fVal);
            }
            else {
                label = "SA" + array + "(";
                count = acr.getVars(pnum, array, index, count, fVal);
            }
            for (int j = 0; j < count; j++, index++) {
                word = "  " + fVal[j];
                if (lLine + word.length() > 80) {
                    if (j != 0) out.println();
                    prefix = label + index + "):";
                    out.print(prefix);
                    lLine = prefix.length();
                }
                out.print(word);
                lLine += word.length();
            }
            break;

        case AcrComm.ACV_DV:
        case AcrComm.ACV_DA:
            double[] dVal = new double[count];
            if (type == AcrComm.ACV_DV) {
                label = "DV(";
                count = acr.getVars(pnum, index, count, dVal);
            }
            else {
                label = "DA" + array + "(";
                count = acr.getVars(pnum, array, index, count, dVal);
            }
            for (int j = 0; j < count; j++, index++) {
                word = "  " + dVal[j];
                if (lLine + word.length() > 80) {
                    if (j != 0) out.println();
                    prefix = label + index + "):";
                    out.print(prefix);
                    lLine = prefix.length();
                }
                out.print(word);
                lLine += word.length();
            }
            break;

        case AcrComm.ACV_$V:
        case AcrComm.ACV_$A:
            String[] sVal = new String[count];
            if (type == AcrComm.ACV_$V) {
                label = "$V(";
                count = acr.getVars(pnum, index, count, sVal);
            }
            else {
                label = "$A" + array + "(";
                count = acr.getVars(pnum, array, index, count, sVal);
            }
            for (int j = 0; j < count; j++, index++) {
                word = "  \"" + sVal[j] + "\"";
                if (lLine + word.length() > 80) {
                    if (j != 0) out.println();
                    prefix = label + index + "):";
                    out.print(prefix);
                    lLine = prefix.length();
                }
                out.print(word);
                lLine += word.length();
            }
            break;
        }

        if (count > 0) out.println();
    }


   /**
    ***************************************************************************
    **
    **  Dump variable values in hexadecimal format
    **
    ***************************************************************************
    */
    private void dumpVars(int pnum, int type, int array, int index, int count)
    {
        String label;

        if (count <= 0) {
            count = acr.getVarCount(pnum, type, array);
            if (count == 0) return;
        }
        if (index < 0) index = 0;
        if (array < 0) array = 0;
        
        int wdSize = 1;
        if (type == AcrComm.ACV_DV || type == AcrComm.ACV_DA) wdSize = 2;
        int[] iVal = new int[wdSize * count];
        if ((type & 1) == 0) {
            label = varTypes.decode(type) + "(";
            count = acr.peekVars(pnum, type, index, count, iVal);
        }
        else {
            label = varTypes.decode(type) + array + "(";
            count = acr.peekVars(pnum, type, array, index, count, iVal);
        }

        int lPref = (label + (8 * (index + count - 1) / 8) + ")").length();
        for (int j = 0; j < wdSize * count; j += wdSize, index++) {
            if ((j & 7) == 0) {
                if (j != 0) out.println();
                String prefix = label + index + ")";
                out.format("%s%s:", prefix,
                           "        ".substring(0, lPref - prefix.length()));
            }
            out.format(" %08x", iVal[j]);
            if (wdSize == 2) out.format(" %08x", iVal[j + 1]);
        }
        if (count > 0) out.println();
    }


   /**
    ***************************************************************************
    **
    **  Display memory data in hexadecimal format
    **
    ***************************************************************************
    */
    private void dumpAddr(int addr, int count, int type)
    {
        int nPeek = (count < 256) ? count : 255, rCount = count, l = 0;
        int[] value = new int[nPeek];

        for (int j = 0; j < count;
             j += nPeek, addr += 4 * nPeek, rCount -= nPeek) {
            if (nPeek > rCount) nPeek = rCount;
            acr.peek(addr, type, value, nPeek);
            for (int k = 0; k < nPeek; k++, l++) {
                if ((l & 7) == 0) {
                    if (l != 0) out.println();
                    out.format("%08x:", addr + 4 * k);
                }
                out.format(" %08x", value[k]);
            }
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Display bit values
    **
    ***************************************************************************
    */
    private void showBits(int bnum, int count)
    {
        int pnum = (bnum >> 5) + 4096,
            value = acr.getIntParm(pnum++) >> (bnum & 0x1f);

        for (int j = 0; j < count; j++, bnum++, value >>= 1) {
            if ((bnum & 0x1f) == 0 && j != 0)
                value = acr.getIntParm(pnum++);
            if ((j & 0x1f) == 0) {
                if (j != 0) out.println();
                out.format("%6s:", "B" + bnum);
            }
            out.print(" " + (value & 1));
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Perform timing
    **
    ***************************************************************************
    */
    private void doTime(int type, int count)
    {
        long sTime = System.currentTimeMillis();

        if (type == TIM_TYP_BINARY) {
            for (int j = 0; j < count; j++) {
                int temp = acr.getIntParm(AcrComm.ENC_POSN_PRM);
            }
        }
        else if (type == TIM_TYP_ASCII) {
            for (int j = 0; j < count; j++)
                acr.sendStr("axis0 ppu 1", 0);
        }
        else if (type == TIM_TYP_NULL) {
            for (int j = 0; j < count; j++);
        }
        else if (type == TIM_TYP_PROG) {
            acr.sendStr("prog0", 0);
            acr.sendStr("new prog0", 0);
            acr.sendStr("program", 0);
            acr.sendStr("clear", 0);
            acr.sendStr("dim lv(1)", 0);
            acr.sendStr("for lv0 = 1 to " + count + " step 1", 0);
            acr.sendStr("axis0 ppu 1", 0);
            acr.sendStr("next", 0);
            acr.sendStr("endp", 0);
            acr.sendStr("lrun prog0", 0);
        }

        float delta = (System.currentTimeMillis() - sTime) / 1000F;
        out.format("Performed %s operations in %.3f seconds\n", count, delta);
    }

}
