package org.lsst.ccs.drivers.orielshutter;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.ascii.Ascii;

/**
 * General access routines for the Oriel Shutter device 
 * 
 * @author Homer Neal
 */
public class OrielShutter extends Ascii {

    /**
     * Private data
     */
    private int addrOffset = 0;

    /**
     * Constructor.
     */
    public OrielShutter() {
        setOptions(Option.NO_NET);
    }

    /**
     * Opens a connection.
     *
     * Overrides the main 4-argument Ascii open method, so gets called by all 
     * other open methods.
     * 
     * @param connType   The connection type
     * @param serialName The device name of the serial port
     * @param baud       The baud rate
     * @param dataChar   The serial data characteristics
     * @throws DriverException
     */
    @Override
    public void open(ConnType connType, String serialName, int baud, int dataChar) throws DriverException
    {
        super.open(connType, serialName, baud, 0);
        init();
    }

    /**
     * Opens a connection.
     *
     * @param device The device name of the serial port
     * @param baud The baud rate
     * @param addr The address offset
     * @throws DriverException 
     */
    public void open(String device, int baud, int addr) throws DriverException
    {
        openSerial(device, baud);
        addrOffset = addr;
    }

    /**
     * Opens a connection.
     *
     * @param device The device name of the serial port
     * @param baud The baud rate
     * @param addr The address offset
     * @throws DriverException 
     */
    public void openftdi(String device, int baud, int addr) throws DriverException
    {
        //System.out.println("opening connection to the Oriel Shutter");
        openFtdi(device, baud);
        addrOffset = addr;
    }

    /**
     * Initializes device.
     */
    public void init()
    {
       setCommandTerm(Terminator.CR);
    }    
 
    /** reset to factory defaults **/
    public void reset() throws DriverException {
        writeOrielShutter("RST");
    }

     /** Gets the OrielShutter identity. **/
    public String getIdent() throws DriverException {
        return readOrielShutter("IDN?") ;
    }

    public void openShutter() throws DriverException {
         writeOrielShutter(String.valueOf(0x40+addrOffset));
    }

    public void closeShutter() throws DriverException {
         writeOrielShutter(String.valueOf(0x41+addrOffset));
    }

    /**
     * Writes a command.
     *
     * @param command The command to write, excluding terminator
     * @throws DriverException
     */
    public synchronized void writeOrielShutter(String command) throws DriverException {
       write(command);
    }

    /**
     * Reads a response.
     *
     * @return The command response string
     * @throws DriverException
     */
    public synchronized String readOrielShutter() throws DriverException {
        return(read());
    }

    /**
     * Reads a response after writing a command.
     *
     * @param command The command to write, excluding terminator
     * @return The command response string
     * @throws DriverException
     */
    public synchronized String readOrielShutter(String command) throws DriverException {
        writeOrielShutter(command);
        return readOrielShutter();
    }

}
