package org.lsst.ccs.drivers.optidrive;

import java.io.PrintStream;
import jline.console.ConsoleReader;
import org.lsst.ccs.utilities.sa.CmndProcess;

/**
 ***************************************************************************
 **
 **  Program to test a Optidrive device
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestOptidrive implements CmndProcess.Dispatch {

    /*
    **  Command codes
    */
    private final static int
        CMD_OPEN        = 0,
        CMD_CLOSE       = 1,
        CMD_REGREAD     = 2,
        CMD_REGWRITE    = 3,
        CMD_SETCONTROL  = 4,
        CMD_SHOWCONTROL = 5,
        CMD_SHOWSTATUS  = 6,
        CMD_SETPARM     = 7,
        CMD_SHOWPARM    = 8,
        NUM_CMDS        = 9;

    /*
    **  Command help
    */
    private final static String[] helpOpen = {
        "Open device connection",
        "open [<index>] [<serial>] [<addr>] [<node>]",
        "index   The index of the FTDI device to connect to (default 0)",
        "serial  The serial number of the FTDI device (default none)",
        "addr    The Modbus address of the drive (default 1)",
        "node    The node from which Modbus is being served (default local)",
    };

    private final static String[] helpClose = {
        "Close device connection",
        "close",
    };

    private final static String[] helpRegread = {
        "Reads and displays a set of holding registers",
        "regread <nmbr> [<count>]",
        "nmbr   The number of the first holding register to read",
        "count  The number of holding registers to read (default 1)",
    };

    private final static String[] helpRegwrite = {
        "Writes a holding register",
        "regwrite <nmbr> <value>",
        "nmbr    The number of the holding register to write",
        "value   The value to write",
    };

    private final static String[] helpSetcontrol = {
        "Sets the value of a control register",
        "setcontrol <ctrl> <value>",
        "ctrl   The number of the control register to set",
        "value  The value to set",
    };

    private final static String[] helpShowcontrol = {
        "Shows the values in a set of control registers",
        "showcontrol <ctrl> [<count>]",
        "ctrl   The number of the first control register to show",
        "count  The number of control registers to show (default 1)",
    };

    private final static String[] helpShowstatus = {
        "Shows the values in a set of status registers",
        "showstatus <stat> [<count>]",
        "stat   The number of the first status register to show",
        "count  The number of status registers to show (default 1)",
    };

    private final static String[] helpSetparm = {
        "Sets the value of a parameter",
        "setparm <parm> <value>",
        "parm   The number of the parameter to set",
        "value  The value to set",
    };

    private final static String[] helpShowparm = {
        "Shows the values of a set of parameters",
        "showparm <parm> [<count>]",
        "parm   The number of the first parameter to show",
        "count  The number of parameters to show (default 1)",
    };

    /*
    **  Command definition table
    */
    private final static CmndProcess.Command cmnd;
    static {
        cmnd = new CmndProcess.Command(NUM_CMDS);
        cmnd.add("open",        CMD_OPEN,        helpOpen,        "isws");
        cmnd.add("close",       CMD_CLOSE,       helpClose,       "");
        cmnd.add("regread",     CMD_REGREAD,     helpRegread,     "Ii");
        cmnd.add("regwrite",    CMD_REGWRITE,    helpRegwrite,    "II");
        cmnd.add("setcontrol",  CMD_SETCONTROL,  helpSetcontrol,  "II");
        cmnd.add("showcontrol", CMD_SHOWCONTROL, helpShowcontrol, "Ii");
        cmnd.add("showstatus",  CMD_SHOWSTATUS,  helpShowstatus,  "Ii");
        cmnd.add("setparm",     CMD_SETPARM,     helpSetparm,     "II");
        cmnd.add("showparm",    CMD_SHOWPARM,    helpShowparm,    "Ii");
    }

    /*
    **  Private fields
    */
    private final static PrintStream out = System.out;
    private final CmndProcess proc = new CmndProcess();
    private Optidrive opt = new Optidrive();
    private short busAddr;
    private boolean devOpen;


   /**
    ***************************************************************************
    **
    **  Main program
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        (new TestOptidrive()).run();

        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Runs the test
    **
    **  <p>
    **  Loops reading and processing each new typed command line.
    **
    ***************************************************************************
    */
    public void run()
    {
        proc.add(this, cmnd);

        try {
            ConsoleReader reader = new ConsoleReader();
            while (true) {
                String line = reader.readLine(">> ");
                if (line == null || !proc.process(line)) break;
            }
            if (devOpen) opt.close();
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }


   /**
    ***************************************************************************
    **
    **  Dispatches command for processing
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int code, int found, Object[] args)
    {
        try {
            switch (code) {
            case CMD_OPEN:
                procOpen(found, args); break;
            case CMD_CLOSE:
                procClose(found, args); break;
            case CMD_REGREAD:
                procRegread(found, args); break;
            case CMD_REGWRITE:
                procRegwrite(found, args); break;
            case CMD_SETCONTROL:
                procSetcontrol(found, args); break;
            case CMD_SHOWCONTROL:
                procShowcontrol(found, args); break;
            case CMD_SHOWSTATUS:
                procShowstatus(found, args); break;
            case CMD_SETPARM:
                procSetparm(found, args); break;
            case CMD_SHOWPARM:
                procShowparm(found, args); break;
            default:
                out.println("Command not fully implemented");
            }
        }
        catch (OptidriveException e) {
            out.println(e);
        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Processes the OPEN command
    **
    ***************************************************************************
    */
    private void procOpen(int found, Object[] args) throws OptidriveException
    {
        int index = (found & 0x01) != 0 ? (Integer)args[0] : 0;
        String serial = (found & 0x02) != 0 ? (String)args[1] : null;
        short addr = (found & 0x04) != 0 ? (Short)args[2] : 1;
        String node = (found & 0x08) != 0 ? (String)args[3] : null;
        opt.open(node, index, serial, addr);
        devOpen = true;
        busAddr = (found & 0x04) != 0 ? (Short)args[2] : 1;
    }


   /**
    ***************************************************************************
    **
    **  Processes the CLOSE command
    **
    ***************************************************************************
    */
    private void procClose(int found, Object[] args) throws OptidriveException
    {
        devOpen = false;
        opt.close();
    }


   /**
    ***************************************************************************
    **
    **  Processes the REGREAD command
    **
    ***************************************************************************
    */
    private void procRegread(int found, Object[] args) throws OptidriveException
    {
        int nmbr = (Integer)args[0];
        int count = (found & 0x02) != 0 ? (Integer)args[1] : 1;
        short[] reply = opt.readRegisters(nmbr, count);
        for (int j = 0; j < reply.length; j++) {
            if ((j % 10) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format("%3s:", nmbr + j);
            }
            out.format(" %04x", reply[j]);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Processes the REGWRITE command
    **
    ***************************************************************************
    */
    private void procRegwrite(int found, Object[] args)
        throws OptidriveException
    {
        opt.writeRegister((Integer)args[0], (Integer)args[1]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SETCONTROL command
    **
    ***************************************************************************
    */
    private void procSetcontrol(int found, Object[] args)
        throws OptidriveException
    {
        opt.setControl((Integer)args[0], (Integer)args[1]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWCONTROL command
    **
    ***************************************************************************
    */
    private void procShowcontrol(int found, Object[] args)
        throws OptidriveException
    {
        int ctrl = (Integer)args[0];
        int count = (found & 0x02) != 0 ? (Integer)args[1] : 1;
        short[] reply = opt.getControl(ctrl, count);
        for (int j = 0; j < reply.length; j++) {
            if ((j % 10) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format("%2s:", ctrl + j);
            }
            out.format(" %04x", reply[j]);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWSTATUS command
    **
    ***************************************************************************
    */
    private void procShowstatus(int found, Object[] args)
        throws OptidriveException
    {
        int stat = (Integer)args[0];
        int count = (found & 0x02) != 0 ? (Integer)args[1] : 1;
        short[] reply = opt.getStatus(stat, count);
        for (int j = 0; j < reply.length; j++) {
            if ((j % 10) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format("%2s:", stat + j);
            }
            out.format(" %04x", reply[j]);
        }
        out.println();
    }


   /**
    ***************************************************************************
    **
    **  Processes the SETPARM command
    **
    ***************************************************************************
    */
    private void procSetparm(int found, Object[] args)
        throws OptidriveException
    {
        opt.setParameter((Integer)args[0], (Integer)args[1]);
    }


   /**
    ***************************************************************************
    **
    **  Processes the SHOWPARM command
    **
    ***************************************************************************
    */
    private void procShowparm(int found, Object[] args)
        throws OptidriveException
    {
        int parm = (Integer)args[0];
        int count = (found & 0x02) != 0 ? (Integer)args[1] : 1;
        short[] reply = opt.getParameters(parm, count);
        for (int j = 0; j < reply.length; j++) {
            if ((j % 10) == 0) {
                if (j != 0) {
                    out.println();
                }
                out.format("%2s:", parm + j);
            }
            out.format(" %04x", reply[j]);
        }
        out.println();
    }

}
