package org.lsst.ccs.drivers.optidrive;

import org.lsst.ccs.drivers.modbus.Modbus;
import org.lsst.ccs.drivers.modbus.ModbusException;

/**
 ***************************************************************************
 **
 **  Accesses an Optidrive controller
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class Optidrive {

    /**
     **  Public constants
     */
    public static final short
        CTRL_CONTROL_WORD         = 1,
        CTRL_FREQUENCY_SETPOINT   = 2,
        CTRL_MODBUS_RAMP_TIME     = 4;

    public static final short
        STAT_ERROR_AND_STATUS     = 6,
        STAT_OUTPUT_FREQUENCY     = 7,
        STAT_MOTOR_CURRENT        = 8,
        STAT_DIGITAL_INPUT_STATUS = 11,
        STAT_RATING_ID            = 12,
        STAT_POWER_RATING         = 13,
        STAT_VOLTAGE_RATING       = 14,
        STAT_IO_SOFTWARE_VERSN    = 15,
        STAT_MOTOR_SOFTWARE_VERSN = 16,
        STAT_DRIVE_TYPE           = 17,
        STAT_ANALOG_INPUT_1       = 20,
        STAT_ANALOG_INPUT_2       = 21,
        STAT_SPEED_REFERENCE      = 22,
        STAT_DC_BUS_VOLTAGE       = 23,
        STAT_DRIVE_TEMPERATURE    = 24;

    public static final short
        PARM_MAX_SPEED_LIMIT      = 1,
        PARM_MIN_SPEED_LIMIT      = 2,
        PARM_ACCEL_RAMP_TIME      = 3,
        PARM_DECEL_RAMP_TIME      = 4,
        PARM_STOP_MODE            = 5,
        PARM_ENERGY_SAVE          = 6,
        PARM_MOTOR_RATED_VOLTAGE  = 7,
        PARM_MOTOR_RATED_CURRENT  = 8,
        PARM_MOTOR_RATED_FREQ     = 9,
        PARM_MOTOR_RATED_SPEED    = 10,
        PARM_BOOST_VOLTAGE        = 11,
        PARM_CONTROL_MODE         = 12,
        PARM_TRIP_LOG             = 13,
        PARM_ACCESS_CODE          = 14,
        PARM_DIGITAL_INPUT_FUNC   = 15,
        PARM_ANALOG_INPUT_FORMAT  = 16,
        PARM_EFF_SWITCH_FREQ      = 17,
        PARM_RELAY_OUTPUT_FUNC    = 18,
        PARM_DIGITAL_OUTPUT_LIMIT = 19,
        PARM_PRESET_SPEED_1       = 20,
        PARM_PRESET_SPEED_2       = 21,
        PARM_PRESET_SPEED_3       = 22,
        PARM_PRESET_SPEED_4       = 23,
        PARM_SECOND_DECEL_RAMP    = 24,
        PARM_ANALOG_OUTPUT_FUNC   = 25,
        PARM_SKIP_FREQUENCY       = 26,
        PARM_SKIP_FREQENCY_BAND   = 27,
        PARM_VF_ADJUST_VOLTAGE    = 28,
        PARM_VF_ADJUST_FREQUENCY  = 29,
        PARM_START_MODE           = 30,
        PARM_KEYPAD_RESTART_MODE  = 31,
        PARM_DC_INJECTION_ENABLE  = 32,
        PARM_SPIN_START_ENABLE    = 33,
        PARM_BRAKE_CIRCUIT_ENABLE = 34,
        PARM_ANALOG_INPUT_SCALING = 35,
        PARM_BAUDRATE_TRIP_TIME   = 36,
        PARM_ACCESS_CODE_DEFN     = 37,
        PARM_PARAMETER_LOCK       = 38,
        PARM_ANALOG_INPUT_OFFSET  = 39,
        PARM_DISPLAY_SCALE_FACTOR = 40,
        PARM_USER_PI_P_GAIN       = 41,
        PARM_USER_PI_I_TIME_CONST = 42,
        PARM_USER_PI_MODE         = 43,
        PARM_USER_PI_REFERENCE    = 44,
        PARM_USER_PI_DIGITAL_REFC = 45,
        PARM_USER_PI_FEEDBACK     = 46,
        PARM_SECOND_ANALOG_FORMAT = 47;

    /**
     **  Private constants
     */
    private static final int
        MIN_CTRL_REG_NUMBER   = 1,
        MAX_CTRL_REG_NUMBER   = 4,
        MIN_STAT_REG_NUMBER   = 6,
        MAX_STAT_REG_NUMBER   = 24,
        MIN_PARM_NUMBER       = 1,
        MAX_PARM_NUMBER       = 47,
        PARM_BASE_REGISTER    = 128;

    /**
     **  Private fields
     */
    private Modbus mbus;
    private short busAddr;


   /**
    ***************************************************************************
    **
    **  Constructors
    **
    ***************************************************************************
    */
    public Optidrive()
    {
        mbus = new Modbus();
    }

    public Optidrive(Optidrive od, int busAddr)
    {
        mbus = od.mbus;
        this.busAddr = (short)busAddr;
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection to the controller
    **
    ***************************************************************************
    */
    public void open(int index, String serial, int busAddr)
        throws OptidriveException
    {
        open(null, index, serial, busAddr);
    }


   /**
    ***************************************************************************
    **
    **  Opens a connection to the controller
    **
    ***************************************************************************
    */
    public void open(String node, int index, String serial, int busAddr)
        throws OptidriveException
    {
        try {
            mbus.open(node, index, serial);
        }
        catch (ModbusException e) {
            throw new OptidriveException(e.getMessage());
        }
        this.busAddr = (short)busAddr;
    }

            
   /**
    ***************************************************************************
    **
    **  Closes the connection to the controller
    **
    ***************************************************************************
    */
    public void close() throws OptidriveException
    {
        try {
            mbus.close();
        }
        catch (ModbusException e) {
            throw new OptidriveException(e.getMessage());
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads a register
    **
    ***************************************************************************
    */
    public short readRegister(int nmbr) throws OptidriveException
    {
        try {
            return mbus.readRegisters(busAddr, (short)nmbr, (short)1)[0];
        }
        catch (ModbusException e) {
            throw new OptidriveException(e.getMessage());
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads multiple registers
    **
    ***************************************************************************
    */
    public short[] readRegisters(int nmbr, int count) throws OptidriveException
    {
        try {
            return mbus.readRegisters(busAddr, (short)nmbr, (short)count);
        }
        catch (ModbusException e) {
            throw new OptidriveException(e.getMessage());
        }
    }


   /**
    ***************************************************************************
    **
    **  Writes a register
    **
    ***************************************************************************
    */
    public void writeRegister(int nmbr, int value) throws OptidriveException
    {
        try {
            mbus.writeRegister(busAddr, (short)nmbr, (short)value);
        }
        catch (ModbusException e) {
            throw new OptidriveException(e.getMessage());
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets a control register
    **
    ***************************************************************************
    */
    public short getControl(int ctrl) throws OptidriveException
    {
        checkControlReg(ctrl);
        return readRegister(ctrl);
    }


   /**
    ***************************************************************************
    **
    **  Gets multiple control registers
    **
    ***************************************************************************
    */
    public short[] getControl(int ctrl, int count) throws OptidriveException
    {
        checkControlReg(ctrl, count);
        return readRegisters(ctrl, count);
    }


   /**
    ***************************************************************************
    **
    **  Sets a control register
    **
    ***************************************************************************
    */
    public void setControl(int ctrl, int value) throws OptidriveException
    {
        checkControlReg(ctrl);
        writeRegister(ctrl, value);
    }


   /**
    ***************************************************************************
    **
    **  Gets a status register
    **
    ***************************************************************************
    */
    public short getStatus(int stat) throws OptidriveException
    {
        checkStatusReg(stat);
        return readRegister(stat);
    }


   /**
    ***************************************************************************
    **
    **  Gets multiple status registers
    **
    ***************************************************************************
    */
    public short[] getStatus(int stat, int count) throws OptidriveException
    {
        checkStatusReg(stat, count);
        return readRegisters(stat, count);
    }


   /**
    ***************************************************************************
    **
    **  Gets a parameter
    **
    ***************************************************************************
    */
    public short getParameter(int parm) throws OptidriveException
    {
        checkParameter(parm);
        return readRegister(PARM_BASE_REGISTER + parm);
    }


   /**
    ***************************************************************************
    **
    **  Gets multiple parameters
    **
    ***************************************************************************
    */
    public short[] getParameters(int parm, int count) throws OptidriveException
    {
        checkParameter(parm, count);
        return readRegisters(PARM_BASE_REGISTER + parm, count);
    }


   /**
    ***************************************************************************
    **
    **  Sets a parameter
    **
    ***************************************************************************
    */
    public void setParameter(int parm, int value) throws OptidriveException
    {
        checkParameter(parm);
        writeRegister(PARM_BASE_REGISTER + parm, value);
    }


   /**
    ***************************************************************************
    **
    **  Checks a control register number
    **
    ***************************************************************************
    */
    private static void checkControlReg(int ctrl) throws OptidriveException
    {
        if (ctrl < MIN_CTRL_REG_NUMBER || ctrl > MAX_CTRL_REG_NUMBER) {
            throw new OptidriveException("Invalid control register number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a range of control register numbers
    **
    ***************************************************************************
    */
    private static void checkControlReg(int ctrl, int count)
        throws OptidriveException
    {
        if (ctrl < MIN_CTRL_REG_NUMBER || count < 1
              || ctrl + count - 1 > MAX_CTRL_REG_NUMBER) {
            throw new OptidriveException("Invalid control register number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a status register number
    **
    ***************************************************************************
    */
    private static void checkStatusReg(int stat) throws OptidriveException
    {
        if (stat < MIN_STAT_REG_NUMBER || stat > MAX_STAT_REG_NUMBER) {
            throw new OptidriveException("Invalid status register number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a range of status register numbers
    **
    ***************************************************************************
    */
    private static void checkStatusReg(int stat, int count)
        throws OptidriveException
    {
        if (stat < MIN_STAT_REG_NUMBER || count < 1
              || stat + count - 1 > MAX_STAT_REG_NUMBER) {
            throw new OptidriveException("Invalid status register number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a parameter number
    **
    ***************************************************************************
    */
    private static void checkParameter(int parm) throws OptidriveException
    {
        if (parm < MIN_PARM_NUMBER || parm > MAX_PARM_NUMBER) {
            throw new OptidriveException("Invalid parameter number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a range of parameter numbers
    **
    ***************************************************************************
    */
    private static void checkParameter(int parm, int count)
        throws OptidriveException
    {
        if (parm < MIN_PARM_NUMBER || count < 1
              || parm + count - 1 > MAX_PARM_NUMBER) {
            throw new OptidriveException("Invalid parameter number");
        }
    }

}
